/**
 *
 *  Based on "batoo" http://people.inf.ethz.ch/adelmanr/batoo/
 *
 *  Copyright (C) 2006 Robert Adelmann
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *  MA  02110-1301, USA
 */


#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#define bool char
#define TRUE 1
#define FALSE 0

static int debug_batoo = FALSE;

static void
transform_to_bw (int w, unsigned char *grey_line, unsigned char *dest) {
    int i;
    int average_illumination = 0;
    for (i = 0; i < w; i++) {
        average_illumination = average_illumination + grey_line[i];
    }
    average_illumination = average_illumination / w;
    // perform the binarization:
    int range = w / 40;

    // temp values:
    int moving_sum;
    int moving_average;
    int v1_index = -range + 1;
    int v2_index = range;
    int v1 = grey_line[0];
    int v2 = grey_line[range];
    int current_value;
    int comparison_value;

    // initialize the moving sum:
    moving_sum = grey_line[0] * range;
    for (i = 0; i < range; i++)
        moving_sum = moving_sum + grey_line[i];

    // apply the adaptive thresholding algorithm:
    for (i = 1; i < w - 1; i++) {
        if (v1_index > 0)
            v1 = grey_line[v1_index];
        if (v2_index < w)
            v2 = grey_line[v2_index];
        else
            v2 = grey_line[w - 1];
        moving_sum = moving_sum - v1 + v2;
        moving_average = moving_sum / (range << 1);
        v1_index++;
        v2_index++;

        current_value = (grey_line[i - 1] + grey_line[i]) >> 1;

        // decide if the current pixel should be black or white:
        comparison_value = (3 * moving_average + average_illumination) >> 2;
        if ((current_value < comparison_value - 3))
            dest[i] = 0;
        else
            dest[i] = 255;
    }
    dest[0] = 255;
    dest[w - 1] = 255;
}

static void
extract_field_information (int w, unsigned char *string, unsigned char **fields, int *field_counter_result) {
    unsigned char *temp_fields = (unsigned char *) malloc (w * 2 * sizeof (unsigned char *));

    int field_counter = 0;
    int i;
    int last_value = string[0];
    int last_fields = 1;
    for (i = 1; i < w; i++) {
        if ((string[i] == last_value) && (i < w - 1)) {
            last_fields++;
        } else {

            // create new field entry:
            temp_fields[field_counter * 2] = last_value;
            temp_fields[field_counter * 2 + 1] = last_fields;

            last_value = string[i];
            last_fields = 0;
            field_counter++;
        }
    }

    *fields = (unsigned char *) malloc (field_counter * 2 * sizeof (unsigned char *));
    for (i = 0; i < field_counter * 2; i++) {
        (*fields)[i] = temp_fields[i];
    }

    free (temp_fields);
    *field_counter_result = field_counter;
}



static int BOTH_TABLES = 0;
static int EVEN_TABLE = 1;
static int ODD_TABLE = 2;

static int code_odd[10][4] = { {30, 20, 10, 10},
    {20, 20, 20, 10},
    {20, 10, 20, 20},
    {10, 40, 10, 10},
    {10, 10, 30, 20},
    {10, 20, 30, 10},
    {10, 10, 10, 40},
    {10, 30, 10, 20},
    {10, 20, 10, 30},
    {30, 10, 10, 20}
};

static int code_even[10][4] = { {10, 10, 20, 30},
    {10, 20, 20, 20},
    {20, 20, 10, 20},
    {10, 10, 40, 10},
    {20, 30, 10, 10},
    {10, 30, 20, 10},
    {40, 10, 10, 10},
    {20, 10, 30, 10},
    {30, 10, 20, 10},
    {20, 10, 10, 30}
};


static bool parity_pattern_list[10][6] = { {FALSE, FALSE, FALSE, FALSE, FALSE, FALSE},
    {FALSE, FALSE, TRUE, FALSE, TRUE, TRUE},
    {FALSE, FALSE, TRUE, TRUE, FALSE, TRUE},
    {FALSE, FALSE, TRUE, TRUE, TRUE, FALSE},
    {FALSE, TRUE, FALSE, FALSE, TRUE, TRUE},
    {FALSE, TRUE, TRUE, FALSE, FALSE, TRUE},
    {FALSE, TRUE, TRUE, TRUE, FALSE, FALSE},
    {FALSE, TRUE, FALSE, TRUE, FALSE, TRUE},
    {FALSE, TRUE, FALSE, TRUE, TRUE, FALSE},
    {FALSE, TRUE, TRUE, FALSE, TRUE, FALSE}
};

typedef struct {
    int digit;
    bool even;
} MatchMakerResult;



static MatchMakerResult *
recognize_number (unsigned char *fields, int code_table_to_use) {

    // convert the pixel lenghts of the four black&white fields into
    // normed values that have together a length of 70;
    int pixel_sum = fields[0 * 2 + 1] + fields[1 * 2 + 1] + fields[2 * 2 + 1] + fields[3 * 2 + 1];
    int b[4];
    int i, j;
    for (i = 0; i < 4; i++) {
        b[i] = round ((((float) fields[i * 2 + 1]) / ((float) pixel_sum)) * 70);
    }

    // print some debugging information:
    if (debug_batoo) {
        printf ("Recognize Number (code table to use: %d)\n", code_table_to_use);
        printf ("lengths: %d %d %d %d\n", fields[1], fields[3], fields[5], fields[7]);
        printf ("normed lengths: %d %d %d %d\n", b[0], b[1], b[2], b[3]);
    }

    // try to detect the digit that is encoded by the set of four normed bar lenghts:
    int max_difference_for_acceptance = 60;
    int temp;


    int even_min_difference = 100000;
    int even_min_difference_index = 0;
    int odd_min_difference = 100000;
    int odd_min_difference_index = 0;
    if ((code_table_to_use == BOTH_TABLES) || (code_table_to_use == EVEN_TABLE)) {
        int even_differences[10];

        for (i = 0; i < 10; i++) {
            even_differences[i] = 0;
            for (j = 0; j < 4; j++) {
                // calculate the differences in the even group:
                temp = b[j] - code_even[i][j];
                if (temp < 0)
                    even_differences[i] = even_differences[i] + ((-temp) << 1);
                else
                    even_differences[i] = even_differences[i] + (temp << 1);

            }
            if (even_differences[i] < even_min_difference) {
                even_min_difference = even_differences[i];
                even_min_difference_index = i;
            }
        }
    }

    if ((code_table_to_use == BOTH_TABLES) || (code_table_to_use == ODD_TABLE)) {
        int odd_differences[10];

        for (i = 0; i < 10; i++) {
            odd_differences[i] = 0;
            for (j = 0; j < 4; j++) {
                // calculate the differences in the odd group:
                temp = b[j] - code_odd[i][j];
                if (temp < 0)
                    odd_differences[i] = odd_differences[i] + ((-temp) << 1);
                else
                    odd_differences[i] = odd_differences[i] + (temp << 1);
            }
            if (odd_differences[i] < odd_min_difference) {
                odd_min_difference = odd_differences[i];
                odd_min_difference_index = i;
            }
        }
    }

    MatchMakerResult *result = (MatchMakerResult *) malloc (sizeof (MatchMakerResult));

    if (debug_batoo)
        printf ("even_min_difference %d   odd_min_difference %d\n", even_min_difference, odd_min_difference);

    // select the digit and parity with the lowest difference to the found pattern:
    if (even_min_difference <= odd_min_difference) {
        if (even_min_difference < max_difference_for_acceptance) {
            result->digit = even_min_difference_index;
            result->even = TRUE;
            return result;

        }
    } else {
        if (odd_min_difference < max_difference_for_acceptance) {
            result->digit = odd_min_difference_index;
            result->even = FALSE;
            return result;

        }
    }

    result->digit = -1;
    result->even = FALSE;
    return result;
}



static bool
is_valid (int *numbers) {
    int i;
    for (i = 0; i < 13; i++) {
        if ((numbers[i] < 0) || (numbers[i] > 9))
            return FALSE;
    }
    // calculate the checksum of the barcode:
    int sum1 = numbers[0] + numbers[2] + numbers[4] + numbers[6] + numbers[8] + numbers[10];
    int sum2 = 3 * (numbers[1] + numbers[3] + numbers[5] + numbers[7] + numbers[9] + numbers[11]);
    int checksum_value = sum1 + sum2;
    int checksum_digit = 10 - (checksum_value % 10);
    if (checksum_digit == 10)
        checksum_digit = 0;

    return (numbers[12] == checksum_digit);
}



static MatchMakerResult *
recognize_system_code (bool * parity_pattern) {

    int i, j;
    // search for a fitting parity pattern:
    bool fits = FALSE;
    MatchMakerResult *result = (MatchMakerResult *) malloc (sizeof (MatchMakerResult));
    for (i = 0; i < 10; i++) {
        fits = TRUE;
        for (j = 0; j < 6; j++) {
            if (parity_pattern_list[i][j] != parity_pattern[j]) {
                fits = FALSE;
                break;
            }
        }
        if (fits) {
            result->even = FALSE;
            result->digit = i;
            return result;
        }
    }
    result->even = FALSE;
    result->digit = -1;

    return result;

}



static int *
decode (unsigned char *fields, int start_i, int end_i, int w) {

    // determine the length of the path in pixels
    int length = 0;
    int i, j;
    for (i = 0; i < w; i++)
        length = length + fields[i * 2 + 1];

    // set the parameters accordingly:
    int max_start_sentry_bar_differences = 0;
    int max_unit_length = 0;
    int min_unit_length = 0;

    if (length <= 500) {
        max_start_sentry_bar_differences = 6;
        max_unit_length = 10;
        min_unit_length = 1;
    } else {
        max_start_sentry_bar_differences = 30;
        max_unit_length = 17;     /*50 */
        min_unit_length = 1;
    }

    // consistency checks:
    if (w <= 0)
        return NULL;
    if (start_i > end_i - 3)
        return NULL;
    if (end_i - start_i < 30)
        return NULL;                // (just a rough value)

    // relevant indexes:
    int start_sentinel_i = 0;
    int end_sentinel_i = 0;
    int left_numbers_i = 0;
    int middle_guard_i = 0;
    int right_numbers_i = 0;

    // relevant parameters:
    float unit_length = 0;

    // results:
    int *numbers = (int *) malloc (13 * sizeof (int));    // = new int[13];

    // determine the relevant positions:

    // Try to detect the start sentinel (a small black-white-black serie):
    start_sentinel_i = -1;
    for (i = start_i; i < end_i - 56; i++) {
        if (fields[i * 2] == 0) {
            if ((fields[i * 2 + 1] >= min_unit_length) && (fields[i * 2 + 1] <= max_unit_length)) {
                if ((abs (fields[i * 2 + 1] - fields[(i + 1) * 2 + 1]) <= max_start_sentry_bar_differences)
                        && (abs (fields[i * 2 + 1] - fields[(i + 2) * 2 + 1]) <= max_start_sentry_bar_differences)
                        && (fields[(i + 3) * 2 + 1] < fields[i * 2 + 1] << 3)) {
                    start_sentinel_i = i;
                    break;
                }
            }
        }
    }

    if (debug_batoo)
        printf ("start_sentinal_index: %d\n", start_sentinel_i);

    if (start_sentinel_i < 0)
        return NULL;

    // calculate the other positions:
    left_numbers_i = start_sentinel_i + 3;
    middle_guard_i = left_numbers_i + 6 * 4;
    right_numbers_i = middle_guard_i + 5;
    end_sentinel_i = right_numbers_i + 6 * 4;

    if (debug_batoo)
        printf ("end_sentinel %d end_i %d\n", end_sentinel_i, end_i);

    if (end_sentinel_i + 3 > end_i)
        return NULL;

    // calculate the average (pixel) length of a bar that is one unit wide:
    // (a complete  barcode consists out of 95 length units)
    int temp_length = 0;
    int field_amount = (end_sentinel_i - start_sentinel_i + 3);

    for (i = start_sentinel_i; i < start_sentinel_i + field_amount; i++)
        temp_length = temp_length + fields[i * 2 + 1];
    unit_length = (float) ((float) temp_length / (float) 95);

    // print out some debugging information:
    if (debug_batoo)
        printf ("unit_width: %f\n", unit_length);
    unsigned char current_number_field[4 * 2];

    if (left_numbers_i + 1 > end_i)
        return NULL;




    // test the side from which we are reading the barcode:
    MatchMakerResult *matchMakerResult;
    for (j = 0; j < 4; j++) {
        current_number_field[j * 2] = fields[(left_numbers_i + j) * 2];
        current_number_field[j * 2 + 1] = fields[(left_numbers_i + j) * 2 + 1];
    }
    matchMakerResult = recognize_number (current_number_field, BOTH_TABLES);



    if (matchMakerResult->even) {

        // we are reading the barcode from the back side:

        // use the already obtained information:
        numbers[12] = matchMakerResult->digit;
        free (matchMakerResult);

        // try to recognize the "right" numbers:
        int counter = 11;

        for (i = left_numbers_i + 4; i < left_numbers_i + 24; i = i + 4) {
            for (j = 0; j < 4; j++) {
                current_number_field[j * 2] = fields[(i + j) * 2];
                current_number_field[j * 2 + 1] = fields[(i + j) * 2 + 1];
            }
            matchMakerResult = recognize_number (current_number_field, EVEN_TABLE);
            numbers[counter] = matchMakerResult->digit;
            free (matchMakerResult);
            counter--;
        }

        bool parity_pattern[6];   // true = even, false = odd

        //(counter has now the value 6)

        // try to recognize the "left" numbers:
        for (i = right_numbers_i; i < right_numbers_i + 24; i = i + 4) {
            for (j = 0; j < 4; j++) {
                current_number_field[j * 2] = fields[(i + j) * 2];
                current_number_field[j * 2 + 1] = fields[(i + j) * 2 + 1];
            }
            matchMakerResult = recognize_number (current_number_field, BOTH_TABLES);
            numbers[counter] = matchMakerResult->digit;
            parity_pattern[counter - 1] = !matchMakerResult->even;
            free (matchMakerResult);
            counter--;
        }

        // try to determine the system code:
        matchMakerResult = recognize_system_code (parity_pattern);
        numbers[0] = matchMakerResult->digit;
        free (matchMakerResult);


    } else {

        // we are reading the abrcode from the "correct" side:

        bool parity_pattern[6];   // true = even, false = odd

        // use the already obtained information:
        numbers[1] = matchMakerResult->digit;
        parity_pattern[0] = matchMakerResult->even;
        free (matchMakerResult);

        // try to recognize the left numbers:
        int counter = 2;
        for (i = left_numbers_i + 4; i < left_numbers_i + 24; i = i + 4) {
            for (j = 0; j < 4; j++) {
                current_number_field[j * 2] = fields[(i + j) * 2];
                current_number_field[j * 2 + 1] = fields[(i + j) * 2 + 1];
            }
            matchMakerResult = recognize_number (current_number_field, BOTH_TABLES);
            numbers[counter] = matchMakerResult->digit;
            parity_pattern[counter - 1] = matchMakerResult->even;
            counter++;
            free (matchMakerResult);
        }

        // try to determine the system code:
        matchMakerResult = recognize_system_code (parity_pattern);
        numbers[0] = matchMakerResult->digit;
        free (matchMakerResult);

        // try to recognize the right numbers:
        counter = 0;
        for (i = right_numbers_i; i < right_numbers_i + 24; i = i + 4) {
            for (j = 0; j < 4; j++) {
                current_number_field[j * 2] = fields[(i + j) * 2];
                current_number_field[j * 2 + 1] = fields[(i + j) * 2 + 1];
            }
            matchMakerResult = recognize_number (current_number_field, ODD_TABLE);
            numbers[counter + 7] = matchMakerResult->digit;
            free (matchMakerResult);
            counter++;
        }

    }

    return numbers;

}

int *
analyse_row (int length, unsigned char *data) {
    unsigned char *row = (unsigned char *) malloc (length * sizeof (unsigned char));
    transform_to_bw (length, data, row);
    unsigned char *fields;
    int field_counter;
    extract_field_information (length, row, &fields, &field_counter);
    int *result = decode (fields, 0, field_counter, field_counter);
    if (result != NULL && !is_valid (result)) {
        free (result);
        result = NULL;
    }
    free (row);
    free (fields);
    return result;
}
