#include <QtGui>

#include "wifi.h"

#include <mbarcode-qt/plugininterfaces.h>
#include <mbarcode-qt/pluginaction.h>
#include <mbarcode-qt/maemobarcodewindow.h>

#include <gq/gconfitem.h>
#include <QUuid>

#include <glib-object.h>
#include <conic/conic.h>
#include <dbus/dbus-glib-lowlevel.h>

void WifiPlugin::initInterface(QWidget *parent)
{
    sink = new WifiSink(this);
    QWidget* widget = parent;
    connect(widget, SIGNAL(barcodeAnalysedSignal(QString,QString)),
            sink, SLOT(barcodeAnalysed(QString,QString)));
}

QSet<PluginAction*> WifiPlugin::getPluginActions()
{
    QSet<PluginAction*> newList;
    newList.insert(sink);
    return newList;
}

Q_EXPORT_PLUGIN2(mbqt_qrplugin, WifiPlugin)


WifiSink::WifiSink(const PluginInterface *interface) : PluginAction(interface)
{
    ready = false;
}

QString WifiSink::getText()
{
    QString text = tr("Connect to wifi network");
    if (ready)
        text += " \"" + wifiInfo.essid + "\"";
    return text;
}

void WifiSink::clickAction(QWidget * parentWindow)
{
    QString error;
    if (!createAndConnectIAP(wifiInfo, error))
        QMessageBox::critical(parentWindow, "Error", error);
}

void WifiSink::barcodeAnalysed(QString barcodeType, QString barcodeData)
{
    ready = false;
    if (barcodeType == "QR-Code" || barcodeType == "dmtx")
    {
        if (barcodeData.startsWith("WIFI:"))
        {
            QString wifiString = barcodeData.mid(5);
            if (parseWifiString(wifiString, wifiInfo))
                ready = true;
        }
    }
}

bool WifiSink::isReady()
{
    return ready;
}

bool WifiSink::extractPiece(QStringList& pieces, QString key, QString& value)
{
    for (int i = 0; i < pieces.size(); i++)
    {
        QStringList keyAndValue = pieces[i].split(":");
        if (keyAndValue.size() == 2 && keyAndValue[0] == key)
        {
            value = keyAndValue[1];
            return true;
        }
    }
    return false;
}

bool WifiSink::parseWifiString(QString str, WifiInfo& info)
{
    QStringList pieces = str.split(";", QString::SkipEmptyParts);
    QString essid, type, key;
    if (!extractPiece(pieces, "S", essid) || 
        !extractPiece(pieces, "T", type) || 
        !extractPiece(pieces, "P", key) ||
        (type != "WPA" && type != "WEP" && type != "nopass"))
    {
        return false;
    }

    if (type == "nopass")
        info.type = WifiInfo::Type_Open;
    else if (type == "WEP")
        info.type = WifiInfo::Type_WEP;
    else
        info.type = WifiInfo::Type_WPA;
    info.essid = essid;
    info.key = key;
    return true;
}

void set(QString uuid, QString key, QVariant value)
{
    GConfItem* item = new GConfItem("/system/osso/connectivity/IAP/" +
                                    uuid + "/" + key);
    item->set(value);
    delete item;
}

bool WifiSink::createAndConnectIAP(WifiInfo info, QString& error)
{
    // This is really ugly; Qt doesn't have APIs to create connections,
    // and the Qt Mobility APIs to connect an IAP don't work. Use
    // glib stuff for now; hopefully we can improve this at some point.

    // First, make sure that we don't already have
    // a connection with this name.
    bool exists = false;
    ConIcConnection* conn = con_ic_connection_new();
    GSList* conns = con_ic_connection_get_all_iaps(conn);
    for (GSList* cur = conns; cur != NULL; cur = cur->next)
    {
        if (cur->data)
        {
            ConIcIap* iap = CON_IC_IAP(cur->data);
            if (info.essid == con_ic_iap_get_name(iap))
                exists = true;
            g_object_unref(iap);
        }
    }
    g_slist_free(conns);
    if (exists)
    {
        error = tr("An internet connection with this name already exists.");
        return false;
    }
    QString uuid = QUuid::createUuid().toString();
    uuid = uuid.remove("{").remove("}");

    // Set IAP settings
    QList<QVariant> list;
    for (int i = 0; i < info.essid.length(); i++)
        list.append((int)info.essid[i].toLatin1());
    set(uuid, "ipv4_type", "AUTO");
    set(uuid, "wlan_ssid", list);
    set(uuid, "name", info.essid);
    set(uuid, "type", "WLAN_INFRA");
    set(uuid, "proxytype", "NONE");
    set(uuid, "wlan_hidden", false);
    set(uuid, "temporary", false);
    switch (info.type)
    {
    case WifiInfo::Type_WEP:
        set(uuid, "wlan_security", "WEP");
        set(uuid, "wlan_wepkey1", info.key);
        set(uuid, "wlan_wepdefkey", 1);
        break;
    case WifiInfo::Type_WPA:
        set(uuid, "wlan_security", "WPA_PSK");
        set(uuid, "EAP_wpa_preshared_passphrase", info.key);
        break;
    case WifiInfo::Type_Open:
        break;
    }

    DBusError dbusError;
    dbus_error_init(&dbusError);
    DBusConnection* dbusConn = dbus_bus_get(DBUS_BUS_SYSTEM, &dbusError);

    if (dbusConn)
        dbus_connection_setup_with_g_main(dbusConn, NULL);
    else
    {
        error = QString(tr("Unable to connect to the D-BUS daemon: ")) +
                 dbusError.message;
        dbus_error_free(&dbusError);
        return false;
    }
         
    // Disconnect any existing connection
    QDBusConnection bus = QDBusConnection::systemBus();
    QDBusInterface method("com.nokia.icd2", "/com/nokia/icd2",
                          "com.nokia.icd2", bus);
    method.call("disconnect_req", (uint)0x8000);
    sleep(5);
    // Connect to the new network
    if (!con_ic_connection_connect_by_id(conn, uuid.toAscii().data(),
        CON_IC_CONNECT_FLAG_NONE))
    {
        error = tr("Unable to connect to network.");
        return false;
    }
    g_object_unref(conn);

    return true;
}

