/*
  MathJinni - A simple formular calculator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Formula.h"

#include <Lum/Base/String.h>

#include <Lum/Def/OneOfMany.h>

#include <Lum/Manager/Behaviour.h>

#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/Wizard.h>

#include "Util.h"

#define LEADING_PAGES 1
#define TRAILING_PAGES 1

void ListCategories(const std::wstring& category, std::vector<std::wstring>& subs)
{
  size_t start,pos;

  start=0;
  pos=0;

  subs.clear();

  while (pos<category.length()) {
    if (category[pos]=='/') {
      subs.push_back(category.substr(start,pos-start));
      start=pos+1;
    }
    pos++;
  }

  if (start<category.length()) {
    subs.push_back(category.substr(start,pos-start));
  }
}

class FormulaModelPainter : public Lum::TableCellPainter
{
private:
  Lum::OS::FontRef font1;
  Lum::OS::FontRef font2;

public:
  FormulaModelPainter()
  : font1(Lum::OS::display->GetFont()),
    font2(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote))
  {
    // no code
  }

  size_t GetProposedHeight() const
  {
    return Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
           font1->height+
           font2->height+
           Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);
  }

  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const Formula* formula=dynamic_cast<const FormulaModel*>(GetModel())->GetEntry(GetRow());

    draw->PushFont(font1,Lum::OS::Font::bold);
    draw->PushForeground(GetTextColor(draw));
    draw->DrawString(x,
                     y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->ascent,
                     formula->GetName());
    draw->PopForeground();
    draw->PopFont();

    draw->PushFont(font1);
    draw->PushForeground(GetTextColor(draw));
    draw->DrawString(x+font1->StringWidth(formula->GetName(),Lum::OS::Font::bold),
                     y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->ascent,
                     std::wstring(L" (")+formula->GetFormula()+L")");
    draw->PopForeground();
    draw->PopFont();

    draw->PushFont(font2);
    draw->PushForeground(GetTextColor(draw));
    draw->DrawString(x,
                     y+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font1->height+
                     font2->ascent,
                     formula->GetDescription());
    draw->PopForeground();
    draw->PopFont();
  }
};

class FormulaFillDialog : public Lum::Dialog
{
private:
  Parser                             parser;

  Formula                            *formula;
  std::vector<Lum::Model::ULongRef>  srcUnits;
  std::vector<Lum::Model::StringRef> srcValues;
  Lum::Model::ULongRef               resUnit;
  std::vector<Lum::Model::ActionRef> stepActions;
  Lum::Model::StringRef              result;

public:
  FormulaFillDialog(Formula* formula)
  : formula(formula),
    result(new Lum::Model::String(L""))
  {
    Observe(GetClosedAction());
  }

  void PreInit()
  {
    Lum::Object                *oneOfMany;
    Lum::Label                 *label;
    Lum::Panel                 *panel;
    Lum::String                *string;
    Lum::Text                  *text;
    Lum::TextValue             *textValue;
    Lum::Wizard                *wizard;

    Lum::Model::ActionRef      stepAction;
    Lum::Model::StringTableRef resUnitList;
    Unit                       *unit;

    wizard=new Lum::Wizard();
    wizard->SetCancelAction(GetClosedAction());

    // Initial page

    panel=Lum::VPanel::Create(true,true);

    text=new Lum::Text(L"You are now calculating the following formula:");
    text->SetFlex(true,false);
    panel->Add(text);

    panel->AddSpace();

    label=Lum::Label::Create(true,false);
    label->AddLabel(L"Formula:",formula->GetFormula());
    label->AddLabel(L"Description:",formula->GetDescription());

    resUnit=new Lum::Model::ULong();
    resUnitList=new Lum::Model::StringTable();
    unit=unitMap[formula->GetUnit()];

    for (size_t j=0; j<unit->GetAlternativeCount(); j++) {
      if (resUnit->IsNull() &&
          unit->GetAlternative(j)->GetFrom()==L"x" &&
          unit->GetAlternative(j)->GetTo()==L"x") {
        resUnit->Set(j+1);
      }
      resUnitList->Append(unit->GetAlternative(j)->GetName());
    }

    if (resUnit->IsNull()) {
      resUnit->Set(1);
    }

    Lum::Def::OneOfMany resUnitDef(Lum::Def::Desc(L"Unit"),
                                resUnit,resUnitList);

    oneOfMany=Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(resUnitDef);
    oneOfMany->SetFlex(true,false);
    label->AddLabel(L"Unit:",oneOfMany);

    panel->Add(label);

    stepAction=new Lum::Model::Action();
    Observe(stepAction);
    stepActions.push_back(stepAction);

    wizard->AddPage(L"Unit of Result",panel,stepAction);

    for (size_t i=0; i<formula->GetVariables().size(); i++) {
      Lum::Model::ULongRef       srcUnit=new Lum::Model::ULong();
      Lum::Model::StringTableRef srcUnitList=new Lum::Model::StringTable();

      Lum::Model::StringRef value=new Lum::Model::String(L"");

      Observe(value);

      unit=unitMap[formula->GetVariables()[i]->GetUnit()];

      for (size_t j=0; j<unit->GetAlternativeCount(); j++) {
        if (srcUnit->IsNull() &&
            unit->GetAlternative(j)->GetFrom()==L"x" &&
            unit->GetAlternative(j)->GetTo()==L"x") {
          srcUnit->Set(j+1);
        }

        srcUnitList->Append(unit->GetAlternative(j)->GetName());
      }

      if (resUnit->IsNull()) {
        resUnit->Set(1);
      }

      panel=Lum::VPanel::Create(true,true);

      label=Lum::Label::Create(true,false);
      label->AddLabel(L"Formula:",formula->GetFormula());
      label->AddLabel(L"Variable:",formula->GetVariables()[i]->GetName());
      label->AddLabel(L"Description:",formula->GetVariables()[i]->GetDescription());

      string=new Lum::String();
      string->SetFlex(true,false);
      string->SetAlignment(Lum::String::right);
      string->SetInputTypeHint(Lum::Object::inputTypeSignedFloatNumber);
      string->SetModel(value);
      label->AddLabel(L"Value:",string);

      Lum::Def::OneOfMany srcUnitDef(Lum::Def::Desc(L"Unit"),
                                     srcUnit,srcUnitList);

      oneOfMany=Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(srcUnitDef);
      oneOfMany->SetFlex(true,false);
      label->AddLabel(L"Unit:",oneOfMany);

      panel->Add(label);

      stepAction=new Lum::Model::Action();
      Observe(stepAction);
      stepAction->Disable();
      stepActions.push_back(stepAction);

      wizard->AddPage(L"Value and unit of variable",panel,stepAction);

      srcValues.push_back(value);
      srcUnits.push_back(srcUnit);
    }

    panel=Lum::VPanel::Create(true,true);

    label=Lum::Label::Create(true,false);
    label->AddLabel(L"Formula:",formula->GetFormula());

    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetModel(result);

    label->AddLabel(L"Result:",textValue);

    panel->Add(label);

    wizard->AddPage(L"Result",panel,GetClosedAction());

    SetMain(wizard);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else {
      for (size_t i=0; i<srcValues.size(); i++) {
        if (model==srcValues[i]) {
          Parser::Expression *expression=NULL;
          Parser::ReturnCode returnCode;
          size_t             pos;

          if (!srcValues[i]->Get().empty()) {
            expression=parser.Parse(Lum::Base::WStringToString(srcValues[i]->Get()),
                                    returnCode,pos,false);
          }

          if (expression==NULL) {
            stepActions[LEADING_PAGES+i]->Disable();
          }
          else {
            stepActions[LEADING_PAGES+i]->Enable();
          }

          delete expression;

          break;
        }
      }

      for (size_t i=LEADING_PAGES; i<stepActions.size(); i++) {
        if (model==stepActions[i] && stepActions[i]->IsFinished()) {
          Parser unitParser;
          Parser::Variable* variable=unitParser.AddVariable("x",0.0);

          for (size_t j=0; j<i; j++) {
            Unit               *unit;
            Parser::Expression *expression;
            Parser::ReturnCode returnCode;
            size_t             pos;
            bool               success;
            double             value;

            // Value
            expression=parser.Parse(Lum::Base::WStringToString(srcValues[j]->Get()),
                                    returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(value,returnCode);
            assert(success);
            delete expression;

            // Factor
            unit=unitMap[formula->GetVariables()[j]->GetUnit()];
            variable->value=value;
            expression=unitParser.Parse(Lum::Base::WStringToString(unit->GetAlternative(srcUnits[j]->Get()-1)->GetFrom()),
                                        returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(value,returnCode);
            assert(success);
            delete expression;

            parser.AddConstant(Lum::Base::WStringToString(formula->GetVariables()[j]->GetName()),
                               value);
          }

          if (i==stepActions.size()-1) {
            Unit               *unit;
            Parser::Expression *expression;
            Parser::ReturnCode returnCode;
            size_t             pos;
            bool               success;
            double             value;

            // Value
            expression=parser.Parse(Lum::Base::WStringToString(formula->GetFormula()),
                                    returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(value,returnCode);
            assert(success);
            delete expression;

            //Factor
            unit=unitMap[formula->GetUnit()];
            variable->value=value;
            expression=unitParser.Parse(Lum::Base::WStringToString(unit->GetAlternative(resUnit->Get()-1)->GetTo()),
                                        returnCode,pos,false);
            assert(expression!=NULL);
            success=expression->Calculate(value,returnCode);
            assert(success);
            delete expression;

            result->Set(DoubleToWStringFloat(value)+L" "+unit->GetAlternative(resUnit->Get()-1)->GetName());
          }

          break;
        }
      }
    }

    Lum::Dialog::Resync(model,msg);
  }
};

FormulaCalc::FormulaCalc()
 : selectAction(new Lum::Model::Action()),
   executeAction(new Lum::Model::Action()),
   category(new Lum::Model::ULong()),
   subCategory(new Lum::Model::ULong()),
   categories(new Lum::Model::StringTable()),
   subCategories(new Lum::Model::StringTable()),
   functions(new FormulaModel(NULL)),
   function(new Lum::Model::SingleLineSelection())
{
  Observe(selectAction);
  Observe(executeAction);

  Observe(category);
  Observe(subCategory);
  Observe(function);
}

void FormulaCalc::CalcSize()
{
  Lum::Label            *label;
  Lum::Model::HeaderRef headerModel;
  Lum::Table            *table;
  Lum::Def::OneOfMany   categoryDef(Lum::Def::Desc(L"Category"),
                                    category,categories);
  Lum::Def::OneOfMany   subCategoryDef(Lum::Def::Desc(L"Subcategory"),
                                       subCategory,subCategories);

  label=new Lum::Label();

  label->AddLabel(L"Category:",
                  Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(categoryDef,true,false));
  label->AddLabel(L"Subcategory:",
                  Lum::Manager::Behaviour::Instance()->GetOneOfManyControl(subCategoryDef,true,false));

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Function",Lum::Base::Size::stdCharWidth,30,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,6);
  table->GetTableView()->SetAutoFitColumns(true);
  table->GetTableView()->SetAutoHSize(true);
  table->SetShowHeader(false);
  table->SetModel(functions);
  table->SetHeaderModel(headerModel);
  table->SetSelection(function);
  table->SetDoubleClickAction(executeAction);
  table->SetPainter(new FormulaModelPainter());
  label->AddLabel(L"Functions:",table);

  container=label;

  Component::CalcSize();
}

void FormulaCalc::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==category) {
    std::set<std::wstring> subCategories;

    for (size_t i=0; i<formulas.size(); i++) {
      std::vector<std::wstring> subs;

      ListCategories(formulas[i]->GetCategory(),subs);

      if (subs.size()!=2) {
        continue;
      }

      subCategories.insert(subs[1]);
    }

    this->subCategories->Off();
    this->subCategories->Clear();
    for (std::set<std::wstring>::const_iterator c=subCategories.begin();
         c!=subCategories.end();
         c++) {
      this->subCategories->Append(*c);
    }
    this->subCategories->On();

    if (subCategories.size()>0) {
      subCategory->Set(1);
    }
    else {
      subCategory->SetNull();
    }
  }
  else if (model==subCategory) {
    functions->Off();
    functions->Clear();

    if (!category->IsNull() &&
        !subCategory->IsNull()) {
      std::wstring c=categories->GetEntry(category->Get())+L"/"+
                     subCategories->GetEntry(subCategory->Get());
      for (size_t i=0; i<formulas.size(); i++) {
        if (formulas[i]->GetCategory()==c) {
          functions->Append(formulas[i]);
        }
      }
    }

    functions->On();
  }
  else if (model==executeAction &&
           executeAction->IsFinished()) {
    if (function->HasSelection()) {
      Formula* formula=functions->GetEntry(function->GetLine());

      FormulaFillDialog *dialog;

      dialog=new FormulaFillDialog(formula);
      dialog->SetParent(GetWindow());
      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
  }
}

void FormulaCalc::LoadData()
{
  std::set<std::wstring> categories;

  for (size_t i=0; i<formulas.size(); i++) {
    std::vector<std::wstring> subs;

    ListCategories(formulas[i]->GetCategory(),subs);

    if (subs.size()!=2) {
      continue;
    }

    categories.insert(subs[0]);
  }

  this->categories->Off();
  for (std::set<std::wstring>::const_iterator c=categories.begin();
       c!=categories.end();
       c++) {
    this->categories->Append(*c);
  }
  this->categories->On();

  category->Set(1);
}

void FormulaCalc::LoadConfig(Lum::Config::Node* /*top*/)
{
  // no code
}

void FormulaCalc::StoreConfig(Lum::Config::Node *top)
{
  // no code
}

