/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FormatConversion.h"

#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>

static Lum::String* CreateString()
{
  Lum::String *string;

  string=new Lum::String();
  string->SetFlex(true,false);
  string->SetAlignment(Lum::String::right);

  return string;
}

FormatConversion::FormatConversion()
: binFormat(new Lum::Model::String(L"")),
  octFormat(new Lum::Model::String(L"")),
  decFormat(new Lum::Model::String(L"")),
  hexFormat(new Lum::Model::String(L"")),
  formatConversion(false)
{
  Observe(binFormat);
  Observe(octFormat);
  Observe(decFormat);
  Observe(hexFormat);
}

void FormatConversion::ConvertFromBin()
{
  unsigned long value;

  if (binFormat->Empty() || !Lum::Base::WStringToNumber(binFormat->Get(),value,2)) {
    octFormat->Set(L"");
    decFormat->Set(L"");
    hexFormat->Set(L"");
    return;
  }

  formatConversion=true;
  octFormat->Set(Lum::Base::NumberToWString(value,8));
  decFormat->Set(Lum::Base::NumberToWString(value,10));
  hexFormat->Set(Lum::Base::NumberToWString(value,16));
  formatConversion=false;
}

void FormatConversion::ConvertFromOct()
{
  unsigned long value;

  if (octFormat->Empty() || !Lum::Base::WStringToNumber(octFormat->Get(),value,8)) {
    binFormat->Set(L"");
    decFormat->Set(L"");
    hexFormat->Set(L"");
    return;
  }

  formatConversion=true;
  binFormat->Set(Lum::Base::NumberToWString(value,2));
  decFormat->Set(Lum::Base::NumberToWString(value,10));
  hexFormat->Set(Lum::Base::NumberToWString(value,16));
  formatConversion=false;
}

void FormatConversion::ConvertFromDec()
{
  unsigned long value;

  if (decFormat->Empty() || !Lum::Base::WStringToNumber(decFormat->Get(),value,10)) {
    binFormat->Set(L"");
    octFormat->Set(L"");
    hexFormat->Set(L"");
    return;
  }

  formatConversion=true;
  binFormat->Set(Lum::Base::NumberToWString(value,2));
  octFormat->Set(Lum::Base::NumberToWString(value,8));
  hexFormat->Set(Lum::Base::NumberToWString(value,16));
  formatConversion=false;
}

void FormatConversion::ConvertFromHex()
{
  unsigned long value;

  if (hexFormat->Empty() || !Lum::Base::WStringToNumber(hexFormat->Get(),value,16)) {
    binFormat->Set(L"");
    octFormat->Set(L"");
    decFormat->Set(L"");
    return;
  }

  formatConversion=true;
  binFormat->Set(Lum::Base::NumberToWString(value,2));
  octFormat->Set(Lum::Base::NumberToWString(value,8));
  decFormat->Set(Lum::Base::NumberToWString(value,10));
  formatConversion=false;
}


void FormatConversion::CalcSize()
{
  Lum::Label  *label;
  Lum::Panel  *panel;

  panel=new Lum::VPanel();

  label=Lum::Label::Create(true,false);

  bin=CreateString();
  bin->SetInputTypeHint(Lum::Object::inputTypeUnsignedBinaryNumber);
  bin->SetModel(binFormat);
  label->AddLabel(L"Binary",bin);

  oct=CreateString();
  oct->SetInputTypeHint(Lum::Object::inputTypeUnsignedOctalNumber);
  oct->SetModel(octFormat);
  label->AddLabel(L"Octal",oct);

  dec=CreateString();
  dec->SetInputTypeHint(Lum::Object::inputTypeUnsignedDecimalNumber);
  dec->SetModel(decFormat);
  label->AddLabel(L"Decimal",dec);

  hex=CreateString();
  hex->SetInputTypeHint(Lum::Object::inputTypeUnsignedHexadecimalNumber);
  hex->SetModel(hexFormat);
  label->AddLabel(L"Hex",hex);

  panel->Add(label);

  container=panel;

  Component::CalcSize();
}

void FormatConversion::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==binFormat && !formatConversion) {
    ConvertFromBin();
  }
  else if (model==octFormat && !formatConversion) {
    ConvertFromOct();
  }
  else if (model==decFormat && !formatConversion) {
    ConvertFromDec();
  }
  else if (model==hexFormat && !formatConversion) {
    ConvertFromHex();
  }

  Lum::Component::Resync(model,msg);
}
