/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FunctionInput.h"

#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/Grid.h>
#include <Lum/Panel.h>
#include <Lum/Table.h>

class ButtonPopup : public Lum::Popup
{
protected:
  std::map<Lum::Model::Action*,std::wstring> actionTextMap;
  std::wstring                               result;

protected:
  Lum::Object* CreateTextButton(const std::wstring& label,
                                const std::wstring& text=L"")
  {
    Lum::Model::ActionRef action;

    action=new Lum::Model::Action();

    Observe(action);

    if (text.empty()) {
      actionTextMap[action.Get()]=label;
    }
    else {
      actionTextMap[action.Get()]=text;
    }

    return Lum::Button::Create(label,action,true,true);
  }

public:
  void Resync(Lum::Base::Model* model,
              const Lum::Base::ResyncMsg& msg)
  {
    std::map<Lum::Model::Action*,std::wstring>::const_iterator iter;

    iter=actionTextMap.find(dynamic_cast<Lum::Model::Action*>(model));
    if (iter!=actionTextMap.end()) {
      result=iter->second;
      Exit();
    }

    Popup::Resync(model,msg);
  }

  std::wstring GetResult()
  {
    return result;
  }
};

class ConstantPopup : public ButtonPopup
{
public:
  void PreInit()
  {
    Lum::Grid *grid;

    grid=new Lum::Grid();
    grid->SetFlex(true,true);
    grid->SetSize(2,1);
    grid->SetEqualDimensions(true,true);
    grid->SetSpace(true,true);

    grid->SetObject(0,0,CreateTextButton(L"\x03c0"/*L"pi"*/,L"pi"));
    grid->SetObject(1,0,CreateTextButton(L"e"));

    SetMain(grid,false,true);

    ButtonPopup::PreInit();
  }
};

class FunctionPopup : public ButtonPopup
{
public:
  void PreInit()
  {
    Lum::Grid *grid;

    grid=new Lum::Grid();
    grid->SetFlex(true,true);
    grid->SetSize(3,1);
    grid->SetEqualDimensions(true,true);
    grid->SetSpace(true,true);

    grid->SetObject(0,0,CreateTextButton(L"^"));
    grid->SetObject(1,0,CreateTextButton(L"ln",L"ln("));
    grid->SetObject(2,0,CreateTextButton(L"lg",L"lg("));

    grid->SetObject(0,1,CreateTextButton(L"\x221a"/*L"sqrt"*/,L"sqrt("));
    grid->SetObject(1,1,CreateTextButton(L"abs",L"abs("));

    SetMain(grid,false,true);

    ButtonPopup::PreInit();
  }
};

class TrigometricPopup : public ButtonPopup
{
public:
  void PreInit()
  {
    Lum::Grid *grid;

    grid=new Lum::Grid();
    grid->SetFlex(true,true);
    grid->SetSize(3,1);
    grid->SetEqualDimensions(true,true);
    grid->SetSpace(true,true);

    grid->SetObject(0,0,CreateTextButton(L"sin",L"sin("));
    grid->SetObject(1,0,CreateTextButton(L"cos",L"cos("));
    grid->SetObject(2,0,CreateTextButton(L"tan",L"tan("));

    grid->SetObject(0,1,CreateTextButton(L"asin",L"asin("));
    grid->SetObject(1,1,CreateTextButton(L"acos",L"acos("));
    grid->SetObject(2,1,CreateTextButton(L"atan",L"atan("));

    grid->SetObject(0,2,CreateTextButton(L"sinh",L"sinh("));
    grid->SetObject(1,2,CreateTextButton(L"cosh",L"cosh"));
    grid->SetObject(2,2,CreateTextButton(L"tanh",L"tanh"));

    grid->SetObject(0,3,CreateTextButton(L"asinh",L"asinh("));
    grid->SetObject(1,3,CreateTextButton(L"acosh",L"acosh("));
    grid->SetObject(2,3,CreateTextButton(L"atanh",L"atanh("));

    SetMain(grid,false,true);

    ButtonPopup::PreInit();
  }
};

FunctionInput::FunctionInput()
 : constantAction(new Lum::Model::Action()),
   functionAction(new Lum::Model::Action()),
   trigometricAction(new Lum::Model::Action()),
   backspaceAction(new Lum::Model::Action()),
   clearAction(new Lum::Model::Action())
{
  Observe(constantAction);
  Observe(functionAction);
  Observe(trigometricAction);
  Observe(backspaceAction);
  Observe(clearAction);
}

void FunctionInput::SetString(Lum::String *string)
{
  this->string=string;
}

void FunctionInput::SetCalculateAction(Lum::Model::Action* action)
{
  calculateAction=action;
}

Lum::Object* FunctionInput::CreateTextButton(const std::wstring& label, const std::wstring& text)
{
  Lum::Model::ActionRef action;

  action=new Lum::Model::Action();

  Observe(action);

  if (text.empty()) {
    actionTextMap[action.Get()]=label;
  }
  else {
    actionTextMap[action.Get()]=text;
  }

  return Lum::Button::Create(label,action,true,true);
}

void FunctionInput::CalcSize()
{
  Lum::Grid   *grid;
  Lum::Panel  *panel;
  Lum::Panel  *panel2;

  panel=Lum::HPanel::Create(true,true);

  panel2=Lum::VPanel::Create(false,true);
  panel2->Add(constantButton=Lum::Button::Create(L"Constants",constantAction,true,false));
  panel2->AddSpace();
  panel2->Add(functionButton=Lum::Button::Create(L"Functions",functionAction,true,false));
  panel2->AddSpace();
  panel2->Add(trigometricButton=Lum::Button::Create(L"Trigonomie",trigometricAction,true,false));
  panel2->AddSpace(true);
  panel->Add(panel2);
  panel->AddSpace(true);

  grid=new Lum::Grid();
  grid->SetFlex(true,true);
  grid->SetSize(5,4);
  grid->SetEqualDimensions(true,true);
  grid->SetSpace(true,true);

  grid->SetObject(0,0,CreateTextButton(L"7"));
  grid->SetObject(1,0,CreateTextButton(L"8"));
  grid->SetObject(2,0,CreateTextButton(L"9"));
  grid->SetObject(3,0,CreateTextButton(L"/"));
  grid->SetObject(4,0,Lum::Button::Create(L"<-",backspaceAction,true,true));

  grid->SetObject(0,1,CreateTextButton(L"4"));
  grid->SetObject(1,1,CreateTextButton(L"5"));
  grid->SetObject(2,1,CreateTextButton(L"6"));
  grid->SetObject(3,1,CreateTextButton(L"*"));
  grid->SetObject(4,1,Lum::Button::Create(L"Clr",clearAction,true,true));

  grid->SetObject(0,2,CreateTextButton(L"1"));
  grid->SetObject(1,2,CreateTextButton(L"2"));
  grid->SetObject(2,2,CreateTextButton(L"3"));
  grid->SetObject(3,2,CreateTextButton(L"\x2212",L"-"));
  grid->SetObject(4,2,CreateTextButton(L"("));

  grid->SetObject(0,3,CreateTextButton(L"0"));
  grid->SetObject(1,3,CreateTextButton(Lum::Base::GetDecimalPointWString()));
  if (calculateAction.Valid()) {
    grid->SetObject(2,3,Lum::Button::Create(L"=",calculateAction,true,true));
  }
  else {
    grid->SetObject(2,3,CreateTextButton(L"x"));
  }
  grid->SetObject(3,3,CreateTextButton(L"+"));
  grid->SetObject(4,3,CreateTextButton(L")"));

  panel->Add(grid);

  container=panel;

  Component::CalcSize();
}

void FunctionInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==constantAction &&
      constantAction->IsFinished()) {
    ConstantPopup* popup=new ConstantPopup();

    if (popup!=NULL) {
      popup->SetParent(GetWindow());
      popup->SetReference(constantButton);

      if (popup->Open()) {
        popup->EventLoop();
        popup->Close();
      }

      if (!popup->GetResult().empty()) {
        string->ActionInsertString(popup->GetResult());
      }

      delete popup;
    }
  }
  else if (model==functionAction &&
           functionAction->IsFinished()) {
    FunctionPopup* popup=new FunctionPopup();

    if (popup!=NULL) {
      popup->SetParent(GetWindow());
      popup->SetReference(functionButton);

      if (popup->Open()) {
        popup->EventLoop();
        popup->Close();
      }

      if (!popup->GetResult().empty()) {
        string->ActionInsertString(popup->GetResult());
      }

      delete popup;
    }
  }
  else if (model==trigometricAction &&
           trigometricAction->IsFinished()) {
    TrigometricPopup* popup=new TrigometricPopup();

    if (popup!=NULL) {
      popup->SetParent(GetWindow());
      popup->SetReference(functionButton);

      if (popup->Open()) {
        popup->EventLoop();
        popup->Close();
      }

      if (!popup->GetResult().empty()) {
        string->ActionInsertString(popup->GetResult());
      }

      delete popup;
    }
  }
  else if (model==backspaceAction && backspaceAction->IsFinished()) {
    string->ActionBackspace();
  }
  else if (model==clearAction && clearAction->IsFinished()) {
    string->ActionEraseAll();
  }
  else if (dynamic_cast<Lum::Model::Action*>(model)!=NULL &&
           dynamic_cast<Lum::Model::Action*>(model)->IsFinished()) {
    std::map<Lum::Model::Action*,std::wstring>::const_iterator iter;

    iter=actionTextMap.find(dynamic_cast<Lum::Model::Action*>(model));
    if (iter!=actionTextMap.end()) {
      string->ActionInsertString(iter->second);
    }
  }

  Component::Resync(model,msg);
}

