#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
#    masstransit.py 
#
#    This file is part of Mass-Transit
#
#    Copyright (C) 2010 Thibault Cohen
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from PyQt4 import QtCore, QtGui
import sys
import os
from datetime import datetime
from lib import lib_masstransit as masstransit
from lib import client
from lib import config
from ui import stopcodelist,lineslist,stopcode,favoritelist
import osso

SERVER_LIST = ['http://titilambert.org:5000',
                ]

class MenuBar(QtGui.QMenuBar):
    def __init__(self, parent=None, ):
        QtGui.QMenuBar.__init__(self, parent)
        self.setObjectName("Menu")
        ### Bus lines
        self.bus_lines = self.addAction(self.tr("&Bus Lines"))
        ### Stop lists
        self.stop_list = self.addAction(self.tr("Stop &List"))
        ### Stop code
        self.stop_code = self.addAction(self.tr("&Stop Code"))
        ### Change server
        self.change_server = self.addAction(self.tr("Change Server"))
        ### Change company
        self.change_company = self.addAction(self.tr("Change Company"))
        ### Subway
        #self.subway = self.addAction(self.tr("Subway"))
        ## Quit
        self.quit = self.addAction(self.tr("&Quit"))


class Ui_MainWindow(QtCore.QObject):
    def __init__(self, app):
        QtCore.QObject.__init__(self)
        self.app = app
        self.config = config.Config(parent=self)
        self.mode = "favorite"

    def change_mode(self, mode="favorites", parent=None, **kwargs):
        root = self
        if self.mode == mode:
            masstransit.banner_notification("Please wait... Processing ")
            return False
        if not parent or parent == self:
            parent = self.main_window
        if mode == "bus_lines":
            ui_lines_list = lineslist.UI_LinesList(parent, root)
            ui_lines_list.show()
        elif mode == "stop_code":
            self.mode = "stop_code"
            ui_stop_code = stopcode.Ui_StopCode(kwargs["stop_code"], False, parent, root)
            ui_stop_code.show()
        elif mode == "stop_list":
            ui_stop_list = stopcodelist.UI_StopCodeList(kwargs["line"], kwargs["direction"], parent, root)
            ui_stop_list.show()
        elif mode == "favorites":
            ui_fav_list = favoritelist.UI_FavoriteList(parent, root)
            self.main_window.setCentralWidget(ui_fav_list)
            self.main_window.setWindowTitle("Mass-Transit - Favorites")
        self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
        self.mode = mode

    def setupUi(self, main_window):
        self.main_window = main_window
        self.main_window.setObjectName("MainWindow")
        self.main_window.resize(800, 400)

        ### Central Widget
        self.change_mode("favorites", self)
        ### Menu
        self.menubar = MenuBar(self.main_window)
        self.main_window.setMenuBar(self.menubar)

        QtCore.QMetaObject.connectSlotsByName(self.main_window)

        # Signals
        ## Stop List
        QtCore.QObject.connect(self.menubar.stop_list, QtCore.SIGNAL("triggered()"), self.stop_list)
        ## Favorite
        QtCore.QObject.connect(self.menubar.bus_lines, QtCore.SIGNAL("triggered()"), self.bus_lines)
        ## Stop code
        QtCore.QObject.connect(self.menubar.stop_code, QtCore.SIGNAL("triggered()"), self.stop_code)
        ## Subway
        #QtCore.QObject.connect(self.menubar.subway, QtCore.SIGNAL("triggered()"), self.subway)
        ## Change server
        QtCore.QObject.connect(self.menubar.change_server, QtCore.SIGNAL("triggered()"), self.change_server_address)
        ## Change company
        QtCore.QObject.connect(self.menubar.change_company, QtCore.SIGNAL("triggered()"), self.change_company)
        ## Quit
        QtCore.QObject.connect(self.menubar.quit, QtCore.SIGNAL("triggered()"), self.app.quit)

        self.config.check_config()
        server_version = client.get_server_version()
        if 'result' in server_version.keys():
            if not server_version['result'] == '0.5':
                masstransit.banner_notification("WARNING: Server version doesn't seem good.")
        else:
            # TODO Change it by server error message
            masstransit.banner_notification("ERROR: There is an error on the server.")
        server_version = client.get_protocol_version()
        if 'result' in server_version.keys():
            if not server_version['result'] == '0.5':
                masstransit.banner_notification("WARNING: Protocol version doesn't seem good.")
        else:
            # TODO Change it by server error message
            masstransit.banner_notification("ERROR: There is an error on the server.")
        
    def change_server_address(self):
        current_server_address = self.config.value("server").toString()
        try:
            select_index = SERVER_LIST.index(current_server_address)
        except ValueError:      
            select_index = 0
        server_address, bool = QtGui.QInputDialog().getItem(
                                self.main_window,
                                "Mass-Transit server address",
                                "Enter server address (http://mass-transit.org)",
                                SERVER_LIST,
                                select_index,
                                False,
                                )
        if not bool and current_server_address == "":
            self.app.quit()
            sys.exit(0)
        else:
            self.config.setValue("server", QtCore.QVariant(server_address))
            self.config.sync()

    def change_company(self):
        # TODO Create a good input dialog .... with combo box
        current_company_name = self.config.value("company_name").toString()
        current_company_id = self.config.value("company_id").toString()
        available_company = [value + " #" + key for key, value in client.get_supported_company()["result"].items()]
        available_company.sort()
        try:
            select_index = available_company.index(current_company_name + " #" + current_company_id)
        except ValueError:
            select_index = 0
        company,bool = QtGui.QInputDialog().getItem(
                                self.main_window,
                                "Select your town",
                                "Select a company available on the server",
                                available_company,
                                select_index,
                                False,
                                )
        if not bool and current_company_name == "":
            self.app.quit()
            sys.exit(0)
        else:
            company_name,company_id = company.split("#")
            self.config.setValue("company_name", QtCore.QVariant(company_name))
            self.config.setValue("company_id", QtCore.QVariant(company_id))
            self.config.sync()

    def subway(self):
        masstransit.banner_notification("Subway not yet implemented")

    def stop_code(self, parent=None):
        self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, True)
        code,bool = QtGui.QInputDialog().getText(
                                self.main_window,
                                "Stop Code Title",
                                "Stop Code",
                                QtGui.QLineEdit.Normal,
                                "",
                                )
        if not bool:
            self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
            return False

        else:
            try:
                int(code)
            except:
                self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

        try:
            self.change_mode(mode="stop_code",stop_code=str(code),parent=parent)
        except:
            masstransit.banner_notification("That bus stop code is invalid. Please try again.")
            self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)

            return False


    def bus_lines(self):
        self.change_mode(mode="bus_lines",parent=self)

    def stop_list(self, line=None, parent=None):
        if parent == None:
            parent = self.main_window
        parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, True)
        if line == None:
            line,bool = QtGui.QInputDialog().getText(
                                    self.main_window,
                                    "Line number Title",
                                    "Line number",
                                    QtGui.QLineEdit.Normal,
                                    "",
                                    )
            if not bool or line == "":
                parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

            else:
                try:
                    int(line)
                except:
                    return False

        directions = client.get_bus_directions(self.company_id)['result']
        direction,bool = QtGui.QInputDialog().getItem(
                                self.main_window,
                                "Direction Title",
                                "Direction",
                                directions,
                                0,
                                False,
                                )
        if not bool:
            parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
            return False
        else:
            try:
                str(direction)
            except:
                parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

        try:
            self.change_mode(mode="stop_list", line=line, direction=direction, parent=parent)
        except:
            masstransit.banner_notification("That bus route number or direction is invalid. Please try again.")
            parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)

            return False


def main():
    app = QtGui.QApplication(sys.argv)
#    app.setOrganizationName("Mass-Transit")
#    app.setOrganizationDomain("Mass-Transit")
    app.setApplicationName("Mass-Transit")
    main_window = QtGui.QMainWindow()
    ui = Ui_MainWindow(app)
    ui.setupUi(main_window)
    main_window.setWindowTitle("Mass-Transit - Favorites")
    main_window.setAttribute(QtCore.Qt.WA_Maemo5StackedWindow, True)
    
#QtCore.Qt.WA_Maemo5AutoOrientation        QtCore.Qt.WA_Maemo5NonComposited          QtCore.Qt.WA_Maemo5ShowProgressIndicator  
#QtCore.Qt.WA_Maemo5LandscapeOrientation   QtCore.Qt.WA_Maemo5PortraitOrientation    QtCore.Qt.WA_Maemo5StackedWindow   
    main_window.show()
    sys.exit(app.exec_()) 
