#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
#    lib/config.py
#
#    This file is part of Mass-Transit
#
#    Copyright (C) 2010 Thibault Cohen
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#


from PyQt4 import QtCore
import os


from datetime import datetime


class Config(QtCore.QSettings):
    def __init__(self, config_filename='/home/' + os.getenv( 'USER' ) + '/.mass-transit.cfg', parent=None):
        QtCore.QSettings.__init__(self, config_filename, 0, parent)
        self.parent = parent

    def check_config(self):
        """
            Check config and write default items if need
        """
        if not self.contains("server"):
            self.parent.change_server_address()
        elif self.value("server").toString() == "":
            self.parent.change_server_address()
        self.parent.server = self.value("server").toString()
        if not self.contains("company_id"):
            self.parent.change_company()
        self.parent.company_id = self.value("company_id").toString()
        if not self.contains("company_name"):
            self.parent.company_name = "Company name not found"
        else:
            self.parent.company_name = self.value("company_name").toString()

    def get_favorite_stop_code_detail(self, stop_code):
        stop_code_detail = {}
        stop_code_detail['bus_stop_code'] = unicode(stop_code)
        stop_code_detail['schedules'] = []

        self.beginGroup("stop_codes/" + stop_code)
        stop_code_detail['bus_stop_name'] = self.value("name").toString()
        temp_line_list = [int(i) for i in self.childGroups()]
        line_list = QtCore.QStringList([QtCore.QString(str(i)) for i in temp_line_list])

        for line in line_list:
            self.beginGroup(line)
            direction = unicode(self.value("direction").toString())
            temp_times = list(self.value("times").toList())
            times = []
            for schedule in temp_times:
                if not schedule.isNull():
                    times.append(schedule.toDateTime())
                else:
                    times.append(None)
            # Try to read line in int in order to sort later
            try:
                line = int(line)
            except:
                line = unicode(line)
            stop_code_detail['schedules'].append({
                                'bus_direction' : unicode(direction),
                                'bus_line' : line,
                                'next_passing_times' : times,
                                })
            self.endGroup()
        self.endGroup()

        # Sort by line
        if len(stop_code_detail['schedules']) > 0:
            stop_code_detail['schedules'].sort()
            # Retranslante line to unicode is necessary
            if isinstance(stop_code_detail['schedules'][0]['bus_line'], int):
                i = 0
                for d in stop_code_detail['schedules']:
                    d['bus_line'] = unicode(d['bus_line'])
                    stop_code_detail['schedules'][i] = d
                    i = i + 1 
            
        return stop_code_detail

    def get_favorites_stop_code(self):
        """ RETURN : code - name - line_number/direction | ...
            exemple : 51205 - Saint-Laurent / de Castelnau - 55/North | ......
        """
        favorites_stop_code = {}
        self.beginGroup("stop_codes")
        for group in self.childGroups():
            self.beginGroup(group)
            stop_name = self.value("name").toString()

            # Get lines
            lines = [int(line) for line in self.childGroups()]
            # Sort by Lines
            lines.sort()
            # Create line_number/direction | line_number/direction ....
            line_direction = " | ".join([ unicode(line) + "/"+ unicode(self.value(unicode(line) + "/direction").toString())
                for line in lines])
            # Create code - name - line_number/direction | ...
            name = stop_name + " - " + line_direction

            favorites_stop_code[group] = (name)
            self.endGroup()
        self.endGroup()
        return favorites_stop_code

    def delete_favorite_stop_code(self, stop_code):
        self.beginGroup("stop_codes")
        self.remove(stop_code)
        self.endGroup()
        self.save_config()

    def add_favorite_stop_code(self, stop_code_shedules=None):
        """ {   'bus_stop_name': u'Cr\xe9mazie / Berri',
                'bus_stop_code': '54206', 'schedules':
                    [{'bus_direction': 'East',
                    'bus_line': '192',
                    'next_passing_times':
                       ['2010-07-04T19:25:00',
                        '2010-07-04T19:55:00',
                        '2010-07-04T20:25:00',
                        '2010-07-04T20:55:00',
                        '2010-07-04T21:25:00',
                         '2010-07-04T21:55:00']
                    }]
            }
        """
        stop_name = stop_code_shedules['bus_stop_name']
        stop_code = stop_code_shedules['bus_stop_code']
        self.beginGroup("stop_codes")
        self.beginGroup(stop_code)
        self.setValue("name", QtCore.QVariant(stop_name))
        for line_schedule in stop_code_shedules['schedules']:
            line = line_schedule['bus_line']
            direction = line_schedule['bus_direction']
            times = line_schedule['next_passing_times']
            self.beginGroup(line)
            self.setValue("direction", QtCore.QVariant(direction))
            self.setValue("times", QtCore.QVariant(times))
            self.endGroup()
        self.endGroup()
        self.endGroup()
        self.save_config()

    def save_config(self):
        self.sync()

