#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
#    masstransit.py 
#
#    This file is part of Mass-Transit
#
#    Copyright (C) 2010 Thibault Cohen
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from PyQt4 import QtCore, QtGui
from module import stm 
import sys
import os
from datetime import datetime
from lib import lib_masstransit as masstransit
from ui import stopcodelist,lineslist,stopcode,favoritelist
try:
    import osso
except:
    pass


class MenuBar(QtGui.QMenuBar):
    def __init__(self, parent=None, ):
        QtGui.QMenuBar.__init__(self, parent)
        self.setObjectName("Menu")
        ### Stop lists
        self.stop_list = self.addAction(self.tr("Stop &List"))
        ### Bus Lines
        self.bus_lines = self.addAction(self.tr("&Bus Lines"))
        ### Stop code
        self.stop_code = self.addAction(self.tr("&Stop Code"))
        ### Subway
        self.subway = self.addAction(self.tr("Subway"))
        ## Quit
        self.quit = self.addAction(self.tr("&Quit"))


class Config(QtCore.QSettings):
    def __init__(self, config_filename='/home/' + os.getenv( 'USER' ) + '/.stm.cfg', parent=None):
        QtCore.QSettings.__init__(self, config_filename, 0, parent)
        self.parent = parent
        self.check_config()

    def check_config(self):
        """
            Check config and write default items if need
        """
        return False
        if not self.contains("stop_codes"):
            stop_code = "52904"
            stop_name = "My stop"
            d = datetime(2010,06,05,06,30)
            time_list = [d]
            self.beginGroup("stop_codes/"+stop_code)
            self.setValue("name", QtCore.QVariant(stop_name))
            self.setValue("14/direction", QtCore.QVariant("North"))
            self.setValue("14/times", QtCore.QVariant(time_list))
            self.endGroup()
            self.save_config()

    def get_favorite_stop_code_detail(self, stop_code):
        schedules = []
        self.beginGroup("stop_codes/" + stop_code)
        name = self.value("name").toString()
        temp_line_list = [int(i) for i in self.childGroups()]
        line_list = QtCore.QStringList([QtCore.QString(str(i)) for i in temp_line_list])

        for line in line_list:
            self.beginGroup(line)
            direction = unicode(self.value("direction").toString())
            temp_times = list(self.value("times").toList())
            times = []
            for schedule in temp_times:
                if not schedule.isNull():
                    times.append(schedule.toDateTime())
                else:
                    times.append(None)
            schedules.append(([line, direction], times))
            self.endGroup()
        self.endGroup()

        return name, schedules 
            

    def get_favorites_stop_code(self):
        """ RETURN : code - name - line_number/direction | ...
            exemple : 51205 - Saint-Laurent / de Castelnau - 55/North | ......
        """
        favorites_stop_code = {}
        self.beginGroup("stop_codes")
        for group in self.childGroups():
            self.beginGroup(group)
            stop_name = self.value("name").toString()

            # Get lines
            lines = [int(line) for line in self.childGroups()]
            # Sort by Lines
            lines.sort()
            # Create line_number/direction | line_number/direction ....
            line_direction = " | ".join([ unicode(line) + "/"+ unicode(self.value(unicode(line) + "/direction").toString())
                for line in lines])
            # Create code - name - line_number/direction | ...
            name = stop_name + " - " + line_direction

            favorites_stop_code[group] = (name)
            self.endGroup()
        self.endGroup()
        return favorites_stop_code

    def delete_favorite_stop_code(self, stop_code):
        self.beginGroup("stop_codes")
        self.remove(stop_code)
        self.endGroup()
        self.save_config()

    def add_favorite_stop_code(self, stop_code, stop_name, schedules=None):
        """ schedules = (
                            [
                            line_number,
                            direction
                            ],
                            [
                                datetime1,
                                datetime2,
                                datetime3,
                                datetime4,
                                None,
                                None,
                            ]
                        )
        """
        self.beginGroup("stop_codes")
        self.beginGroup(stop_code)
        self.setValue("name", QtCore.QVariant(stop_name))
        for line_schedule in schedules:
            line = line_schedule[0][0]
            direction = line_schedule[0][1]
            times = line_schedule[1]
            self.beginGroup(line)
            self.setValue("direction", QtCore.QVariant(direction))
            self.setValue("times", QtCore.QVariant(times))
            self.endGroup()
        self.endGroup()
        self.endGroup()
        self.save_config()

    def save_config(self):
        self.sync()


class Ui_MainWindow(QtCore.QObject):
    def __init__(self, app):
        QtCore.QObject.__init__(self)
        self.app = app
        self.config = Config()
        self.mode = "favorite"

    def change_mode(self, mode="favorites", parent=None, **kwargs):
        root = self
        if self.mode == mode:
            masstransit.banner_notification("Please wait... Processing ")
            return False
        if not parent or parent == self:
            parent = self.main_window
        if mode == "bus_lines":
            ui_lines_list = lineslist.UI_LinesList(kwargs["lines"], parent, root)
            ui_lines_list.show()
        elif mode == "stop_code":
            self.mode = "stop_code"
            ui_stop_code = stopcode.Ui_StopCode(kwargs["stop_code"], False, parent, root)
            ui_stop_code.show()
        elif mode == "stop_list":
            ui_stop_list = stopcodelist.UI_StopCodeList(kwargs["line"], kwargs["direction"], parent, root)
            ui_stop_list.show()
        elif mode == "favorites":
            ui_fav_list = favoritelist.UI_FavoriteList(parent, root)
            self.main_window.setCentralWidget(ui_fav_list)
            self.main_window.setWindowTitle("Mass-Transit - Favorites")
        self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
        self.mode = mode

    def setupUi(self, main_window):
        self.main_window = main_window
        self.main_window.setObjectName("MainWindow")
        self.main_window.resize(800, 400)

        ### Central Widget
        self.change_mode("favorites", self)
        ### Menu
        self.menubar = MenuBar(self.main_window)
        self.main_window.setMenuBar(self.menubar)

        QtCore.QMetaObject.connectSlotsByName(self.main_window)

        # Signals
        ## Stop List
        QtCore.QObject.connect(self.menubar.stop_list, QtCore.SIGNAL("triggered()"), self.stop_list)
        ## Favorite
        QtCore.QObject.connect(self.menubar.bus_lines, QtCore.SIGNAL("triggered()"), self.bus_lines)
        ## Stop code
        QtCore.QObject.connect(self.menubar.stop_code, QtCore.SIGNAL("triggered()"), self.stop_code)
        ## Subway
        QtCore.QObject.connect(self.menubar.subway, QtCore.SIGNAL("triggered()"), self.subway)
        ## Quit
        QtCore.QObject.connect(self.menubar.quit, QtCore.SIGNAL("triggered()"), self.app.quit)

    def subway(self):
        masstransit.banner_notification("Subway not yet implemented")

    def stop_code(self, parent=None):
        self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, True)
        code,bool = QtGui.QInputDialog().getText(
                                self.main_window,
                                "Stop Code Title",
                                "Stop Code",
                                QtGui.QLineEdit.Normal,
                                "",
                                )
        if not bool:
            self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
            return False

        else:
            try:
                int(code)
            except:
                self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

        try:
            self.change_mode(mode="stop_code",stop_code=str(code),parent=parent)
        except:
            masstransit.banner_notification("That bus stop code is invalid. Please try again.")
            self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)

            return False


    def bus_lines(self):
        self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, True)
        line_list = [   "Routes 10 to 99",
                        "Routes 100 to 169",
                        "Routes 170 to 268, 410 to 535 and 747",
                        "Night routes (350 to 382)"]
        lines,bool = QtGui.QInputDialog().getItem(
                                self.main_window,
                                "Lines Title",
                                "Lines",
                                line_list,
                                0,
                                False,
                                )
        if not bool:
            self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
            return False
        else:
            try:
                str(lines)
            except:
                self.main_window.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

        lines_index = line_list.index(lines)
        self.change_mode(mode="bus_lines", lines=lines_index, parent=self)

    def stop_list(self, line=None, parent=None):
        if parent == None:
            parent = self.main_window
        parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, True)
        if line == None:
            line,bool = QtGui.QInputDialog().getText(
                                    self.main_window,
                                    "Line number Title",
                                    "Line number",
                                    QtGui.QLineEdit.Normal,
                                    "",
                                    )
            if not bool or line == "":
                parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

            else:
                try:
                    int(line)
                except:
                    return False

        directions = ["South", "West", "East", "North"]
        direction,bool = QtGui.QInputDialog().getItem(
                                self.main_window,
                                "Direction Title",
                                "Direction",
                                directions,
                                0,
                                False,
                                )
        if not bool:
            parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
            return False
        else:
            try:
                str(direction)
            except:
                parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)
                return False

        try:
            self.change_mode(mode="stop_list", line=line, direction=direction, parent=parent)
        except:
            masstransit.banner_notification("That bus route number or direction is invalid. Please try again.")
            parent.setAttribute(QtCore.Qt.WA_Maemo5ShowProgressIndicator, False)

            return False


def main():
    app = QtGui.QApplication(sys.argv)
#    app.setOrganizationName("Mass-Transit")
#    app.setOrganizationDomain("Mass-Transit")
    app.setApplicationName("Mass-Transit")
    main_window = QtGui.QMainWindow()
    ui = Ui_MainWindow(app)
    ui.setupUi(main_window)
    main_window.setWindowTitle("Mass-Transit - Favorites")
    main_window.setAttribute(QtCore.Qt.WA_Maemo5StackedWindow, True)
    
#QtCore.Qt.WA_Maemo5AutoOrientation        QtCore.Qt.WA_Maemo5NonComposited          QtCore.Qt.WA_Maemo5ShowProgressIndicator  
#QtCore.Qt.WA_Maemo5LandscapeOrientation   QtCore.Qt.WA_Maemo5PortraitOrientation    QtCore.Qt.WA_Maemo5StackedWindow   
    main_window.show()
    sys.exit(app.exec_()) 
