#ifndef _PATH_H
#define _PATH_H

#include <glib.h>
#include <gtk/gtk.h>

#define ARRAY_CHUNK_SIZE (1024)

#define TRACKS_MASK 0x00000001
#define ROUTES_MASK 0x00000002

#define MACRO_PATH_INIT(path) { \
	(path).head = (path).tail = g_new0(Point, ARRAY_CHUNK_SIZE); \
	*((path).tail) = _point_null; \
	(path).cap = (path).head + ARRAY_CHUNK_SIZE; \
	(path).whead = g_new0(WayPoint, ARRAY_CHUNK_SIZE); \
	(path).wtail = (path).whead - 1; \
	(path).wcap = (path).whead + ARRAY_CHUNK_SIZE; \
}

#define MACRO_PATH_FREE(path) if((path).head) { \
	WayPoint *curr; \
	g_free((path).head); \
	(path).head = (path).tail = (path).cap = NULL; \
	for(curr = (path).whead - 1; curr++ != (path).wtail; ) \
		g_free(curr->desc); \
	g_free((path).whead); \
	(path).whead = (path).wtail = (path).wcap = NULL; \
}

#define MACRO_PATH_INCREMENT_TAIL(route) { \
	if(++(route).tail == (route).cap) \
		path_resize(&(route), (route).cap - (route).head + ARRAY_CHUNK_SIZE);\
}

#define MACRO_PATH_INCREMENT_WTAIL(route) { \
	if(++(route).wtail == (route).wcap) \
		path_wresize(&(route), (route).wcap - (route).whead + ARRAY_CHUNK_SIZE); \
}

/* Route list */
typedef enum {
	ROUTE_LATLON,
	ROUTE_DISTANCE,
	ROUTE_WAYPOINT,
	ROUTE_LAT,
	ROUTE_LON,
	ROUTE_NUM_COLUMNS
} RouteList;

typedef enum {
	WPT_POINT=0,
	WPT_CONTINUE,
	WPT_TURN_LEFT,
	WPT_TURN_S_LEFT,
	WPT_TURN_H_LEFT,
	WPT_TURN_RIGHT,
	WPT_TURN_S_RIGHT,
	WPT_TURN_H_RIGHT,
	WPT_U_TURN
} WayPointType;

/** A lat/lon/alt position */
typedef struct _Position Position;
struct _Position {
	gchar *desc;
	gboolean valid;
	gdouble lat;
	gdouble lon;
	gfloat altitude;
	gfloat angle;		/* Course from current position to this one */
	time_t time;
	guint type;			/* Type of position */
};

/** A general definition of a point in the Mapper unit system. */
typedef struct _Point Point;
struct _Point {
	guint unitx;
	guint unity;
	time_t time;
	gfloat altitude;
};

/** A WayPoint, which is a Point with a description. */
typedef struct _WayPoint WayPoint;
struct _WayPoint {
	Point *point;
	gchar *desc;
	WayPointType t;
	guint flags;
	guint sat;
	gfloat hdop;
	gfloat vdop;
	gfloat pdop;
};

/** A Path is a set of PathPoints and WayPoints. */
typedef struct _Path Path;
struct _Path {
	/* Path points */
	Point *head;		/* points to first element in array; NULL if empty. */
	Point *tail;		/* points to last element in array. */
	Point *cap;			/* points after last slot in array. */

	/* Path waypoints */
	WayPoint *whead;	/* points to first element in array; NULL if empty. */
	WayPoint *wtail;	/* points to last element in array. */
	WayPoint *wcap;		/* points after last slot in array. */

	/* Path statistics */
	guint32	points;
	gdouble length;
	gdouble avgspeed;
	gfloat maxspeed;

	/* xxx, not the right place but who cares for the old branch.. */
	GtkListStore *store;

	/* GPX metadata fields */
	gchar *name;
	gchar *desc;
	gchar *author;
	gchar *keywords;
	time_t time;
	gchar *copyright;
	gchar *src;
};

/* Null point */
Point _point_null;

/* Special positions */
Position _home;
Position _dest;

Path *path_new(void);
void path_free(Path *p);
void path_clear(Path *p);

gboolean path_insert_break(Path *path);
gboolean path_insert_mark_text(Path *path, gchar *text);

void path_resize(Path *path, guint size);
void path_wresize(Path *path, guint wsize);

gdouble path_calculate_distance_from(Path *path, Point *point, gdouble lat, gdouble lon);

gboolean path_load(Path *path, const gchar *config_dir, const gchar *file);
gboolean path_save(Path *path, const gchar *config_dir, const gchar *file);

gboolean path_update_store(Path *path);

void position_set(Position *pos, gboolean valid, gdouble lat, gdouble lon);
void position_swap(Position *p1, Position *p2);

#endif
