/*
 * This file is part of mapper
 *
 * Copyright (C) 2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>

#include "gps.h"
#include "latlon.h"
#include "gps-liblocation.h"

static LocationGPSDControl *control=NULL;
static LocationGPSDevice *device=NULL;

static void
liblocation_on_error_cb(LocationGPSDControl *control, LocationGPSDControlError error, gpointer user_data)
{ 
Gps *gps=(Gps *)user_data;
g_return_if_fail(control);
g_return_if_fail(gps);

switch (error) {
	case LOCATION_ERROR_USER_REJECTED_DIALOG:
	g_debug("User didn't enable requested methods");
	break;
	case LOCATION_ERROR_USER_REJECTED_SETTINGS:
	g_debug("User changed settings, which disabled location");
	break;
	case LOCATION_ERROR_BT_GPS_NOT_AVAILABLE:
	g_debug("Problems with BT GPS");
	break;
	case LOCATION_ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
	g_debug("Requested method is not allowed in offline mode");
	break;
	case LOCATION_ERROR_SYSTEM:
	g_debug("System error");
	break;
}
gps_disconnect(gps);
}

static void
liblocation_set_gps_satellite_data(Gps *gps, gint i, LocationGPSDeviceSatellite *sat)
{
g_return_if_fail(gps);
g_return_if_fail(sat);
g_return_if_fail(i<=GPS_SAT_MAX);
gps->data.sat[i].prn=sat->prn;
gps->data.sat[i].elevation=sat->elevation/100;
gps->data.sat[i].azimuth=sat->azimuth;
gps->data.sat[i].snr=sat->signal_strength;
gps->data.sat[i].fix=sat->in_use;
}

static void
liblocation_on_changed_cb(LocationGPSDevice *device, gpointer user_data)
{
Gps *gps=(Gps *)user_data;
gint s;

g_return_if_fail(device);
g_return_if_fail(gps);

if (!device->fix) {
	gps_conn_set_state(gps, RCVR_DOWN);
	return ;
}

switch (device->fix->mode) {
case LOCATION_GPS_DEVICE_MODE_NOT_SEEN:
	gps->data.newly_fixed=FALSE;
	gps_conn_set_state(gps, RCVR_UP);
	gps->data.fix=FIX_NOFIX;
break;
case LOCATION_GPS_DEVICE_MODE_NO_FIX:
	gps->data.newly_fixed=FALSE;
	gps_conn_set_state(gps, RCVR_UP);
	gps->data.fix=FIX_NOFIX;
break;
case LOCATION_GPS_DEVICE_MODE_2D:
	gps->data.fix=FIX_2D;
	if (gps->io.conn!=RCVR_FIXED) {
		gps->data.newly_fixed=TRUE;
		gps_conn_set_state(gps, RCVR_FIXED);
	}
break;
case LOCATION_GPS_DEVICE_MODE_3D:
	gps->data.fix=FIX_3D;
	if (gps->io.conn!=RCVR_FIXED) {
		gps->data.newly_fixed=TRUE;
		gps_conn_set_state(gps, RCVR_FIXED);
	}
break;
}

if (device->fix->fields & LOCATION_GPS_DEVICE_TIME_SET) {
	gps->data.time=(gint)round(device->fix->time);
} else {
	gps->data.time=time(NULL);
}

if (device->fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) {
	gps->data.lat=device->fix->latitude;
	gps->data.lon=device->fix->longitude;
} else {
	gps->data.lat=NAN;
	gps->data.lon=NAN;
}
gps->data.altitude=(device->fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) ? device->fix->altitude : NAN;

if (device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET) {
	/* liblocation does not give the raw NMEA speed, so adjust it back */
	gps->data.speed=device->fix->speed/NM_TO_KM;
	gps->data.maxspeed=MAX(device->fix->speed, gps->data.maxspeed);
}

/* Adjust to meters */
gps->data.hdop=device->fix->eph/100;
gps->data.vdop=device->fix->epv/100;
/* Fake this as liblocation does not give us raw data, sigh... */
gps->data.pdop=(gps->data.hdop+gps->data.vdop)/2;

gps->data.lheading=gps->data.heading;
gps->data.heading=(device->fix->fields & LOCATION_GPS_DEVICE_TRACK_SET) ? device->fix->track : NAN;

gps->data.satinview=CLAMP(device->satellites_in_view, 0, GPS_SAT_MAX);
gps->data.satinuse=CLAMP(device->satellites_in_use, 0, GPS_SAT_MAX);

for (s=0;s<=GPS_SAT_MAX;s++)
	gps->data.sat[s].fix=FALSE;

for (s=0;device->satellites && s<device->satellites->len;s++)
	liblocation_set_gps_satellite_data(gps, s, g_ptr_array_index(device->satellites, s));

gps_data_integerize(&gps->data);
gps_location_update(gps);

if (gps->update_satellite)
	g_idle_add((GSourceFunc)gps->update_satellite, gps);
if (gps->update_info)
	g_idle_add((GSourceFunc)gps->update_info, gps);
}

gboolean
gps_liblocation_connect(Gps *gps)
{
if (control)
	return TRUE;
control=location_gpsd_control_get_default();
if (device)
	return TRUE;
device=g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);
g_object_set(G_OBJECT(control), "preferred-method", LOCATION_METHOD_GNSS | LOCATION_METHOD_AGNSS, NULL);
g_signal_connect(device, "changed", G_CALLBACK(liblocation_on_changed_cb), gps);
g_signal_connect(device, "error-verbose", G_CALLBACK(liblocation_on_error_cb), gps);
location_gpsd_control_start(control);
return TRUE;
}

gboolean
gps_liblocation_disconnect(Gps *gps)
{
if (control) {
	location_gpsd_control_stop(control);
	g_object_unref(control);
}
if (device)
	g_object_unref(device);
device=NULL;
control=NULL;
return TRUE;
}

