/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <libintl.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <libgnomevfs/gnome-vfs.h>
#include <curl/multi.h>

#include "utils.h"
#include "gps.h"
#include "route.h"
#include "ui-common.h"
#include "settings.h"
#include "mapper-types.h"
#include "map.h"
#include "file.h"
#include "latlon.h"
#include "map.h"
#include "map-download.h"
#include "iap.h"
#include "gpx.h"
#include "dialogs.h"
#include "announcements.h"

#define DISTANCE_SQUARED(a, b) \
   ((guint64)((((gint64)(b).unitx)-(a).unitx)*(((gint64)(b).unitx)-(a).unitx))\
  + (guint64)((((gint64)(b).unity)-(a).unity)*(((gint64)(b).unity)-(a).unity)))

static gboolean show_directions=TRUE;

typedef struct _OriginToggleInfo OriginToggleInfo;
struct _OriginToggleInfo {
	GtkWidget *rad_use_gps;
	GtkWidget *rad_use_route;
	GtkWidget *rad_use_text;
	GtkWidget *chk_auto;
	GtkWidget *txt_from;
	GtkWidget *txt_to;
};

static guint64 _next_way_dist_squared=-1;

void route_show_distance_to_last(void);
void route_set_destination_from_last(void);

static gboolean auto_route_dl_idle(void);

void
route_clear(void)
{
GtkWidget *confirm;

confirm = hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), _("Really clear the route?"));

if (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm))) {
	route_cancel_autoroute(FALSE);
	path_clear(_route);
	route_find_nearest_point();
	map_force_redraw();
}
gtk_widget_destroy(confirm);
}

/**
 * Check if we should announce upcoming waypoints. 
 */
void
route_check_waypoint_announce(GpsData *gps)
{
guint an_wpt_now, an_wpt_msg;

if (!gps || !_announce_waypoints || !_next_way)
	return;

if (_next_way->flags>=2)
	return;

an_wpt_msg=(20+(guint)gps->speed)*_announce_notice_ratio*3;
an_wpt_now=(2+(guint)gps->speed)*_announce_notice_ratio;

if (_next_way_dist_squared > (an_wpt_msg * an_wpt_msg))
	return;

if (_next_way_dist_squared < (an_wpt_now*an_wpt_now) && _next_way->flags < 2) {
	_next_way->flags=2;
	announce_waypoint(_next_way);
} else if (_next_way_dist_squared < (an_wpt_msg*an_wpt_msg) && _next_way->flags==0) {
	_next_way->flags=1;
	announce_waypoint(_next_way);
}
mapper_info_banner_set_waypoint_direction(_next_way);
}

/**
 * Check if we should re-calculate route
 */
void
route_autoroute_check(void)
{
if (_autoroute_data.enabled && !_autoroute_data.in_progress && _near_point_dist_squared > 400) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Recalculating directions..."));
	_autoroute_data.in_progress = TRUE;
	show_directions = FALSE;
	g_idle_add((GSourceFunc)auto_route_dl_idle, NULL);
}
}

gboolean 
route_open_file(void)
{
gchar *buffer;
guint size;

if (file_open_get_contents(&_route_dir_uri, &buffer, &size)) {
	/* If auto is enabled, append the route, otherwise replace it. */
	if (gpx_parse(_route, buffer, size, _autoroute_data.enabled ? GPX_PATH_APPEND : GPX_PATH_NEW)) {
		route_cancel_autoroute(FALSE);

		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Route Opened"));
		/* Find the nearest route point, if we're connected. */
		route_find_nearest_point();
		map_force_redraw();
		route_set_destination_from_last();
		return TRUE;
	} else {
		popup_error(mapp.mainwindow, _("Error parsing GPX file."));
		g_free(buffer);
		return FALSE;
	}
}
return FALSE;
}

/**
 * Ask user to save route
 */
gboolean
route_save(void)
{
GnomeVFSHandle *handle;

if (_route->head==_route->tail) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("No route exist."));
	return FALSE;
}

if (file_save(&_route_dir_uri, &_route_dir_uri, &handle)) {
	if (gpx_write(_route, handle)) {
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Route Saved"));
	} else {
		popup_error(mapp.mainwindow, _("Error writing GPX file."));
	}
	gnome_vfs_close(handle);
	return TRUE;
}
return FALSE;
}

static void
route_set_entry_latlon(GtkWidget *entry, gdouble lat, gdouble lon)
{
gchar buffer[72];
gchar strlat[32];
gchar strlon[32];

g_ascii_formatd(strlat, 32, "%.06f", lat);
g_ascii_formatd(strlon, 32, "%.06f", lon);
g_snprintf(buffer, sizeof(buffer), "%s,%s", strlat, strlon);
gtk_entry_set_text(GTK_ENTRY(entry), buffer);
}

/**
 *
 */
static gboolean 
origin_type_selected_cb(GtkWidget * toggle, OriginToggleInfo * oti)
{
if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle))) {
	if (toggle == oti->rad_use_gps) {
		route_set_entry_latlon(oti->txt_from, _gps->data.lat, _gps->data.lon);
	} else if (toggle == oti->rad_use_route) {
		Point *p;
		gdouble lat, lon;

		/* Use last non-zero route point. */
		for (p = _route->tail; !p->unity; p--) {
		}

		unit2latlon(p->unitx, p->unity, lat, lon);
		route_set_entry_latlon(oti->txt_from, lat, lon);
	}
	gtk_widget_set_sensitive(oti->txt_from, toggle == oti->rad_use_text);
	gtk_widget_set_sensitive(oti->chk_auto, toggle == oti->rad_use_gps);
}
return TRUE;
}

/**
 * Cancel the current auto-route.
 */
void 
route_cancel_autoroute(gboolean temporary)
{
if (!_autoroute_data.enabled)
	return;

if (!temporary) {
	_autoroute_data.enabled = FALSE;
	g_free(_autoroute_data.dest);
	_autoroute_data.dest = NULL;

	g_free(_autoroute_data.src_str);
	_autoroute_data.src_str = NULL;
}

if (_autoroute_data.curl_easy) {
	if (_curl_multi)
		curl_multi_remove_handle(_curl_multi, _autoroute_data.curl_easy);
	curl_easy_cleanup(_autoroute_data.curl_easy);
	_autoroute_data.curl_easy = NULL;
}

g_free(_autoroute_data.rdl_data.bytes);
_autoroute_data.rdl_data.bytes = NULL;
_autoroute_data.rdl_data.bytes_read = 0;
_autoroute_data.in_progress = FALSE;
}

/**
 * Read the data provided by the given handle as GPX data, updating the
 * auto-route with that data.
 */
static size_t
route_dl_cb_read(void *ptr, size_t size, size_t nmemb, RouteDownloadData * rdl_data)
{
size_t old_size = rdl_data->bytes_read;

rdl_data->bytes_read += size * nmemb;
rdl_data->bytes = g_renew(gchar, rdl_data->bytes, rdl_data->bytes_read);
g_memmove(rdl_data->bytes + old_size, ptr, size * nmemb);

return (size * nmemb);
}

static gboolean
auto_route_dl_idle(void)
{
gchar latstr[32], lonstr[32], *latlonstr;

g_ascii_dtostr(latstr, 32, _gps->data.lat);
g_ascii_dtostr(lonstr, 32, _gps->data.lon);
latlonstr = g_strdup_printf("%s,%s", latstr, lonstr);
_autoroute_data.src_str = g_strdup_printf(_route_dl_url, latlonstr, _autoroute_data.dest);
g_free(latlonstr);

MACRO_CURL_EASY_INIT(_autoroute_data.curl_easy);
curl_easy_setopt(_autoroute_data.curl_easy, CURLOPT_URL, _autoroute_data.src_str);
curl_easy_setopt(_autoroute_data.curl_easy, CURLOPT_WRITEFUNCTION, route_dl_cb_read);
curl_easy_setopt(_autoroute_data.curl_easy, CURLOPT_WRITEDATA, &_autoroute_data.rdl_data);
curl_multi_add_handle(_curl_multi, _autoroute_data.curl_easy);

if (iap_is_connected() && !_curl_sid)
	_curl_sid=g_idle_add((GSourceFunc)map_download_timeout, NULL);
_autoroute_data.in_progress = TRUE;
return FALSE;
}

/**
 * Display a dialog box to the user asking them to download a route.  The
 * "From" and "To" textfields may be initialized using the first two
 * parameters.  The third parameter, if set to TRUE, will cause the "Use GPS
 * Location" checkbox to be enabled, which automatically sets the "From" to the
 * current GPS position (this overrides any value that may have been passed as
 * the "To" initializer).
 * None of the passed strings are freed - that is left to the caller, and it is
 * safe to free either string as soon as this function returns.
 */
gboolean 
route_download(void)
{
GtkWidget *dialog;
GtkWidget *table;
GtkWidget *label;
GtkWidget *hbox;
OriginToggleInfo oti;
GtkEntryCompletion *from_comp;
GtkEntryCompletion *to_comp;
gboolean ret=FALSE;

dialog = gtk_dialog_new_with_buttons(_("Download Route"),
			     GTK_WINDOW(mapp.mainwindow),
			     GTK_DIALOG_MODAL, GTK_STOCK_OK,
			     GTK_RESPONSE_ACCEPT,
			     GTK_STOCK_CANCEL,
			     GTK_RESPONSE_REJECT, NULL);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
	   table = gtk_table_new(2, 5, FALSE), TRUE, TRUE, 0);

from_comp = gtk_entry_completion_new();
gtk_entry_completion_set_model(from_comp, GTK_TREE_MODEL(_loc_model));
gtk_entry_completion_set_text_column(from_comp, 0);
to_comp = gtk_entry_completion_new();
gtk_entry_completion_set_model(to_comp, GTK_TREE_MODEL(_loc_model));
gtk_entry_completion_set_text_column(to_comp, 0);

/* Auto. */
gtk_table_attach(GTK_TABLE(table),
		 hbox = gtk_hbox_new(FALSE, 6),
		 0, 2, 1, 2, GTK_FILL, 0, 2, 4);
gtk_box_pack_start(GTK_BOX(hbox),
		   oti.rad_use_gps = gtk_radio_button_new_with_label(NULL, _("Use GPS Location")),
		   TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), oti.chk_auto =
		   gtk_check_button_new_with_label(_("Auto-Update")),
		   TRUE, TRUE, 0);
gtk_widget_set_sensitive(oti.chk_auto, FALSE);

/* Use End of Route. */
gtk_table_attach(GTK_TABLE(table),
		 hbox = gtk_hbox_new(FALSE, 6),
		 0, 2, 2, 3, GTK_FILL, 0, 2, 4);
gtk_box_pack_start(GTK_BOX(hbox),
		   oti.rad_use_route =
		   gtk_radio_button_new_with_label_from_widget
		   (GTK_RADIO_BUTTON(oti.rad_use_gps),
		    _("Use End of Route")), TRUE, TRUE, 0);
gtk_widget_set_sensitive(oti.rad_use_route, _route->head != _route->tail);

/* Origin. */
gtk_table_attach(GTK_TABLE(table),
		 oti.rad_use_text =
		 gtk_radio_button_new_with_label_from_widget
		 (GTK_RADIO_BUTTON(oti.rad_use_gps), _("Origin")), 0, 1,
		 3, 4, GTK_FILL, 0, 2, 4);
gtk_table_attach(GTK_TABLE(table), oti.txt_from =
		 gtk_entry_new(), 1, 2, 3, 4, GTK_EXPAND | GTK_FILL, 0,
		 2, 4);
gtk_entry_set_completion(GTK_ENTRY(oti.txt_from), from_comp);
gtk_entry_set_width_chars(GTK_ENTRY(oti.txt_from), 25);

/* Destination. */
gtk_table_attach(GTK_TABLE(table),
		 label = gtk_label_new(_("Destination")),
		 0, 1, 4, 5, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table),
	 oti.txt_to = gtk_entry_new(), 1, 2, 4, 5, GTK_EXPAND | GTK_FILL, 0, 2, 4);
gtk_entry_set_completion(GTK_ENTRY(oti.txt_to), to_comp);
gtk_entry_set_width_chars(GTK_ENTRY(oti.txt_to), 25);

g_signal_connect(G_OBJECT(oti.rad_use_gps), "toggled", G_CALLBACK(origin_type_selected_cb), &oti);
g_signal_connect(G_OBJECT(oti.rad_use_route), "toggled", G_CALLBACK(origin_type_selected_cb), &oti);
g_signal_connect(G_OBJECT(oti.rad_use_text), "toggled", G_CALLBACK(origin_type_selected_cb), &oti);

/* Initialize fields. */
if (_dest.valid) {
	route_set_entry_latlon(GTK_ENTRY(oti.txt_to), _dest.lat, _dest.lon);
} else if (_home.valid) {
	route_set_entry_latlon(GTK_ENTRY(oti.txt_to), _home.lat, _home.lon);
} else {
	gtk_entry_set_text(GTK_ENTRY(oti.txt_to), "");
}

gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oti.rad_use_text), TRUE);

/* Use "End of Route" by default if they have a route. */
if (_route->head != _route->tail) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oti.rad_use_route), TRUE);
	gtk_widget_grab_focus(oti.rad_use_route);
} else if (_enable_gps) {
	/* Else use "GPS Location" if they have GPS enabled. */
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oti.rad_use_gps), TRUE);
	gtk_widget_grab_focus(oti.rad_use_gps);
} else if (_home.valid) {
	route_set_entry_latlon(GTK_ENTRY(oti.txt_from), _home.lat, _home.lon);
} else {
	/* Else use text. */
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(oti.rad_use_text), TRUE);
	gtk_widget_grab_focus(oti.txt_from);
}

iap_connect();

gtk_widget_show_all(dialog);

while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	CURL *curl_easy;
	RouteDownloadData rdl_data = { 0, 0 };
	gchar buffer[BUFFER_SIZE];
	const gchar *from, *to;

	from=gtk_entry_get_text(GTK_ENTRY(oti.txt_from));
	if (!strlen(from)) {
		popup_error(dialog, _("Please specify a start lat,lon location."));
		continue;
	}

	to=gtk_entry_get_text(GTK_ENTRY(oti.txt_to));
	if (!strlen(to)) {
		popup_error(dialog, _("Please specify an end lat,lon location."));
		continue;
	}

	g_snprintf(buffer, sizeof(buffer), _route_dl_url, from, to);
	g_debug("RouteURL: %s", buffer);

	/* Attempt to download the route from the server. */
	MACRO_CURL_EASY_INIT(curl_easy);
	curl_easy_setopt(curl_easy, CURLOPT_URL, buffer);
	curl_easy_setopt(curl_easy, CURLOPT_WRITEFUNCTION, route_dl_cb_read);
	curl_easy_setopt(curl_easy, CURLOPT_WRITEDATA, &rdl_data);
	if (CURLE_OK != curl_easy_perform(curl_easy)) {
		popup_error(dialog, _("Failed to connect to GPX Directions server"));
		curl_easy_cleanup(curl_easy);
		g_free(rdl_data.bytes);
		/* Let them try again */
		continue;
	}
	curl_easy_cleanup(curl_easy);

	if (strncmp(rdl_data.bytes, "<?xml", strlen("<?xml"))) {
		/* Not an XML document - must be bad locations. */
		popup_error(dialog, _("Could not generate directions. Make sure your source and destination are valid."));
		g_free(rdl_data.bytes);
		/* Let them try again. */
	}
	/* Else, if GPS is enabled, replace the route, otherwise append it. */
	else if (gpx_parse(_route, rdl_data.bytes, rdl_data.bytes_read,
			(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(oti.rad_use_gps)) ? GPX_PATH_NEW : GPX_PATH_APPEND))) {
		GtkTreeIter iter;

		/* Find the nearest route point, if we're connected. */
		route_find_nearest_point();

		/* Cancel any autoroute that might be occurring. */
		route_cancel_autoroute(FALSE);

		map_force_redraw();

		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(oti.chk_auto))) {
			/* Kick off a timeout to start the first update. */
			_autoroute_data.dest = gnome_vfs_escape_string(to);
			_autoroute_data.enabled = TRUE;
		}

		/* Save Origin in Route Locations list if not from GPS. */
		if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(oti.rad_use_gps))
		    && !g_slist_find_custom(_loc_list, from, (GCompareFunc) strcmp)) {
			_loc_list = g_slist_prepend(_loc_list, g_strdup(from));
			gtk_list_store_insert_with_values(_loc_model, &iter, INT_MAX, 0, from, -1);
		}

		/* Save Destination in Route Locations list. */
		if (!g_slist_find_custom(_loc_list, to, (GCompareFunc) strcmp)) {
			_loc_list = g_slist_prepend(_loc_list, g_strdup(to));
			gtk_list_store_insert_with_values(_loc_model, &iter, INT_MAX, 0, to, -1);
		}

		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Route Downloaded"));
		g_free(rdl_data.bytes);
		route_set_destination_from_last();

		/* Success! Get out of the while loop. */
		ret=TRUE;
		break;
	} else {
		popup_error(dialog, _("Error parsing GPX file."));
		g_free(rdl_data.bytes);
		/* Let them try again. */
		ret=FALSE;
	}
}

gtk_widget_hide(dialog);	/* Destroying causes a crash (!?!?!??!) */

return ret;
}

WayPoint *
route_find_nearest_waypoint(guint unitx, guint unity)
{
WayPoint *wcurr;
WayPoint *wnear;
guint64 nearest_squared;
Point pos = { unitx, unity, 0, NAN };

wcurr = wnear = _route->whead;
if (wcurr && wcurr->point && wcurr != _route->wtail) {
	nearest_squared = DISTANCE_SQUARED(pos, *(wcurr->point));

	while (wcurr++ != _route->wtail) {
		guint64 test_squared = DISTANCE_SQUARED(pos, *(wcurr->point));
		if (test_squared < nearest_squared) {
			wnear = wcurr;
			nearest_squared = test_squared;
		}
	}
}

if (wnear && wnear->point) {
	/* Only use the waypoint if it is within a 6*_draw_width square drawn
	 * around the position. This is consistent with select_poi(). */
	if (abs(unitx - wnear->point->unitx) < pixel2unit(3 * _draw_width) &&
			abs(unity - wnear->point->unity) < pixel2unit(3 * _draw_width))
		return wnear;
}

MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("There are no waypoints."));

return NULL;
}

/**
 * Updates _near_point, _next_way, and _next_wpt.  If quick is FALSE (as
 * it is when this function is called from route_find_nearest_point), then
 * the entire list (starting from _near_point) is searched.  Otherwise, we
 * stop searching when we find a point that is farther away.
 */
gboolean 
route_update_nears(gboolean quick)
{
gboolean ret = FALSE;
Point *curr, *near;
WayPoint *wcurr, *wnext;
guint64 near_dist_squared;

/* If we have waypoints (_next_way != NULL), then determine the "next
 * waypoint", which is defined as the waypoint after the nearest point,
 * UNLESS we've passed that waypoint, in which case the waypoint after
 * that waypoint becomes the "next" waypoint. */
if (_next_way) {
	/* First, set near_dist_squared with the new distance from
	 * _near_point. */
	near = _near_point;
	near_dist_squared = DISTANCE_SQUARED(_gps->data, *near);

	/* Now, search _route for a closer point.  If quick is TRUE, then we'll
	 * only search forward, only as long as we keep finding closer points.
	 */
	for (curr = _near_point; curr++ != _route->tail;) {
		if (curr->unity) {
			guint dist_squared = DISTANCE_SQUARED(_gps->data, *curr);
			if (dist_squared <= near_dist_squared) {
				near = curr;
				near_dist_squared = dist_squared;
			} else if (quick)
				break;
		}
	}

	/* Update _near_point. */
	_near_point = near;
	_near_point_dist_squared = near_dist_squared;

	for (wnext = wcurr = _next_way; wcurr != _route->wtail; wcurr++) {
		if (wcurr->point < near || (wcurr->point == near && quick 
				&& (_next_wpt && (DISTANCE_SQUARED(_gps->data, *near) > _next_way_dist_squared
				&& DISTANCE_SQUARED(_gps->data, *_next_wpt) < _next_wpt_dist_squared))))
		    /* Okay, this else if expression warrants explanation.  If the
		     * nearest track point happens to be a waypoint, then we want to
		     * check if we have "passed" that waypoint.  To check this, we
		     * test the distance from _gps to the waypoint and from _gps to
		     * _next_wpt, and if the former is increasing and the latter is
		     * decreasing, then we have passed the waypoint, and thus we
		     * should skip it.  Note that if there is no _next_wpt, then
		     * there is no next waypoint, so we do not skip it in that case. */
			wnext = wcurr + 1;
		else
			break;
	}

	if (wnext == _route->wtail && (wnext->point < near || (wnext->point == near && quick
					  && (_next_wpt && (DISTANCE_SQUARED (_gps->data, *near) > _next_way_dist_squared
					       && DISTANCE_SQUARED(_gps->data, *_next_wpt) < _next_wpt_dist_squared)))))
	{
		_next_way = NULL;
		_next_wpt = NULL;
		_next_way_dist_squared = -1;
		_next_wpt_dist_squared = -1;
		ret = TRUE;
	}
	/* Only update _next_way (and consequently _next_wpt) if _next_way is
	 * different, and record that fact for return. */
	else {
		if (!quick || _next_way != wnext) {
			_next_way = wnext;
			_next_wpt = wnext->point;
			if (_next_wpt == _route->tail)
				_next_wpt = NULL;
			else {
				while (!(++_next_wpt)->unity) {
					if (_next_wpt == _route->tail) {
						_next_wpt = NULL;
						break;
					}
				}
			}
			ret = TRUE;
		}
		_next_way_dist_squared = DISTANCE_SQUARED(_gps->data, *wnext->point);
		if (_next_wpt)
			_next_wpt_dist_squared = DISTANCE_SQUARED(_gps->data, *_next_wpt);
	}
}
return ret;
}

/**
 * Reset the _near_point data by searching the entire route for the nearest
 * route point and waypoint.
 */
void 
route_find_nearest_point()
{
/* Initialize _near_point to first non-zero point. */
_near_point = _route->head;
while (!_near_point->unity && _near_point != _route->tail)
	_near_point++;

/* Initialize _next_way. */
if (_route->wtail == _route->whead - 1 || (_autoroute_data.enabled && _route->wtail == _route->whead))
	_next_way = NULL;
else
	/* We have at least one waypoint. */
	_next_way = (_autoroute_data.enabled ? _route->whead + 1 : _route->whead);

_next_way_dist_squared = -1;

/* Initialize _next_wpt. */
_next_wpt = NULL;
_next_wpt_dist_squared = -1;

route_update_nears(FALSE);
}

/**
 * Show the distance from the current GPS location to the given point,
 * following the route.  If point is NULL, then the distance is shown to the
 * next waypoint.
 */
gdouble
route_get_distance_to(Point *point)
{
gdouble lat1, lon1, lat2, lon2;
gdouble sum = 0.0;

/* If point is NULL, use the next waypoint. */
if (point == NULL && _next_way)
	point = _next_way->point;

/* If point is still NULL, return an error. */
if (point == NULL)
	return -1;

unit2latlon(_gps->data.unitx, _gps->data.unity, lat1, lon1);
if (point > _near_point) {
	Point *curr;
	/* Skip _near_point in case we have already passed it. */
	for (curr = _near_point + 1; curr <= point; ++curr) {
		if (curr->unity) {
			unit2latlon(curr->unitx, curr->unity, lat2, lon2);
			sum += calculate_distance(lat1, lon1, lat2, lon2);
			lat1 = lat2;
			lon1 = lon2;
		}
	}
} else if (point < _near_point) {
	Point *curr;
	/* Skip _near_point in case we have already passed it. */
	for (curr = _near_point - 1; curr >= point; --curr) {
		if (curr->unity) {
			unit2latlon(curr->unitx, curr->unity, lat2, lon2);
			sum += calculate_distance(lat1, lon1, lat2, lon2);
			lat1 = lat2;
			lon1 = lon2;
		}
	}
} else {
	/* Waypoint _is_ the nearest point. */
	unit2latlon(_near_point->unitx, _near_point->unity, lat2, lon2);
	sum += calculate_distance(lat1, lon1, lat2, lon2);
}
return sum;
}

gboolean 
route_show_distance_to(Point *point)
{
gchar buffer[80];
gdouble d;

d=route_get_distance_to(point);
g_snprintf(buffer, sizeof(buffer), "%s: %.02f %s", _("Distance"), d * UNITS_CONVERT[_units], UNITS_TEXT[_units]);
MACRO_BANNER_SHOW_INFO(mapp.mainwindow, buffer);

return TRUE;
}

void 
route_show_distance_to_next()
{
if (!route_show_distance_to(NULL)) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("There is no next waypoint."));
}
}

void
route_set_destination_from_last(void)
{
Point *p;
gdouble lat,lon;
if (_route->head == _route->tail)
	return;

/* Find last non-zero point. */
for (p = _route->tail; !p->unity; p--) {
}
unit2latlon(p->unitx, p->unity, lat, lon);
_dest.valid=TRUE;
_dest.lat=lat;
_dest.lon=lon;
}

void 
route_show_distance_to_last(void)
{
Point *p;

if (_route->head != _route->tail) {
	/* Find last non-zero point. */
	for (p = _route->tail; !p->unity; p--) {
	}
	route_show_distance_to(p);
} else {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("The current route is empty."));
	}
}

/***/

