/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <libintl.h>
#include <locale.h>
#include <sqlite3.h>

#include "hildon-mapper.h"

#include "utils.h"
#include "poi.h"
#include "gps.h"
#include "map.h"
#include "mapper-types.h"
#include "latlon.h"
#include "ui-common.h"
#include "settings.h"
#include "db.h"
#include "gps-scan.h"
#include "osm-db.h"
#include "settings-gui.h"
#include "config-gconf.h"
#include "hildon-wrappers.h"
#include "dialogs.h"
#include "speak.h"

typedef struct _KeysDialogInfo KeysDialogInfo;
struct _KeysDialogInfo {
	GtkWidget *cmb[CUSTOM_KEY_ENUM_COUNT];
};

typedef struct _ColorsDialogInfo ColorsDialogInfo;
struct _ColorsDialogInfo {
	GtkWidget *col[COLORABLE_ENUM_COUNT];
};

typedef struct _GpsDialogInfo GpsDialogInfo;
struct _GpsDialogInfo {
	GtkWidget *cmb_gps_type;
	GtkWidget *txt_rcvr_mac;
	GtkWidget *spn_rcvr_port;
	GtkWidget *btn_scan;
	GtkWidget *addr_label;
	GtkTreeIter active;
	GtkListStore *store;
	GtkTreeIter iter;
};

#define UPDATE_GC(gc) gdk_gc_set_line_attributes(gc, _draw_width, GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);

void 
update_gcs(void)
{
gint i;

for (i=0; i < COLORABLE_ENUM_COUNT; i++) {
	gdk_color_alloc(gtk_widget_get_colormap(mapp.map_widget),&_color[i]);
	if (_gc[i])
		g_object_unref(_gc[i]);
	_gc[i]=gdk_gc_new(map_pixmap_get());
	gdk_gc_set_foreground(_gc[i], &_color[i]);
	gdk_gc_set_line_attributes(_gc[i], _draw_width, GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);
}

}

static gboolean
settings_dialog_browse_forfile(GtkWidget *widget, BrowseInfo *browse_info)
{
GtkWidget *dialog;
gint res;

dialog=GTK_WIDGET(hildon_file_chooser_dialog_new(GTK_WINDOW(browse_info->dialog),GTK_FILE_CHOOSER_ACTION_OPEN));

gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(dialog), TRUE);
gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), gtk_entry_get_text(GTK_ENTRY(browse_info->txt)));

res=gtk_dialog_run(GTK_DIALOG(dialog));
if (res==GTK_RESPONSE_OK || res==GTK_RESPONSE_ACCEPT) {
	gchar *filename=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
	gtk_entry_set_text(GTK_ENTRY(browse_info->txt), filename);
	g_free(filename);
}

gtk_widget_destroy(dialog);

return TRUE;
}

static gboolean
settings_dialog_hardkeys_reset(GtkWidget *widget, KeysDialogInfo *cdi)
{
GtkWidget *confirm;

confirm=hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), _("Reset all hardware keys to their original defaults?"));

if (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm))) {
	gint i;
	for (i=0; i < CUSTOM_KEY_ENUM_COUNT; i++)
		gtk_combo_box_set_active(GTK_COMBO_BOX(cdi->cmb[i]), CUSTOM_KEY_DEFAULT[i]);
}
gtk_widget_destroy(confirm);
return TRUE;
}

gboolean 
settings_dialog_hardkeys(void)
{
gint i;
GtkWidget *dialog;
GtkWidget *table;
GtkWidget *label;
KeysDialogInfo bdi;
GtkWidget *btn_defaults;

dialog=gtk_dialog_new_with_buttons(_("Hardware Keys"), GTK_WINDOW(mapp.mainwindow), GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), btn_defaults=gtk_button_new_with_label(_("Reset...")));
g_signal_connect(G_OBJECT(btn_defaults), "clicked",	G_CALLBACK(settings_dialog_hardkeys_reset), &bdi);

gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table=gtk_table_new(2, 9, FALSE), TRUE, TRUE, 0);

for (i=0; i < CUSTOM_KEY_ENUM_COUNT; i++) {
	gint j;
	gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(""), 0, 1, i, i + 1, GTK_FILL, 0, 2, 1);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
#ifdef WITH_HILDON
	gtk_label_set_markup(GTK_LABEL(label), CUSTOM_KEY_ICON[i]);
#endif
	gtk_table_attach(GTK_TABLE(table), bdi.cmb[i]=gtk_combo_box_new_text(), 1, 2, i, i + 1, GTK_FILL, 0, 2, 1);
		for (j=0; j < CUSTOM_ACTION_ENUM_COUNT; j++)
			gtk_combo_box_append_text(GTK_COMBO_BOX(bdi.cmb[i]), CUSTOM_ACTION_TEXT[j]);
		gtk_combo_box_set_active(GTK_COMBO_BOX(bdi.cmb[i]), _action[i]);
}

gtk_widget_show_all(dialog);

OUTER_WHILE:
while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	/* Check for duplicates. */
	for (i=0; i < CUSTOM_KEY_ENUM_COUNT; i++) {
		gint j;
		for (j=i + 1; j < CUSTOM_KEY_ENUM_COUNT; j++) {
			if (gtk_combo_box_get_active(GTK_COMBO_BOX(bdi.cmb[i])) == gtk_combo_box_get_active(GTK_COMBO_BOX(bdi.cmb[j]))) {
				GtkWidget *confirm;
				gchar *buffer=g_strdup_printf("%s:\n    %s\n%s", _("The following action is mapped to multiple keys"),
						    CUSTOM_ACTION_TEXT[gtk_combo_box_get_active(GTK_COMBO_BOX(bdi.cmb[i]))], _("Continue?"));
				confirm=hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), buffer);

				if (GTK_RESPONSE_OK != gtk_dialog_run(GTK_DIALOG(confirm)))
				{
					gtk_widget_destroy(confirm);
					goto OUTER_WHILE;
				}
				gtk_widget_destroy(confirm);
			}
		}
	}
	for (i=0; i < CUSTOM_KEY_ENUM_COUNT; i++)
		_action[i]=gtk_combo_box_get_active(GTK_COMBO_BOX(bdi.cmb[i]));
	break;
}

gtk_widget_destroy(dialog);

return TRUE;
}

static gboolean
settings_dialog_colors_reset(GtkWidget *widget, ColorsDialogInfo *cdi)
{
GtkWidget *confirm;

confirm=hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), _("Reset all colors to their original defaults?"));

if (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm))) {
	gint i;
	for (i=0; i < COLORABLE_ENUM_COUNT; i++) {
		gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi->col[i]), &COLORABLE_DEFAULT[i]);
	}
}
gtk_widget_destroy(confirm);

return TRUE;
}

gboolean 
settings_dialog_colors(void)
{
GtkWidget *dialog;
GtkWidget *table;
GtkWidget *label;
GtkWidget *btn_defaults;
ColorsDialogInfo cdi;

dialog=gtk_dialog_new_with_buttons(_("Colors"), GTK_WINDOW(mapp.mainwindow), GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), btn_defaults=gtk_button_new_with_label(_("Reset...")));
g_signal_connect(G_OBJECT(btn_defaults), "clicked", G_CALLBACK(settings_dialog_colors_reset), &cdi);

gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table=gtk_table_new(4, 3, TRUE), TRUE, TRUE, 0);

/* GPS. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("GPS")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_MARK]=gtk_color_button_new(), 1, 2, 0, 1, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_MARK]), &_color[COLORABLE_MARK]);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_MARK_VELOCITY]=gtk_color_button_new(), 2, 3, 0, 1, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_MARK_VELOCITY]), &_color[COLORABLE_MARK_VELOCITY]);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_MARK_OLD]=gtk_color_button_new(), 3, 4, 0, 1, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_MARK_OLD]), &_color[COLORABLE_MARK_OLD]);

/* Track. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Track")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_TRACK]=gtk_color_button_new(), 1, 2, 1, 2, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_TRACK]), &_color[COLORABLE_TRACK]);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_TRACK_MARK]=gtk_color_button_new(), 2, 3, 1, 2, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_TRACK_MARK]), &_color[COLORABLE_TRACK_MARK]);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_TRACK_BREAK]=gtk_color_button_new(), 3, 4, 1, 2, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_TRACK_BREAK]), &_color[COLORABLE_TRACK_BREAK]);

/* Route. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Route")), 0, 1, 2, 3, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_ROUTE]=gtk_color_button_new(), 1, 2, 2, 3, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_ROUTE]), &_color[COLORABLE_ROUTE]);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_ROUTE_WAY]=gtk_color_button_new(), 2, 3, 2, 3, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_ROUTE_WAY]), &_color[COLORABLE_ROUTE_WAY]);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_ROUTE_BREAK]=gtk_color_button_new(), 3, 4, 2, 3, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_ROUTE_BREAK]), &_color[COLORABLE_ROUTE_BREAK]);

/* POI. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("POI")), 0, 1, 3, 4, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), cdi.col[COLORABLE_POI]=gtk_color_button_new(), 1, 2, 3, 4, 0, 0, 2, 4);
gtk_color_button_set_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_POI]), &_color[COLORABLE_POI]);

gtk_widget_show_all(dialog);
if (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_MARK]), &_color[COLORABLE_MARK]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_MARK_VELOCITY]), &_color[COLORABLE_MARK_VELOCITY]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_MARK_OLD]), &_color[COLORABLE_MARK_OLD]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_TRACK]), &_color[COLORABLE_TRACK]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_TRACK_MARK]), &_color[COLORABLE_TRACK_MARK]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_TRACK_BREAK]),&_color[COLORABLE_TRACK_BREAK]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_ROUTE]),&_color[COLORABLE_ROUTE]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_ROUTE_WAY]),&_color[COLORABLE_ROUTE_WAY]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_ROUTE_BREAK]),&_color[COLORABLE_ROUTE_BREAK]);
	gtk_color_button_get_color(GTK_COLOR_BUTTON(cdi.col[COLORABLE_POI]),&_color[COLORABLE_POI]);
	update_gcs();
}

gtk_widget_destroy(dialog);

return TRUE;
}

static void
gtk_widget_hideshow(GtkWidget *widget, gboolean show)
{
g_assert(widget);
if (show==TRUE)
	gtk_widget_show(widget);
else
	gtk_widget_hide(widget);
}

static gboolean
gps_type_changed_cb(GtkWidget *widget, gpointer data)
{
GpsDialogInfo *gdi=(GpsDialogInfo *)data;
gboolean scan, port, address;
GpsIOSourceType type;
guint dport;

if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(gdi->cmb_gps_type), &gdi->iter)==FALSE)
	return TRUE;

gtk_tree_model_get(GTK_TREE_MODEL(gdi->store), &gdi->iter, 0, &type, 2, &scan, 3, &address, 4, &port, 5, &dport, -1);
g_debug("GPS: %d", type);

gtk_widget_hideshow(gdi->btn_scan, scan);
gtk_widget_hideshow(gdi->txt_rcvr_mac, address);
gtk_widget_hideshow(gdi->addr_label, address);
gtk_widget_hideshow(gdi->spn_rcvr_port, port);
if (port && dport!=0 && (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gdi->spn_rcvr_port))==0)
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gdi->spn_rcvr_port), (guint)dport);

return TRUE;
}

gboolean
settings_dialog_gps(Gps *gps)
{
GtkWidget *dialog;
GtkWidget *vbox;
GtkWidget *hbox;
GtkWidget *label;
GtkCellRenderer *renderer;
GpsDialogInfo gdi;
GpsScanInfo scan_info;
gboolean rcvr_changed=TRUE, settype=FALSE;
GpsTypes *gpstypes;

/* Don't allow configuring GPS if it's connected */
if (gps->io.conn!=RCVR_OFF) {
	popup_error(mapp.mainwindow, "Disable GPS before re-configuring.");
	return FALSE;
}

dialog=gtk_dialog_new_with_buttons(_("GPS Settings"), GTK_WINDOW(mapp.mainwindow), GTK_DIALOG_MODAL, 
	GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox=gtk_vbox_new(FALSE, 4), TRUE, TRUE, 0);

/* Type */
hbox=gtk_hbox_new(FALSE, 3);
gtk_box_pack_start(GTK_BOX(hbox), label=gtk_label_new(_("GPS Type")), TRUE, TRUE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

gdi.cmb_gps_type=gtk_combo_box_new_with_model(GTK_TREE_MODEL(gtk_list_store_new(6, G_TYPE_INT, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_INT)));

gtk_box_pack_start(GTK_BOX(hbox), gdi.cmb_gps_type, TRUE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

/* Add GPS types */
renderer=gtk_cell_renderer_text_new();
gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(gdi.cmb_gps_type), renderer, TRUE);
gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(gdi.cmb_gps_type), renderer, "text", 1, NULL);

/* Receiver MAC/File/IP Address. */
hbox=gtk_hbox_new(FALSE, 3);
gtk_box_pack_start(GTK_BOX(hbox), gdi.addr_label=gtk_label_new(_("Address: (MAC/IP/Device)")), TRUE, TRUE, 0);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_box_pack_start(GTK_BOX(hbox), gdi.txt_rcvr_mac=gtk_entry_new(), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), gdi.spn_rcvr_port=gtk_spin_button_new_with_range(0, 65535, 1), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), gdi.btn_scan=gtk_button_new_with_label(_("Scan...")), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, FALSE, 0);

/* Set up the view for the combo box. */
gdi.store=GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(gdi.cmb_gps_type)));
gtk_list_store_clear(gdi.store);
gpstypes=gps_get_supported_types();
while (gpstypes->type!=0) {
	gtk_list_store_append(gdi.store, &gdi.iter);
	gtk_list_store_set(gdi.store, &gdi.iter, 
		0, gpstypes->type, 
		1, gpstypes->name, 
		2, gpstypes->scan, 
		3, gpstypes->address, 
		4, gpstypes->port, 
		5, gpstypes->port_def, 
		-1);
	if (gps->io.type==gpstypes->type) {
		gdi.active=gdi.iter;
		settype=TRUE;
	} else if (settype==FALSE && gps->io.type==GPS_IO_SIMULATION) {
		gdi.active=gdi.iter;
		settype=TRUE;
	}
	gpstypes++;
}

scan_info.window=dialog;
scan_info.txt_rcvr_mac=gdi.txt_rcvr_mac;
scan_info.dbus_conn=mapper_app.dbus_conn;

/* Connect signals. */
g_signal_connect(G_OBJECT(gdi.btn_scan), "clicked", G_CALLBACK(gps_scan_bluetooth), &scan_info);
g_signal_connect(G_OBJECT(gdi.cmb_gps_type), "changed", G_CALLBACK(gps_type_changed_cb), &gdi);

gtk_entry_set_text(GTK_ENTRY(gdi.txt_rcvr_mac), gps->io.address ? gps->io.address : "");
gtk_spin_button_set_value(GTK_SPIN_BUTTON(gdi.spn_rcvr_port), gps->io.port);

gtk_window_set_default_size(GTK_WINDOW(dialog), 640, 250);
gtk_widget_show_all(dialog);

if (settype==FALSE)
	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(gdi.store), &gdi.active);
gtk_combo_box_set_active_iter(GTK_COMBO_BOX(gdi.cmb_gps_type), &gdi.active);

while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	/* Set _rcvr_mac if necessary. */
	if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(gdi.cmb_gps_type), &gdi.iter)==TRUE) {
		gtk_tree_model_get(GTK_TREE_MODEL(gdi.store), &gdi.iter, 0, &(gps->io.type), -1);
		g_debug("GPS: Selected %d", gps->io.type);
	} else {
		popup_error(dialog, _("Please select GPS type"));
		continue;
	}

	if (!*gtk_entry_get_text(GTK_ENTRY(gdi.txt_rcvr_mac))) {
		/* User specified no rcvr mac - set _rcvr_mac to NULL. */
		if (gps->io.address) {
			g_free(gps->io.address);
			gps->io.address=NULL;
			rcvr_changed=TRUE;
			set_action_sensitive("gps_details", FALSE);
		}
		if (gps_type_needs_config(gps->io.type)) {
			g_debug("GPS: Disable, set to sim");
			set_action_activate("gps_details", FALSE);
			popup_error(dialog, _("No GPS Receiver MAC provided.\nGPS will be disabled."));
			rcvr_changed=TRUE;
			gps->io.type=GPS_IO_SIMULATION;
			set_action_sensitive("gps_details", FALSE);
		}
	} else if (!gps->io.address || strcmp(gps->io.address, gtk_entry_get_text(GTK_ENTRY(gdi.txt_rcvr_mac)))) {
		/* User specified a new rcvr mac. */
		g_free(gps->io.address);
		gps->io.address=g_strdup(gtk_entry_get_text(GTK_ENTRY(gdi.txt_rcvr_mac)));
		gps->io.port=(guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gdi.spn_rcvr_port));
		rcvr_changed=TRUE;
	}

	config_save();
	break;
}

g_debug("GPS: Set to %d", gps->io.type);
gtk_widget_destroy(dialog);	
return rcvr_changed;
}

/* XXX: Find a better place for this */
static void
reinit_database(gchar *newdb)
{
if (_db) {
	poi_deinit_db(_db);
	osm_deinit();
	db_close(&_db);
	_db=NULL;
}
if (_mapper_db) {
	g_free(_mapper_db);
	_mapper_db=NULL;
}
_mapper_db=newdb;
if (_mapper_db==NULL)
	return;
db_connect(&_db, _mapper_db);
poi_init_db(&_db);
osm_init(&_db);
}

gboolean 
settings_dialog_osm(void)
{
GtkWidget *dialog;
GtkWidget *notebook;
GtkWidget *table;
GtkWidget *hbox;
GtkWidget *label;
GtkWidget *txt_mapper_db;
GtkWidget *btn_browsedb;
GtkWidget *num_sens;
BrowseInfo browse_info={ 0, 0 };

dialog=gtk_dialog_new_with_buttons(_("OSM Settings"), GTK_WINDOW(mapp.mainwindow), GTK_DIALOG_MODAL, 
		GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook=gtk_notebook_new(), TRUE, TRUE, 0);

/* Database. */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("Database")));
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Database")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), hbox=gtk_hbox_new(FALSE, 4), 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);
gtk_box_pack_start(GTK_BOX(hbox), txt_mapper_db=gtk_entry_new(), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), btn_browsedb=gtk_button_new_with_label(_("Browse...")), FALSE, FALSE, 0);
gtk_entry_set_text(GTK_ENTRY(txt_mapper_db), _mapper_db ? _mapper_db : "");

#if 0
/* Advanced */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("Advanced")));
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Sensitivity")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), num_sens=hildon_controlbar_new(), 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 2, 4);
hildon_controlbar_set_range(HILDON_CONTROLBAR(num_sens), 1, 10);
force_min_visible_bars(HILDON_CONTROLBAR(num_sens), 1);
hildon_controlbar_set_value(HILDON_CONTROLBAR(num_sens), _map_location_sens);
#endif

browse_info.dialog=dialog;
browse_info.txt=txt_mapper_db;
g_signal_connect(G_OBJECT(btn_browsedb), "clicked", G_CALLBACK(settings_dialog_browse_forfile), &browse_info);

gtk_window_set_default_size(GTK_WINDOW(dialog), 710, 350);
gtk_widget_show_all(dialog);

if (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
#if 0
	_map_location_sens=hildon_controlbar_get_value(HILDON_CONTROLBAR(num_sens));
#endif
	if (strlen(gtk_entry_get_text(GTK_ENTRY(txt_mapper_db)))>0) {
		gchar *tmp;
		tmp=g_strdup(gtk_entry_get_text(GTK_ENTRY(txt_mapper_db)));
		if (tmp && _mapper_db && strcmp(tmp, _mapper_db)!=0) {
			reinit_database(tmp);
		}
	} else {
		reinit_database(NULL);
	}
	config_save();
}
gtk_widget_destroy(dialog);	

/* Sensitize menus based on whether we have a database conn. */
gtk_widget_set_sensitive(_cmenu_loc_add_poi, _db != NULL);
gtk_widget_set_sensitive(_cmenu_way_add_poi, _db != NULL);
gtk_widget_set_sensitive(_cmenu_poi, _db != NULL);

return TRUE;
}

/**
 * Bring up the Settings dialog.
 */
gboolean 
settings_dialog()
{
GtkWidget *dialog;
GtkWidget *notebook;
GtkWidget *table;
GtkWidget *hbox;
GtkWidget *vbox;
GtkWidget *hbox2;
GtkWidget *label;
GtkWidget *num_center_ratio;
GtkWidget *num_lead_ratio;
GtkWidget *num_announce_notice;
GtkWidget *chk_enable_voice;
GtkWidget *num_voice_speed;
GtkWidget *num_voice_pitch;
GtkWidget *lbl_voice_speed;
GtkWidget *lbl_voice_pitch;
GtkWidget *num_draw_width;
GtkWidget *chk_always_keep_on;
GtkWidget *cmb_units;
GtkWidget *cmb_degformat;
GtkWidget *num_poi_zoom;
GtkWidget *chk_speed_on;
GtkWidget *num_speed;
GtkWidget *cmb_info_font_size;

GtkWidget *announce_destination;
GtkWidget *announce_overspeed;
GtkWidget *announce_waypoints;
guint i;

dialog=gtk_dialog_new_with_buttons(_("Settings"), GTK_WINDOW(mapp.mainwindow), GTK_DIALOG_MODAL, 
	GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook=gtk_notebook_new(), TRUE, TRUE, 0);

/* Auto-Center page. */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 2, FALSE), label=gtk_label_new(_("Auto-Center")));

/* Auto-Center Sensitivity. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Sensitivity")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), num_center_ratio=hildon_controlbar_new(), 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 2, 4);
hildon_controlbar_set_range(HILDON_CONTROLBAR(num_center_ratio), 1, 10);
force_min_visible_bars(HILDON_CONTROLBAR(num_center_ratio), 1);

/* Lead Amount. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Lead Amount")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

gtk_table_attach(GTK_TABLE(table), num_lead_ratio=hildon_controlbar_new(), 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);
hildon_controlbar_set_range(HILDON_CONTROLBAR(num_lead_ratio), 1, 10);
force_min_visible_bars(HILDON_CONTROLBAR(num_lead_ratio), 1);

/* Announcement page */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("Announce")));

/* Announcement Advance Notice. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Advance Notice")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), num_announce_notice=hildon_controlbar_new(), 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 2, 4);
hildon_controlbar_set_range(HILDON_CONTROLBAR(num_announce_notice), 1, 20);
force_min_visible_bars(HILDON_CONTROLBAR(num_announce_notice), 1);

gtk_table_attach(GTK_TABLE(table), vbox=gtk_vbox_new(FALSE, 4), 0, 2, 2, 3, GTK_EXPAND | GTK_FILL, 0, 2, 6);
gtk_box_pack_start(GTK_BOX(vbox), announce_waypoints=gtk_check_button_new_with_label(_("Enable waypoint announcements")), TRUE, TRUE, 4);
gtk_box_pack_start(GTK_BOX(vbox), announce_destination=gtk_check_button_new_with_label(_("Enable destination announcements")), TRUE, TRUE, 4);
gtk_box_pack_start(GTK_BOX(vbox), announce_overspeed=gtk_check_button_new_with_label(_("Enable overspeed announcements")), TRUE, TRUE, 4);

/* Voice page */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("Voice")));

/* Enable Voice. */
gtk_table_attach(GTK_TABLE(table), chk_enable_voice=gtk_check_button_new_with_label(_("Enable Voice Synthesis")), 0, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);
gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chk_enable_voice), _enable_voice);

/* Voice Speed and Pitch. */
gtk_table_attach(GTK_TABLE(table), hbox=gtk_hbox_new(FALSE, 4), 0, 2, 2, 3, GTK_EXPAND | GTK_FILL, 0, 2, 6);
gtk_box_pack_start(GTK_BOX(hbox), hbox2=gtk_hbox_new(FALSE, 4), TRUE, TRUE, 4);
gtk_box_pack_start(GTK_BOX(hbox2), lbl_voice_speed=gtk_label_new(_("Speed (words/minute)")), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox2), num_voice_speed=hildon_number_editor_new(80, 370), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), hbox2=gtk_hbox_new(FALSE, 4), TRUE, TRUE, 4);
gtk_box_pack_start(GTK_BOX(hbox2), lbl_voice_pitch=gtk_label_new(_("Pitch")), TRUE, TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox2), num_voice_pitch=hildon_number_editor_new(1, 99), TRUE, TRUE, 0);

/* Display settings page. */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("Display")));

/* Line Width. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Line Width")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), num_draw_width=hildon_controlbar_new(), 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 2, 4);
hildon_controlbar_set_range(HILDON_CONTROLBAR(num_draw_width), 1, 20);
force_min_visible_bars(HILDON_CONTROLBAR(num_draw_width), 1);

/* Keep Display On Only When Fullscreen. */
gtk_table_attach(GTK_TABLE(table), chk_always_keep_on=gtk_check_button_new_with_label(_("Keep Display On Only in Fullscreen Mode")), 0, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);

/* Information Font Size. */
gtk_table_attach(GTK_TABLE(table), hbox=gtk_hbox_new(FALSE, 4), 0, 2, 2, 3, GTK_FILL, 0, 2, 4);
gtk_box_pack_start(GTK_BOX(hbox), label=gtk_label_new(_("Information Font Size")), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), cmb_info_font_size=gtk_combo_box_new_text(), FALSE, FALSE, 0);
for (i=0; i < INFO_FONT_ENUM_COUNT; i++)
	gtk_combo_box_append_text(GTK_COMBO_BOX(cmb_info_font_size), INFO_FONT_TEXT[i]);

/* Units page. */
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("Units")));

/* Units. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Units")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), cmb_units=gtk_combo_box_new_text(), 1, 2, 0, 1, GTK_FILL, 0, 2, 4);
for (i=0; i < UNITS_ENUM_COUNT; i++)
	gtk_combo_box_append_text(GTK_COMBO_BOX(cmb_units), UNITS_TEXT[i]);

/* Degrees format */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Degrees Format")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), label=gtk_alignment_new(0.f, 0.5f, 0.f, 0.f), 1, 2, 1, 2, GTK_FILL, 0, 2, 4);
gtk_container_add(GTK_CONTAINER(label), cmb_degformat=gtk_combo_box_new_text());
for (i=0; i < DEG_FORMAT_ENUM_COUNT; i++)
	gtk_combo_box_append_text(GTK_COMBO_BOX(cmb_degformat), DEG_FORMAT_TEXT[i]);
gtk_table_attach(GTK_TABLE(table), gtk_label_new(""), 2, 3, 0, 2, GTK_FILL | GTK_EXPAND, 0, 2, 4);

/* Speed warner. */
gtk_table_attach(GTK_TABLE(table), hbox=gtk_hbox_new(FALSE, 4), 0, 3, 2, 3, GTK_FILL, 0, 2, 4);
gtk_box_pack_start(GTK_BOX(hbox), chk_speed_on=gtk_check_button_new_with_label(_("Display Speed")), FALSE, FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), label=gtk_alignment_new(0.f, 0.5f, 0.f, 0.f), FALSE, FALSE, 0);
gtk_container_add(GTK_CONTAINER(label), num_speed=hildon_number_editor_new(0, 999));

gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table=gtk_table_new(2, 3, FALSE), label=gtk_label_new(_("POIs")));

/* Show POI below zoom. */
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Show POI below zoom")), 0, 1, 2, 3, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
gtk_table_attach(GTK_TABLE(table), label=gtk_alignment_new(0.f, 0.5f, 0.f, 0.f), 1, 2, 2, 3, GTK_EXPAND | GTK_FILL, 0, 2, 4);
gtk_container_add(GTK_CONTAINER(label), num_poi_zoom=hildon_number_editor_new(0, 10));

/* Initialize fields. */
hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(num_poi_zoom), _poi_zoom);
hildon_controlbar_set_value(HILDON_CONTROLBAR(num_center_ratio), _center_ratio);
hildon_controlbar_set_value(HILDON_CONTROLBAR(num_lead_ratio), _lead_ratio);
hildon_controlbar_set_value(HILDON_CONTROLBAR(num_announce_notice), _announce_notice_ratio);
hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(num_voice_speed), _voice_speed);
hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(num_voice_pitch), _voice_pitch);
hildon_controlbar_set_value(HILDON_CONTROLBAR(num_draw_width), _draw_width);
gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chk_always_keep_on), !_always_keep_on);
gtk_combo_box_set_active(GTK_COMBO_BOX(cmb_units), _units);
gtk_combo_box_set_active(GTK_COMBO_BOX(cmb_degformat), _degformat);
gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chk_speed_on), _speed_on);
hildon_number_editor_set_range(HILDON_NUMBER_EDITOR(num_speed), 1, 300);
hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(num_speed), _speed_limit);
gtk_combo_box_set_active(GTK_COMBO_BOX(cmb_info_font_size), _info_font_size);

gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(announce_waypoints), _announce_waypoints);
gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(announce_overspeed), _announce_overspeed);
gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(announce_destination), _announce_destination);

gtk_window_set_default_size(GTK_WINDOW(dialog), 710, 350);
gtk_widget_show_all(dialog);

if (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	_center_ratio=hildon_controlbar_get_value(HILDON_CONTROLBAR(num_center_ratio));
	_lead_ratio=hildon_controlbar_get_value(HILDON_CONTROLBAR(num_lead_ratio));
	_draw_width=hildon_controlbar_get_value(HILDON_CONTROLBAR(num_draw_width));
	_always_keep_on=!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chk_always_keep_on));
	_units=gtk_combo_box_get_active(GTK_COMBO_BOX(cmb_units));
	_degformat=gtk_combo_box_get_active(GTK_COMBO_BOX(cmb_degformat));
	_speed_on=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chk_speed_on));
	_speed_limit=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(num_speed));
	_info_font_size=gtk_combo_box_get_active(GTK_COMBO_BOX(cmb_info_font_size));
	_announce_notice_ratio=hildon_controlbar_get_value(HILDON_CONTROLBAR(num_announce_notice));
	_voice_speed=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(num_voice_speed));
	_voice_pitch=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(num_voice_pitch));
	_enable_voice=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chk_enable_voice));
	_poi_zoom=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(num_poi_zoom));

	_announce_waypoints=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(announce_waypoints));
	_announce_overspeed=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(announce_overspeed));
	_announce_destination=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(announce_destination));

	speak_set_parameters(_voice_speed, _voice_pitch);
	update_gcs();
	config_save();
}

#ifndef WITH_HILDON
/* Destroying causes a crash (!?!?!??!)  huh? in hildon (770?) only, plain gtk is ok...*/
gtk_widget_destroy(dialog);	
#else
gtk_widget_hide(dialog);
#endif

return FALSE;
}
