/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * Default map data provided by http://www.openstreetmap.org/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <libintl.h>
#include <locale.h>
#include <math.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>

#include "hildon-mapper.h"

#include "utils.h"
#include "gps.h"
#include "map.h"
#include "latlon.h"
#include "route.h"
#include "track.h"
#include "settings.h"
#include "mapper-types.h"
#include "map-download.h"
#include "ui-common.h"
#include "dialogs.h"
#include "config-gconf.h"
#include "map-repo.h"

typedef struct _RepoEditInfo RepoEditInfo;
struct _RepoEditInfo {
	gchar *name;
	GtkWidget *txt_url;
	GtkWidget *txt_cache_dir;
	GtkWidget *num_dl_zoom_steps;
	GtkWidget *num_view_zoom_steps;
	GtkWidget *chk_double_size;
	GtkWidget *chk_nextable;
	GtkWidget *btn_browse;
	BrowseInfo browse_info;
};

typedef struct _MapmanInfo MapmanInfo;
struct _MapmanInfo {
	GtkWidget *dialog;
	GtkWidget *notebook;
	GtkWidget *tbl_area;

	/* The "Setup" tab. */
	GtkWidget *rad_download;
	GtkWidget *rad_delete;
	GtkWidget *chk_overwrite;
	GtkWidget *rad_by_area;
	GtkWidget *rad_by_route;
	GtkWidget *rad_by_track;
	GtkWidget *num_route_radius;
	GtkWidget *num_track_radius;

	/* The "Area" tab. */
	GtkWidget *txt_topleft_lat;
	GtkWidget *txt_topleft_lon;
	GtkWidget *txt_botright_lat;
	GtkWidget *txt_botright_lon;

	/* The "Zoom" tab. */
	GtkWidget *chk_zoom_levels[MAX_ZOOM];
};

void set_repo_type(RepoData * repo)
{
if (repo->url && *repo->url) {
	gchar *url = g_utf8_strdown(repo->url, -1);

	/* Determine type of repository. */
	if (strstr(url, "service=wms"))
		repo->type = REPOTYPE_WMS;
	else if (strstr(url, "%s"))
		repo->type = REPOTYPE_QUAD_QRST;
	else if (strstr(url, "%0d"))
		repo->type = REPOTYPE_XYZ_INV;
	else if (strstr(url, "%0s"))
		repo->type = REPOTYPE_QUAD_ZERO;
	else
		repo->type = REPOTYPE_XYZ;

	g_free(url);
} else
	repo->type = REPOTYPE_NONE;
}

gboolean
repo_make_cache_dir(gchar * name, const gchar * cache_dir, GtkWidget * parent)
{
if (g_mkdir_with_parents(cache_dir, 0755)) {
	/* Failed to create Map Cache directory. */
	gchar buffer[BUFFER_SIZE];
	g_snprintf(buffer, sizeof(buffer), "%s: %s",
		 _("Unable to create cache directory for repository"), name);
	popup_error(parent, buffer);
	return FALSE;
}
return g_file_test(cache_dir, G_FILE_TEST_EXISTS);
}

gboolean 
repo_set_curr(RepoData * rd)
{
_curr_repo = rd;
return repo_make_cache_dir(rd->name, rd->cache_dir, mapp.mainwindow);
}

static gboolean
mapman_by_area(gfloat start_lat, gfloat start_lon,
	       gfloat end_lat, gfloat end_lon, MapmanInfo * mapman_info,
	       gboolean is_deleting, gboolean is_overwriting)
{
guint start_unitx, start_unity, end_unitx, end_unity;
guint num_maps = 0;
guint i;
gchar buffer[80];
GtkWidget *confirm;

latlon2unit(start_lat, start_lon, start_unitx, start_unity);
latlon2unit(end_lat, end_lon, end_unitx, end_unity);

/* Swap if they specified flipped lats or lons. */
if (start_unitx > end_unitx) {
	guint swap = start_unitx;
	start_unitx = end_unitx;
	end_unitx = swap;
}
if (start_unity > end_unity) {
	guint swap = start_unity;
	start_unity = end_unity;
	end_unity = swap;
}

/* First, get the number of maps to download. */
for (i = 0; i < MAX_ZOOM; i++) {
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->chk_zoom_levels[i]))) {
		guint start_tilex, start_tiley, end_tilex, end_tiley;
		start_tilex = unit2ztile(start_unitx, i);
		start_tiley = unit2ztile(start_unity, i);
		end_tilex = unit2ztile(end_unitx, i);
		end_tiley = unit2ztile(end_unity, i);
		num_maps += (end_tilex - start_tilex + 1) * (end_tiley - start_tiley + 1);
		}
}

if (is_deleting) {
	g_snprintf(buffer, sizeof(buffer), "%s %d %s", _("Confirm DELETION of"), num_maps, _("maps "));
} else {
	g_snprintf(buffer, sizeof(buffer), "%s %d %s\n(%s %.2f MB)\n", _("Confirm download of"),
		num_maps, _("maps"), _("up to about"), num_maps * (strstr(_curr_repo->url, "%s") ? 18e-3 : 6e-3));
}
confirm = hildon_note_new_confirmation(GTK_WINDOW(mapman_info->dialog), buffer);

if (GTK_RESPONSE_OK != gtk_dialog_run(GTK_DIALOG(confirm))) {
	gtk_widget_destroy(confirm);
	return FALSE;
}

for (i = 0; i < MAX_ZOOM; i++) {
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->chk_zoom_levels[i]))) {
		guint start_tilex, start_tiley, end_tilex, end_tiley;
		guint tilex, tiley;
		start_tilex = unit2ztile(start_unitx, i);
		start_tiley = unit2ztile(start_unity, i);
		end_tilex = unit2ztile(end_unitx, i);
		end_tiley = unit2ztile(end_unity, i);

		for (tiley = start_tiley; tiley <= end_tiley; tiley++)
			for (tilex = start_tilex; tilex <= end_tilex; tilex++)
				map_initiate_download(tilex, tiley, i, is_deleting ? 0 : (is_overwriting ?
							 -INITIAL_DOWNLOAD_RETRIES : INITIAL_DOWNLOAD_RETRIES));
	}
}
gtk_widget_destroy(confirm);
return TRUE;
}

static gboolean
mapman_by_path(Path *path, MapmanInfo *mapman_info, guint radius, gboolean is_deleting, gboolean is_overwriting)
{
GtkWidget *confirm;
guint prev_tilex, prev_tiley, num_maps = 0, i;
Point *curr;
gchar buffer[80];

/* First, get the number of maps to download. */
for (i = 0; i < MAX_ZOOM; i++) {
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->chk_zoom_levels[i]))) {
		prev_tilex = 0;
		prev_tiley = 0;

		for (curr = path->head - 1; curr++ != path->tail;) {
			if (curr->unity) {
				guint tilex = unit2ztile(curr->unitx, i);
				guint tiley = unit2ztile(curr->unity, i);
				if (tilex != prev_tilex || tiley != prev_tiley) {
					if (prev_tiley)
						num_maps += (abs((gint) tilex - prev_tilex) + 1) * (abs((gint) tiley - prev_tiley) + 1) - 1;
					prev_tilex = tilex;
					prev_tiley = tiley;
				}
			}
		}
	}
}
num_maps *= 0.625 * pow(radius + 1, 1.85);

if (is_deleting) {
	g_snprintf(buffer, sizeof(buffer), "%s %s %d %s",
		 _("Confirm DELETION of"), _("about"), num_maps, _("maps "));
} else {
	g_snprintf(buffer, sizeof(buffer),
		 "%s %s %d %s\n(%s %.2f MB)\n",
		 _("Confirm download of"), _("about"), num_maps,
		 _("maps"), _("up to about"),
		 num_maps * (strstr(_curr_repo->url, "%s") ? 18e-3 : 6e-3));
}
confirm = hildon_note_new_confirmation(GTK_WINDOW(mapman_info->dialog), buffer);

if (GTK_RESPONSE_OK != gtk_dialog_run(GTK_DIALOG(confirm))) {
	gtk_widget_destroy(confirm);
	return FALSE;
}

/* Now, do the actual download. */
for (i = 0; i < MAX_ZOOM; i++) {
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->chk_zoom_levels[i]))) {
		prev_tilex = 0;
		prev_tiley = 0;

		for (curr = path->head - 1; curr++ != path->tail;) {
			if (curr->unity) {
				guint tilex = unit2ztile(curr->unitx, i);
				guint tiley = unit2ztile(curr->unity, i);
				if (tilex != prev_tilex || tiley != prev_tiley) {
					guint minx, miny, maxx, maxy, x, y;
					if (prev_tiley != 0) {
						minx = MIN(tilex, prev_tilex) - radius;
						miny = MIN(tiley, prev_tiley) - radius;
						maxx = MAX(tilex, prev_tilex) + radius;
						maxy = MAX(tiley, prev_tiley) + radius;
					} else {
						minx = tilex - radius;
						miny = tiley - radius;
						maxx = tilex + radius;
						maxy = tiley + radius;
					}
					for (x = minx; x <= maxx; x++)
						for (y = miny; y <= maxy; y++)
							map_initiate_download(x, y, i, is_deleting ? 0
							     : (is_overwriting ? -INITIAL_DOWNLOAD_RETRIES : INITIAL_DOWNLOAD_RETRIES));
					prev_tilex = tilex;
					prev_tiley = tiley;
				}
			}
		}
	}
}
gtk_widget_destroy(confirm);
return TRUE;
}

static void 
mapman_clear(GtkWidget * widget, MapmanInfo * mapman_info)
{
guint i;

if (gtk_notebook_get_current_page(GTK_NOTEBOOK(mapman_info->notebook))) {
	/* This is the second page (the "Zoom" page) - clear the checks. */
	for (i = 0; i < MAX_ZOOM; i++)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mapman_info->chk_zoom_levels[i]), FALSE);
} else {
	/* This is the first page (the "Area" page) - clear the text fields. */
	gtk_entry_set_text(GTK_ENTRY(mapman_info->txt_topleft_lat), "");
	gtk_entry_set_text(GTK_ENTRY(mapman_info->txt_topleft_lon), "");
	gtk_entry_set_text(GTK_ENTRY(mapman_info->txt_botright_lat), "");
	gtk_entry_set_text(GTK_ENTRY(mapman_info->txt_botright_lon), "");
}
}

static void 
mapman_update_state(GtkWidget * widget, MapmanInfo * mapman_info)
{
gtk_widget_set_sensitive(mapman_info->chk_overwrite, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->rad_download)));

if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->rad_by_area)))
	gtk_widget_show(mapman_info->tbl_area);
else if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(mapman_info->notebook)) == 3)
	gtk_widget_hide(mapman_info->tbl_area);

gtk_widget_set_sensitive(mapman_info->num_route_radius, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->rad_by_route)));
gtk_widget_set_sensitive(mapman_info->num_track_radius, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info->rad_by_track)));
}

gboolean menu_cb_mapman(GtkAction * action)
{
	GtkWidget *dialog;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *lbl_gps_lat;
	GtkWidget *lbl_gps_lon;
	GtkWidget *lbl_center_lat;
	GtkWidget *lbl_center_lon;
	MapmanInfo mapman_info;
	gchar buffer[80];
	gfloat lat, lon;
	guint i;

	mapman_info.dialog = dialog = gtk_dialog_new_with_buttons(_("Manage Maps"), GTK_WINDOW(mapp.mainwindow),
					GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

	/* Clear button. */
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area),
			  button = gtk_button_new_with_label(_("Clear")));
	g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(mapman_clear), &mapman_info);

	/* Cancel button. */
	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), mapman_info.notebook = gtk_notebook_new(), TRUE, TRUE, 0);

	/* Setup page. */
	gtk_notebook_append_page(GTK_NOTEBOOK(mapman_info.notebook), vbox = gtk_vbox_new(FALSE, 2), label = gtk_label_new(_("Setup")));
	gtk_notebook_set_tab_label_packing(GTK_NOTEBOOK(mapman_info.notebook), vbox, FALSE, FALSE, GTK_PACK_START);

	gtk_box_pack_start(GTK_BOX(vbox), hbox = gtk_hbox_new(FALSE, 4), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), mapman_info.rad_download = gtk_radio_button_new_with_label(NULL, _("Download Maps")), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), label = gtk_alignment_new(0.f, 0.5f, 0.f, 0.f), FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(label), mapman_info.chk_overwrite =
			  gtk_check_button_new_with_label(_("Overwrite"))),
	    gtk_box_pack_start(GTK_BOX(vbox), mapman_info.rad_delete =
			       gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(mapman_info.rad_download), _("Delete Maps")), FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), mapman_info.rad_by_area  = gtk_radio_button_new_with_label(NULL, _("By Area (see tab)")), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox = gtk_hbox_new(FALSE, 4), FALSE, FALSE, 0);

	/* Route */
	gtk_box_pack_start(GTK_BOX(hbox), mapman_info.rad_by_route = gtk_radio_button_new_with_label_from_widget
			   (GTK_RADIO_BUTTON(mapman_info.rad_by_area), _("Along Route - Radius (tiles):")), FALSE, FALSE, 0);
	gtk_widget_set_sensitive(mapman_info.rad_by_route, _route->head != _route->tail);
	gtk_box_pack_start(GTK_BOX(hbox), mapman_info.num_route_radius = hildon_number_editor_new(0, 100), FALSE, FALSE, 0);
	hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(mapman_info.num_route_radius), _route_dl_radius);

	/* Track */
	gtk_box_pack_start(GTK_BOX(vbox), hbox = gtk_hbox_new(FALSE, 4), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), mapman_info.rad_by_track = gtk_radio_button_new_with_label_from_widget
			   (GTK_RADIO_BUTTON(mapman_info.rad_by_area), _("Along Track - Radius (tiles):")), FALSE, FALSE, 0);
	gtk_widget_set_sensitive(mapman_info.rad_by_track, _track->head != _track->tail);
	gtk_box_pack_start(GTK_BOX(hbox), mapman_info.num_track_radius = hildon_number_editor_new(0, 100), FALSE, FALSE, 0);
	hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(mapman_info.num_track_radius), _track_dl_radius);

	/* Zoom page. */
	gtk_notebook_append_page(GTK_NOTEBOOK(mapman_info.notebook),
				 table = gtk_table_new(5, 5, FALSE),
				 label = gtk_label_new(_("Zoom")));
	gtk_notebook_set_tab_label_packing(GTK_NOTEBOOK(mapman_info.notebook), table, FALSE, FALSE, GTK_PACK_START);
	gtk_table_attach(GTK_TABLE(table), label =
			 gtk_label_new(_("Zoom Levels to Download: (0 = most detail)")),
			 0, 4, 0, 1, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
	for (i = 0; i < MAX_ZOOM; i++) {
		g_snprintf(buffer, sizeof(buffer), "%d", i);
		gtk_table_attach(GTK_TABLE(table),
				 mapman_info.chk_zoom_levels[i]
				 = gtk_check_button_new_with_label(buffer),
				 i % 4, i % 4 + 1, i / 4 + 1, i / 4 + 2,
				 GTK_EXPAND | GTK_FILL, 0, 4, 0);
	}

	/* Area page. */
	gtk_notebook_append_page(GTK_NOTEBOOK(mapman_info.notebook),
				 mapman_info.tbl_area =
				 gtk_table_new(3, 4, FALSE), label =
				 gtk_label_new(_("Area")));

	/* Label Columns. */
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 label = gtk_label_new(_("Latitude")),
			 1, 2, 0, 1, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 label = gtk_label_new(_("Longitude")),
			 2, 3, 0, 1, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

	/* GPS. */
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 label = gtk_label_new(_("GPS Location")),
			 0, 1, 1, 2, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 lbl_gps_lat = gtk_label_new(""),
			 1, 2, 1, 2, GTK_FILL, 0, 4, 0);
	gtk_label_set_selectable(GTK_LABEL(lbl_gps_lat), TRUE);
	gtk_misc_set_alignment(GTK_MISC(lbl_gps_lat), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 lbl_gps_lon = gtk_label_new(""),
			 2, 3, 1, 2, GTK_FILL, 0, 4, 0);
	gtk_label_set_selectable(GTK_LABEL(lbl_gps_lon), TRUE);
	gtk_misc_set_alignment(GTK_MISC(lbl_gps_lon), 1.f, 0.5f);

	/* Center. */
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 label = gtk_label_new(_("View Center")),
			 0, 1, 2, 3, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 lbl_center_lat = gtk_label_new(""),
			 1, 2, 2, 3, GTK_FILL, 0, 4, 0);
	gtk_label_set_selectable(GTK_LABEL(lbl_center_lat), TRUE);
	gtk_misc_set_alignment(GTK_MISC(lbl_center_lat), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 lbl_center_lon = gtk_label_new(""),
			 2, 3, 2, 3, GTK_FILL, 0, 4, 0);
	gtk_label_set_selectable(GTK_LABEL(lbl_center_lon), TRUE);
	gtk_misc_set_alignment(GTK_MISC(lbl_center_lon), 1.f, 0.5f);

	/* default values for Top Left and Bottom Right are defined by the
	 * rectangle of the current and the previous Center */

	/* Top Left. */
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 label = gtk_label_new(_("Top-Left")),
			 0, 1, 3, 4, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 mapman_info.txt_topleft_lat = gtk_entry_new(),
			 1, 2, 3, 4, GTK_EXPAND | GTK_FILL, 0, 4, 0);
	gtk_entry_set_alignment(GTK_ENTRY(mapman_info.txt_topleft_lat), 1.f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 mapman_info.txt_topleft_lon = gtk_entry_new(),
			 2, 3, 3, 4, GTK_EXPAND | GTK_FILL, 0, 4, 0);
	gtk_entry_set_alignment(GTK_ENTRY(mapman_info.txt_topleft_lon), 1.f);

	/* Bottom Right. */
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 label = gtk_label_new(_("Bottom-Right")),
			 0, 1, 4, 5, GTK_FILL, 0, 4, 0);
	gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 mapman_info.txt_botright_lat = gtk_entry_new(),
			 1, 2, 4, 5, GTK_EXPAND | GTK_FILL, 0, 4, 0);
	gtk_entry_set_alignment(GTK_ENTRY(mapman_info.txt_botright_lat), 1.f);
	gtk_table_attach(GTK_TABLE(mapman_info.tbl_area),
			 mapman_info.txt_botright_lon = gtk_entry_new(),
			 2, 3, 4, 5, GTK_EXPAND | GTK_FILL, 0, 4, 0);
	gtk_entry_set_alignment(GTK_ENTRY(mapman_info.txt_botright_lon), 1.f);

#if defined(WITH_DEVICE_MAEMO) && !defined(WITH_HILDON_1)
	/* Set hildon input hints */
	g_object_set(G_OBJECT(mapman_info.txt_topleft_lon),
		     HILDON_INPUT_MODE_HINT,
		     HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
	g_object_set(G_OBJECT(mapman_info.txt_topleft_lat),
		     HILDON_INPUT_MODE_HINT,
		     HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
	g_object_set(G_OBJECT(mapman_info.txt_botright_lon),
		     HILDON_INPUT_MODE_HINT,
		     HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
	g_object_set(G_OBJECT(mapman_info.txt_botright_lat),
		     HILDON_INPUT_MODE_HINT,
		     HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
#endif

	/* Default action is to download by area. */
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mapman_info.rad_by_area), TRUE);

	/* Initialize fields.  Do no use g_ascii_formatd; these strings will be
	 * output (and parsed) as locale-dependent. */

	g_snprintf(buffer, sizeof(buffer), "%.06f", _gps->data.lat);
	gtk_label_set_text(GTK_LABEL(lbl_gps_lat), buffer);
	g_snprintf(buffer, sizeof(buffer), "%.06f", _gps->data.lon);
	gtk_label_set_text(GTK_LABEL(lbl_gps_lon), buffer);

	unit2latlon(_center.unitx, _center.unity, lat, lon);
	g_snprintf(buffer, sizeof(buffer), "%.06f", lat);
	gtk_label_set_text(GTK_LABEL(lbl_center_lat), buffer);
	g_snprintf(buffer, sizeof(buffer), "%.06f", lon);
	gtk_label_set_text(GTK_LABEL(lbl_center_lon), buffer);

	/* Initialize to the bounds of the screen. */
	unit2latlon(x2unit(0), y2unit(0), lat, lon);
	g_snprintf(buffer, sizeof(buffer), "%.06f", lat);
	gtk_entry_set_text(GTK_ENTRY(mapman_info.txt_topleft_lat), buffer);
	g_snprintf(buffer, sizeof(buffer), "%.06f", lon);
	gtk_entry_set_text(GTK_ENTRY(mapman_info.txt_topleft_lon), buffer);

	unit2latlon(x2unit(_screen_width_pixels), y2unit(_screen_height_pixels), lat, lon);
	g_snprintf(buffer, sizeof(buffer), "%.06f", lat);
	gtk_entry_set_text(GTK_ENTRY(mapman_info.txt_botright_lat), buffer);
	g_snprintf(buffer, sizeof(buffer), "%.06f", lon);
	gtk_entry_set_text(GTK_ENTRY(mapman_info.txt_botright_lon), buffer);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mapman_info.chk_zoom_levels[_zoom]), TRUE);

	gtk_widget_show_all(dialog);

	mapman_update_state(NULL, &mapman_info);

	/* Connect signals. */
	if (_curr_repo->type != REPOTYPE_NONE) {
		g_signal_connect(G_OBJECT(mapman_info.rad_download), "clicked", G_CALLBACK(mapman_update_state), &mapman_info);
		gtk_widget_set_sensitive(mapman_info.rad_download, TRUE);
	} else {
		gtk_widget_set_sensitive(mapman_info.rad_download, FALSE);
		popup_error(dialog, _("NOTE: You must set a Map URI in the current repository in order to download maps."));
	}

	g_signal_connect(G_OBJECT(mapman_info.rad_delete), "clicked", G_CALLBACK(mapman_update_state), &mapman_info);
	g_signal_connect(G_OBJECT(mapman_info.rad_by_area), "clicked", G_CALLBACK(mapman_update_state), &mapman_info);
	g_signal_connect(G_OBJECT(mapman_info.rad_by_route), "clicked", G_CALLBACK(mapman_update_state), &mapman_info);
	g_signal_connect(G_OBJECT(mapman_info.rad_by_track), "clicked", G_CALLBACK(mapman_update_state), &mapman_info);

	while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
		gboolean is_deleting = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info.rad_delete));
		gboolean is_overwriting = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info.chk_overwrite));
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info.rad_by_route))) {
			_route_dl_radius = hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(mapman_info.num_route_radius));
			if (mapman_by_path(_route, &mapman_info, _route_dl_radius, is_deleting, is_overwriting))
				break;
		} else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mapman_info.rad_by_track))) {
			_track_dl_radius = hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(mapman_info.num_track_radius));
			if (mapman_by_path(_track, &mapman_info, _track_dl_radius, is_deleting, is_overwriting))
				break;
		} else {
			const gchar *text;
			gchar *error_check;
			gfloat start_lat, start_lon, end_lat, end_lon;

			text = gtk_entry_get_text(GTK_ENTRY(mapman_info.txt_topleft_lat));
			start_lat = strtof(text, &error_check);
			if (text == error_check || start_lat < -90.f || start_lat > 90.f) {
				popup_error(dialog, _("Invalid Top-Left Latitude"));
				continue;
			}

			text = gtk_entry_get_text(GTK_ENTRY(mapman_info.txt_topleft_lon));
			start_lon = strtof(text, &error_check);
			if (text == error_check || start_lon < -180.f || start_lon > 180.f) {
				popup_error(dialog, _("Invalid Top-Left Longitude"));
				continue;
			}

			text = gtk_entry_get_text(GTK_ENTRY(mapman_info.txt_botright_lat));
			end_lat = strtof(text, &error_check);
			if (text == error_check || end_lat < -90.f || end_lat > 90.f) {
				popup_error(dialog, _("Invalid Bottom-Right Latitude"));
				continue;
			}

			text = gtk_entry_get_text(GTK_ENTRY(mapman_info.txt_botright_lon));
			end_lon = strtof(text, &error_check);
			if (text == error_check || end_lon < -180.f || end_lon > 180.f) {
				popup_error(dialog, _("Invalid Bottom-Right Longitude"));
				continue;
			}

			if (mapman_by_area(start_lat, start_lon, end_lat, end_lon, &mapman_info, is_deleting, is_overwriting))
				break;
		}
	}

	gtk_widget_hide(dialog);	/* Destroying causes a crash (!?!?!??!) */

	return TRUE;
}
