/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <gtk/gtk.h>

#include "utils.h"
#include "poi.h"
#include "gps.h"
#include "map.h"
#include "settings.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "map-poi.h"
#include "latlon.h"
#include "osm-db.h"

static GdkGC *poi_gc;
static GtkListStore *poi_store=NULL;
static guint last_zoom=0;
static guint prev_ux=0, prev_uy=0;
static guint prev_gx=0, prev_gy=0;

typedef struct _poibox poibox;
struct _poibox {
	gboolean valid;
	guint cnt;
	gdouble lat_max,lat_min,lon_max,lon_min;
};

static poibox poi_b;

#define POI_CACHE_RANGE (4)

/****************************************************************************/

gboolean 
map_poi_init(GtkWidget *map_widget)
{
poi_gc=gdk_gc_new(map_widget->window);
poi_b.valid=FALSE;
poi_b.cnt=0;
return TRUE;
}

void
map_poi_deinit(void)
{
map_poi_cache_clear();
}

/** 
 * Set POI color from given #rrggbb string, fallback to default if it fails
 */
static inline GdkGC *
map_set_poi_fg_color_from_string(gchar *hexcolor, GdkGC *def)
{
GdkColor color;

if (!hexcolor)
	return def;

if (gdk_color_parse(hexcolor, &color)) {
	gdk_gc_set_rgb_fg_color(poi_gc, &color);
	return poi_gc;
}

return def;
}

void
map_poi_cache_clear(void)
{
if (!poi_store)
	return;
gtk_list_store_clear(poi_store);
poi_b.valid=FALSE;
}

#define MP_IS_INSIDE(nlat, nlon) (nlat > poi_b.lat_min && nlat < poi_b.lat_max && nlon > poi_b.lon_min && nlon < poi_b.lon_max)

static gboolean is_inside(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2)
{
if (!MP_IS_INSIDE(lat1, lon1) || !MP_IS_INSIDE(lat2, lon2))
	return FALSE;
return TRUE;
}

static gboolean
map_poi_reload_poi(gdouble lat1, gdouble lon1, gdouble lat2, gdouble lon2)
{
guint pois;
gboolean r;
gdouble lat_d,lon_d;

/* Check if we are inside the cached area and we are on same zoom level, skip reload if so */
if (poi_b.valid==TRUE && is_inside(lat1, lon1, lat2, lon2) && last_zoom==_zoom)
	return TRUE;

map_poi_cache_clear();

if (_zoom<8) {
	lat_d=fabs(lat2-lat1)*(POI_CACHE_RANGE - poi_b.cnt>900 ? 2 : 0);
	lon_d=fabs(lon2-lon1)*(POI_CACHE_RANGE - poi_b.cnt>900 ? 2 : 0);
} else {
	lat_d=lon_d=0.05;
}

lat1-=lat_d;
lon1-=lat_d;
lat2+=lon_d;
lon2+=lon_d;

r=poi_get_list_inside(lat1, lon1, lat2, lon2, &poi_store, &pois);

/* Store area */
poi_b.valid=r;
poi_b.lat_min=lat1;
poi_b.lat_max=lat2;
poi_b.lon_min=lon1;
poi_b.lon_max=lon2;
poi_b.cnt=pois;

return r;
}

gboolean
map_poi_find_at_map_xy(gint x, gint y, gint range, guint *poi_id)
{
GtkTreeIter iter;
gboolean found=FALSE;
gboolean valid;
gint hr=range/2;

if (_poi_zoom <= _zoom) 
	return FALSE;
if (!poi_store)
	return FALSE;

valid=gtk_tree_model_get_iter_first(GTK_TREE_MODEL(poi_store), &iter);
if (valid==FALSE)
	return FALSE;

while (valid) {
	gdouble lat,lon;
	guint id, ux, uy;
	gint px, py;

	gtk_tree_model_get(GTK_TREE_MODEL(poi_store),
			&iter,
			ITEM_ID, &id,
			ITEM_LAT, &lat,
			ITEM_LON, &lon,
			-1);

	latlon2unit(lat, lon, ux, uy);
	px=unit2x(ux);
	py=unit2y(uy);

	if (px-hr<x && px+hr>x && py-hr<y && py+hr>y) {
		found=TRUE;
		*poi_id=id;
	}

	valid=gtk_tree_model_iter_next(GTK_TREE_MODEL(poi_store), &iter);
}

return found;
}

gboolean
map_poi_find_at_latlon(gdouble lat, gdouble lon, guint *poi_id)
{
GtkTreeIter iter;
gboolean found=FALSE;
gboolean valid;
gdouble pdist=99999.0;

g_return_val_if_fail(poi_store, FALSE);
if (_poi_zoom <= _zoom) 
	return FALSE;

valid=gtk_tree_model_get_iter_first(GTK_TREE_MODEL(poi_store), &iter);
if (valid==FALSE)
	return FALSE;

while (valid) {
	gdouble tmp, plat, plon;
	guint id;

	gtk_tree_model_get(GTK_TREE_MODEL(poi_store),
			&iter,
			ITEM_ID, &id,
			ITEM_LAT, &plat,
			ITEM_LON, &plon,
			-1);

	/* XXX: Use quicker and simple distance check */
	tmp=calculate_distance(lat, lon, plat, plon);
	if ((tmp<pdist) && (tmp<0.011)) {
		found=TRUE;
		pdist=tmp;
		*poi_id=id;
	}

	valid=gtk_tree_model_iter_next(GTK_TREE_MODEL(poi_store), &iter);
}

return found;
}

/**
 * Render all the POI data.
 * This should be done before rendering track data.
 */
void
map_render_all_pois(guint width, guint height)
{
GtkTreeIter iter;
guint unitx, unity;
gdouble lat1, lat2, lon1, lon2;
gboolean valid;

if (_poi_zoom <= _zoom) 
	return;

unitx=x2unit(0);
unity=y2unit(height);
unit2latlon(unitx, unity, lat1, lon1);

unitx=x2unit(width);
unity=y2unit(0);
unit2latlon(unitx, unity, lat2, lon2);

if (map_poi_reload_poi(lat1, lon1, lat2, lon2)==FALSE)
	return;

valid=gtk_tree_model_get_iter_first(GTK_TREE_MODEL(poi_store), &iter);
if (!valid)
	return;

/* Store previous location */
last_zoom=_zoom;
prev_ux=unitx;
prev_uy=unity;
prev_gx=unit2grid(unitx);
prev_gy=unit2grid(unity);

while (valid) {
	GdkPixbuf *pixbuf;
	GdkGC *gc;
	gchar *icon, *color;
	gboolean highlight=FALSE;
	gdouble dist;
	gint cid, poix, poiy;
	guint w,h;

	gtk_tree_model_get(GTK_TREE_MODEL(poi_store),
			&iter,
			ITEM_LAT, &lat1,
			ITEM_LON, &lon1,
			ITEM_CATID, &cid,
			-1);

	if (!poi_category_get_icon_and_color(cid, &icon, &color))
		continue;

	latlon2unit(lat1, lon1, unitx, unity);
	poix=unit2bufx(unitx);
	poiy=unit2bufy(unity);

	switch (cid) {
	case NODE_AMENITY_SPEEDCAM:
		pixbuf=poi_get_icon(icon, TRUE);
		dist=calculate_distance(_gps->data.lat, _gps->data.lon, lat1, lon1);
		if (dist<0.5) {
			highlight=TRUE;
			if (_gps->data.fix!=FIX_NOFIX && _gps->data.speed>1) {
				
			}
		}	
	break;
	case NODE_HIGHWAY_BUS_STOP:
		pixbuf=poi_get_icon(icon, FALSE);
	break;
	case NODE_BUILDING:
		pixbuf=NULL;
	break;
	default:
		pixbuf=poi_get_icon(icon, FALSE);
	}
	gc=map_set_poi_fg_color_from_string(color, _gc[COLORABLE_POI]);

	if (!pixbuf) {
		/* No icon for POI or for category - draw default. */
		w=h=16;
		gdk_draw_rectangle(map_pixmap_get(), gc, TRUE, poix-w/2-1, poiy-h/2-1, w+1, h+1);
	} else {
		w=gdk_pixbuf_get_width(pixbuf);
		h=gdk_pixbuf_get_height(pixbuf);
		gdk_draw_pixbuf(map_pixmap_get(), gc, pixbuf, 0, 0, poix-w/2, poiy-h/2, -1, -1, GDK_RGB_DITHER_NONE, 0, 0);
		if (highlight)
			gdk_draw_rectangle(map_pixmap_get(), gc, FALSE, poix-w/2-1, poiy-h/2-1, w+1, h+1);
	}

	valid=gtk_tree_model_iter_next(GTK_TREE_MODEL(poi_store), &iter);
}
}
