/*
 * Copyright (c) 2010 Andreas Henriksson <andreas@fatal.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTRCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tracking.h"

#include <math.h>
#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>
#include <gio/gio.h>
#include <string.h>

G_DEFINE_TYPE (MapBuddyTracking, map_buddy_tracking, G_TYPE_OBJECT)

#define MAP_BUDDY_TRACKING_GET_PRIVATE(obj)	\
			(G_TYPE_INSTANCE_GET_PRIVATE((obj), \
			MAP_BUDDY_TYPE_TRACKING, MapBuddyTrackingPrivate))

struct TrackingDetails
{
	/* location_fixes is a list of LocationGPSDeviceFix.
	 *
	 * The reason this struct exists is that ChamplainPoint is
	 * not a real GObject. That means we can't inherit it and
	 * store extra GPS position information that want to have
	 * later when exporting to GPX.
	 *
	 * Each location_fixes item should match a point in the track.
	 * The track is what we draw, and the location_fixes is what
	 * we save.
	 */
	GList *location_fixes;
	ChamplainPolygon *track;
};

struct _MapBuddyTrackingPrivate
{
	struct TrackingDetails *tracking_details;
	LocationGPSDControlInterval old_gps_interval;
};

static void
map_buddy_tracking_class_init (MapBuddyTrackingClass *klass)
{
	g_type_class_add_private (klass, sizeof (MapBuddyTrackingPrivate));
}

static void
map_buddy_tracking_init (MapBuddyTracking *self)
{
	MapBuddyTrackingPrivate *priv;

	self->priv = priv = MAP_BUDDY_TRACKING_GET_PRIVATE (self);

	priv->tracking_details = NULL;
}

MapBuddyTracking *
map_buddy_tracking_new (void)
{
	return MAP_BUDDY_TRACKING (g_object_new (MAP_BUDDY_TYPE_TRACKING, NULL));
}

/* private helper functions */
static void
write_gpx_trkpt (gpointer data, gpointer user_data)
{
  LocationGPSDeviceFix *fix = data;
  GFileOutputStream *fos = user_data;
  gchar *buf;
  gsize written;

  g_return_if_fail (fix != NULL);
  g_return_if_fail ((fix->fields & LOCATION_GPS_DEVICE_LATLONG_SET) != 0);

  buf = g_strdup_printf("\t<trkpt lat=\"%f\" lon=\"%f\">\n",
      fix->latitude, fix->longitude);
  g_output_stream_write_all (G_OUTPUT_STREAM(fos),
          buf, strlen (buf), &written, NULL, NULL);
  g_free (buf);

  if ((fix->fields & LOCATION_GPS_DEVICE_ALTITUDE_SET) != 0)
  {
    buf = g_strdup_printf("\t\t<ele>%f</ele>\n", fix->altitude);
    g_output_stream_write_all (G_OUTPUT_STREAM(fos),
            buf, strlen (buf), &written, NULL, NULL);
    g_free (buf);
  }

  if ((fix->fields & LOCATION_GPS_DEVICE_TIME_SET) != 0)
  {
    time_t time1;
    struct tm time2;
    time1 = (time_t)(fix->time);
    localtime_r(&time1, &time2);

    buf = g_malloc (200);
    strftime (buf, 100, "\t\t<time>%FT%T", localtime(&time1));
    snprintf(buf+strlen(buf), 200-strlen(buf), "%+03ld:%02ld</time>\n",
        (time2.tm_gmtoff / 60 / 60),
        (time2.tm_gmtoff / 60) % 60);
    g_output_stream_write_all (G_OUTPUT_STREAM(fos),
            buf, strlen (buf), &written, NULL, NULL);
    g_free (buf);
  }

  buf = NULL;
  if ((fix->mode & LOCATION_GPS_DEVICE_MODE_2D) != 0)
    buf = "\t\t<fix>2d</fix>\n";
  else if ((fix->mode & LOCATION_GPS_DEVICE_MODE_3D) != 0)
    buf = "\t\t<fix>3d</fix>\n";
  if (buf != NULL)
    g_output_stream_write_all (G_OUTPUT_STREAM(fos),
            buf, strlen (buf), &written, NULL, NULL);

  buf = "\t</trkpt>\n";
  g_output_stream_write_all (G_OUTPUT_STREAM(fos),
          buf, strlen (buf), &written, NULL, NULL);

}

static double deg2rad(double deg) {
  return (deg * G_PI / 180);
}

static double rad2deg(double rad) {
  return (rad * 180 / G_PI);
}

static double
calc_distance(double lat1, double lon1, double lat2, double lon2, char unit) {
  double theta, dist;
  theta = lon1 - lon2;
  dist = sin(deg2rad(lat1)) * sin(deg2rad(lat2)) + cos(deg2rad(lat1)) * cos(deg2rad(lat2)) * cos(deg2rad(theta));
  dist = acos(dist);
  dist = rad2deg(dist);
  dist = dist * 60 * 1.1515;
  switch(unit) {
    case 'M':
      break;
    case 'K':
      dist = dist * 1.609344;
      break;
    case 'N':
      dist = dist * 0.8684;
      break;
  }
  return (dist);
}


static gboolean
check_certain_movement(struct TrackingDetails *td,
		LocationGPSDevice *device)
{
	LocationGPSDeviceFix *position = device->fix;
	LocationGPSDeviceFix *last_position;
	GList *tmp;
	double distance;
	double last_eph;

	/* find last position */
	tmp = g_list_last (td->location_fixes);
	if (tmp == NULL)
		return TRUE;

	last_position = tmp->data;

	/* calculate distance between last and current position */
	distance = calc_distance (last_position->latitude,
			last_position->longitude,
			position->latitude, position->longitude, 'K') * 1000.0;

	/* if we don't have any gps uncertainty, require minimum 25 meters */
	if (isnan(last_position->eph) && isnan(position->eph)
			&& (distance < 25.0))
		return FALSE;

	last_eph = last_position->eph;
	if (!isnan (last_eph) && !isnan(position->eph))
	{
		/* don't care for last position uncertainty if the current
		 * position uncertainty is twice as good.
		 */
		if ((last_eph / 2) > position->eph)
			last_eph = 0;

	}

	/* movement should be atleast half of the uncertainty radius */
	if (!isnan(last_eph) && distance < ((last_eph / 100.0)/2.0))
		return FALSE;
	if (!isnan(position->eph) && distance < ((position->eph / 100.0)/2.0))
		return FALSE;

	return TRUE;
}

/* public methods */

/**
 * map_buddy_tracking_start:
 * @self: the #MapBuddyTracking item.
 * @view: the #ChamplainView to draw the track on.
 * @gps_control: the gps controller to use for track points.
 *
 * Enables recording of a track and draws it.
 */
void
map_buddy_tracking_start (MapBuddyTracking *self, ChamplainView *view,
                          LocationGPSDControl *gps_control)
{
	struct TrackingDetails *td;
	MapBuddyTrackingPrivate *priv = MAP_BUDDY_TRACKING_GET_PRIVATE (self);

	if (priv->tracking_details != NULL)
	{
		g_warning ("enable_tracking called when tracking " \
				"already enabled!");
		return;
	}

	td = g_malloc0 (sizeof (struct TrackingDetails));

	td->track = champlain_polygon_new();
	champlain_view_add_polygon (view, td->track);

	/* save gps update interval, then raise it to max */
	g_object_get (G_OBJECT(gps_control),
			"preferred-interval", priv->old_gps_interval,
			NULL);
	location_gpsd_control_stop (
			LOCATION_GPSD_CONTROL (gps_control));
	g_object_set (G_OBJECT (gps_control),
			"preferred-interval", LOCATION_INTERVAL_1S,
			NULL);
	location_gpsd_control_start (
			LOCATION_GPSD_CONTROL (gps_control));

	priv->tracking_details = td;
}

/**
 * map_buddy_tracking_record_location:
 * @self: the #MapBuddyTracking item, NULL is ok.
 * @device: the #LocationGPSDevice structure from liblocation.
 *
 * Potentially records the current location as part of the track.
 *
 * Return value: %TRUE when value was recorded, %FALSE when discarded.
 */
gboolean
map_buddy_tracking_record_location (MapBuddyTracking *self, 
		LocationGPSDevice *device)
{
	MapBuddyTrackingPrivate *priv;
	struct TrackingDetails *td;
	LocationGPSDeviceFix *position;

	/* bail if tracking isn't enabled */
	if (self == NULL)
		return FALSE;

	/* bail if we don't have a fix on the location */
	g_return_val_if_fail (device->fix != NULL, FALSE);

	priv = MAP_BUDDY_TRACKING_GET_PRIVATE (self);
	td = priv->tracking_details;


	/* bail if current position is too close to last position */
	if (!check_certain_movement(td, device))
		return FALSE;

	/* append gps position details to tracking list. */
	position = g_memdup (device->fix, sizeof(*(device->fix)));
	td->location_fixes = g_list_append (td->location_fixes,
			position);

	/* append ChamplainPoint to ChamplainPolygon for tracking */
	champlain_polygon_append_point (td->track,
			position->latitude, position->longitude);
	champlain_polygon_set_mark_points (td->track, TRUE);
	champlain_polygon_show (td->track);

	return TRUE;
}

/**
 * map_buddy_tracking_stop:
 * @self: the #MapBuddyTracking item.
 * @track: the #ChamplainView to draw the track on.
 * @gps_control: the gps control to record track points.
 *
 * Stops a previously started tracking and discards
 * all recorded locations that was part of track.
 */
void
map_buddy_tracking_stop (MapBuddyTracking *self, ChamplainView *view,
                         LocationGPSDControl *gps_control)
{
	struct TrackingDetails *td;
	MapBuddyTrackingPrivate *priv = MAP_BUDDY_TRACKING_GET_PRIVATE (self);

	g_return_if_fail (priv->tracking_details != NULL);

	td = priv->tracking_details;
	priv->tracking_details = NULL;

	//champlain_view_remove_polygon (view, td->track);

	/* restore gps update interval to what it was when tracking started */
	location_gpsd_control_stop (
			LOCATION_GPSD_CONTROL (gps_control));
	g_object_set (G_OBJECT (gps_control),
			"preferred-interval", priv->old_gps_interval,
			NULL);
	location_gpsd_control_start (
			LOCATION_GPSD_CONTROL (gps_control));

	g_list_foreach (td->location_fixes, (GFunc)g_free, NULL);
	g_list_free (td->location_fixes);

	champlain_polygon_clear_points (td->track);
	champlain_view_remove_polygon (view, td->track);

	g_free (td);
}

/**
 * map_buddy_tracking_export_gpx:
 * @self: the #MapBuddyTracking item.
 * @uri: the uri where the file should be saved.
 *
 * Exports the current track to the given uri as a GPX file.
 */
void
map_buddy_tracking_export_gpx (MapBuddyTracking *self, gchar *uri)
{
	MapBuddyTrackingPrivate *priv = MAP_BUDDY_TRACKING_GET_PRIVATE (self);
	GFile *f;
	GFileOutputStream *fos;
	char *buf;
	gsize written;
	struct TrackingDetails *td;

	g_return_if_fail (priv->tracking_details != NULL);

	td = priv->tracking_details;

	f = g_file_new_for_uri (uri);

	fos = g_file_replace (f, NULL, TRUE, 0, NULL, NULL);
	if (fos == NULL)
	{
		g_object_unref (f);
		return;
	}

	/* write gpx header */
	buf = "<?xml version=\"1.0\"?>\n \
<gpx\n \
  version=\"1.1\"\n \
  creator=\"Map Buddy - Maemo map viewer application\"\n \
  xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n \
  xmlns=\"http://www.topografix.com/GPX/1/1\"\n \
  xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd\">\n \
<trk><trkseg>\n";
	g_output_stream_write_all (G_OUTPUT_STREAM(fos),
			buf, strlen (buf), &written, NULL, NULL);

	/* write each position */
	g_list_foreach (td->location_fixes, write_gpx_trkpt, fos);

	/* write gpx footer */
	buf = "\n</trkseg></trk>\n</gpx>\n";
	g_output_stream_write_all (G_OUTPUT_STREAM(fos),
			buf, strlen (buf), &written, NULL, NULL);

	g_object_unref (fos);
	g_object_unref (f);
}

/**
 * map_buddy_tracking_has_location_fixes:
 * @self: the #MapBuddyTracking item.
 *
 * Helper function to check if the current track has any
 * recorded locations.
 *
 * Return value: %TRUE when atleast one location has been recorded, %FALSE otherwise.
 */
gboolean map_buddy_tracking_has_location_fixes (MapBuddyTracking *self)
{
	MapBuddyTrackingPrivate *priv;

	g_return_val_if_fail (self != NULL, FALSE);

	priv = MAP_BUDDY_TRACKING_GET_PRIVATE (self);

	if (priv->tracking_details != NULL &&
			priv->tracking_details->location_fixes != NULL)
		return TRUE;
	else
		return FALSE;
}


