/*
 * poi-manager.c - Source for MapBuddyPoiManager
 * Copyright (C) 2010 Collabora Ltd
 * @author Cosimo Alfarano <cosimo.alfarano@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"
#include "poi-manager.h"

#include "poi.h"
#include "poi-osm-fetcher.h"
#include "mapbuddy-marshallers.h"

#include <glib.h>

G_DEFINE_TYPE(MapBuddyPoiManager, poi_manager, G_TYPE_OBJECT)

/* signal enum */
enum
{
    POI_ADDED,
    POI_REMOVED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

typedef struct
{
  /* ((char*) feature) -> g_hash_table (((gint) poi_id) -> (MapBuddyPoi *)) */
  GHashTable *pois;
  /* reverse lookup dictionary: ((gint) poi_id) -> ((char *) feature) */
  GHashTable *ids;
  /* POI fetcher */
  MapBuddyPoiOsmFetcher *fetcher;

} MapBuddyPoiManagerPrivate;

#define POI_MANAGER_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), \
      MAP_BUDDY_POI_MANAGER_TYPE, MapBuddyPoiManagerPrivate))


static void
unref_poi_and_emit_signal (MapBuddyPoi *poi)
{
  MapBuddyPoiManager *self = map_buddy_poi_manager_dup ();
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);

  g_return_if_fail (MAP_BUDDY_IS_POI_MANAGER (self));
  g_return_if_fail (MAP_BUDDY_IS_POI (poi));

  /* update the reverse-lookup dict @priv->ids */
  g_hash_table_remove (priv->ids,
      GINT_TO_POINTER (map_buddy_poi_get_id (poi)));

  /* it's important to emit the signal and THEN unref the object. */
  g_signal_emit (self, signals[POI_REMOVED], 0, poi);

  g_object_unref (poi);
  g_object_unref (self);
}

static void
poi_manager_init (MapBuddyPoiManager *obj)
{
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (obj);

  priv->pois = g_hash_table_new_full (g_str_hash,  g_str_equal,
      (GDestroyNotify) g_free, (GDestroyNotify) g_hash_table_unref);

  priv->ids = g_hash_table_new_full (g_direct_hash,  g_direct_equal,
      NULL, (GDestroyNotify) g_free);
}

static void
poi_manager_dispose (GObject *object)
{
  MapBuddyPoiManager *self = MAP_BUDDY_POI_MANAGER (object);
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);

  if (priv->pois != NULL)
    {
      g_hash_table_unref (priv->pois);
      priv->pois = NULL;
    }

  if (priv->ids != NULL)
    {
      g_hash_table_unref (priv->ids);
      priv->ids = NULL;
    }

  if (priv->fetcher != NULL)
    {
      g_object_unref (priv->fetcher);
      priv->fetcher = NULL;
    }

  if (G_OBJECT_CLASS (poi_manager_parent_class)->dispose)
    G_OBJECT_CLASS (poi_manager_parent_class)->dispose (object);
}

static MapBuddyPoiManager G_GNUC_MAY_ALIAS *manager_singleton = NULL;
static GObject *
poi_manager_constructor (GType type, guint n_props,
    GObjectConstructParam *props)
{
  GObject *retval = NULL;

  if (manager_singleton)
    retval = g_object_ref (manager_singleton);
  else
    {
      retval = G_OBJECT_CLASS (poi_manager_parent_class)->constructor (
          type, n_props, props);
      if (retval == NULL)
        return NULL;

      manager_singleton = MAP_BUDDY_POI_MANAGER (retval);
      g_object_add_weak_pointer (retval, (gpointer *) &manager_singleton);
    }

  return retval;
}

static void
poi_manager_class_init (MapBuddyPoiManagerClass *poi_manager_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (poi_manager_class);

  g_type_class_add_private (poi_manager_class, sizeof (MapBuddyPoiManagerPrivate));

  object_class->dispose = poi_manager_dispose;
  object_class->constructor = poi_manager_constructor;

  signals[POI_ADDED] = g_signal_new ("poi-added",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    mapbuddy_marshal_VOID__OBJECT,
    G_TYPE_NONE, 1, G_TYPE_OBJECT);

  signals[POI_REMOVED] = g_signal_new ("poi-removed",
    G_TYPE_FROM_CLASS (object_class),
    G_SIGNAL_RUN_LAST,
    0, NULL, NULL,
    g_cclosure_marshal_VOID__OBJECT,
    G_TYPE_NONE, 1, G_TYPE_OBJECT);
}

MapBuddyPoiManager *
map_buddy_poi_manager_dup (void)
{
  return g_object_new (MAP_BUDDY_POI_MANAGER_TYPE, NULL);
}

static inline const gchar *
table_lookup_feature_of_id (MapBuddyPoiManager *self,
    gint id)
{
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);

  return g_hash_table_lookup (priv->ids, GINT_TO_POINTER (id));
}

static inline gboolean
table_remove_feature (MapBuddyPoiManager *self,
    const gchar *feature)
{
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);

  /* NOTE: reverse-lookup priv->ids is updated in unref_poi_and_emit_signal()
   * destructor assigned to the second level dict for priv->pois */

  return g_hash_table_remove (priv->pois, feature);
}

/* to either add or update/replace a value. no signal emitted */
static inline void
table_add_poi (MapBuddyPoiManager *self,
    MapBuddyPoi *poi)
{
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);
  GHashTable *featured_ids;

  featured_ids = g_hash_table_lookup (priv->pois, map_buddy_poi_get_feature (poi));
  if (featured_ids == NULL)
    {
      featured_ids = g_hash_table_new_full (g_direct_hash, g_direct_equal,
          NULL, (GDestroyNotify) unref_poi_and_emit_signal);

      g_hash_table_insert (priv->pois,
          g_strdup (map_buddy_poi_get_feature (poi)),
          featured_ids);
    }

  /* update direct and reverse-lookup dict */
  g_hash_table_insert (featured_ids,
      GINT_TO_POINTER (map_buddy_poi_get_id (poi)),
      g_object_ref (poi));

  g_hash_table_insert (priv->ids,
      GINT_TO_POINTER (map_buddy_poi_get_id (poi)),
      g_strdup (map_buddy_poi_get_feature (poi)));
}

static void
on_poi_added (MapBuddyPoiOsmFetcher *fetcher,
    MapBuddyPoi *poi,
    gpointer user_data)
{
  MapBuddyPoiManager *self = MAP_BUDDY_POI_MANAGER (user_data);

  g_return_if_fail (MAP_BUDDY_IS_POI_OSM_FETCHER (fetcher));
  g_return_if_fail (MAP_BUDDY_IS_POI_MANAGER (self));
  g_return_if_fail (MAP_BUDDY_IS_POI (poi));

  /* add or replace @poi */
  table_add_poi (self, poi);
  g_signal_emit (self, signals[POI_ADDED], 0, poi);
}

static void
on_feature_removed (MapBuddyPoiOsmFetcher *fetcher,
    const gchar *feature,
    gpointer user_data)
{
  MapBuddyPoiManager *self = MAP_BUDDY_POI_MANAGER (user_data);

  /* the registered GDestroy function on GHashTable will emit a poi-removed
   * signal for every POI currently registered with @feature */
  table_remove_feature (self, feature);
}

/* FIXME: at some point, when more than one fetcher will be available, it
 * would be a good idea to make it a GInterface */
MapBuddyPoiOsmFetcher *
map_buddy_poi_manager_get_fetcher (MapBuddyPoiManager *self)
{
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);

  g_return_val_if_fail (MAP_BUDDY_IS_POI_MANAGER (self), NULL);

  return priv->fetcher;
}

void
map_buddy_poi_manager_start (MapBuddyPoiManager *self)
{
  MapBuddyPoiManagerPrivate *priv = POI_MANAGER_GET_PRIVATE (self);

  g_return_if_fail (MAP_BUDDY_IS_POI_MANAGER (self));
  g_return_if_fail (priv->fetcher == NULL);

  priv->fetcher = map_buddy_poi_osm_fetcher_new ();

  g_signal_connect (priv->fetcher, "poi-added",
      G_CALLBACK (on_poi_added), self);

  g_signal_connect (priv->fetcher, "feature-removed",
      G_CALLBACK (on_feature_removed), self);

  map_buddy_poi_osm_fetcher_start (priv->fetcher);
}
