/*
 * Copyright (C) 2009 Pierre-Luc Beaudoin <pierre-luc@pierlux.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"

#include "merchant-marker.h"

#include <clutter/clutter.h>
#include <glib.h>
#include <glib-object.h>
#include <cairo.h>
#include <math.h>
#include <string.h>


enum
{
  SHOW_DETAILS,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _MapBuddyMerchantMarkerPrivate
{
  ClutterActor *point;
  ClutterActor *arrow;
  ClutterActor *area;
  ClutterActor *text;
};

G_DEFINE_TYPE (MapBuddyMerchantMarker, map_buddy_merchant_marker, CHAMPLAIN_TYPE_MARKER);

#define MAP_BUDDY_MERCHANT_MARKER_GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_TYPE_MERCHANT_MARKER, MapBuddyMerchantMarkerPrivate))

static void draw_marker (ChamplainMarker *merchant_marker);

static void
map_buddy_merchant_marker_dispose (GObject *object)
{

  MapBuddyMerchantMarker *marker = MAP_BUDDY_MERCHANT_MARKER (object);
  MapBuddyMerchantMarkerPrivate *priv = marker->priv;

  if (priv->text)
    {
      g_object_unref (priv->text);
      priv->text = NULL;
    }

  if (priv->arrow)
    {
      g_object_unref (priv->arrow);
      priv->arrow = NULL;
    }

  G_OBJECT_CLASS (map_buddy_merchant_marker_parent_class)->dispose (object);
}

static void
map_buddy_merchant_marker_class_init (MapBuddyMerchantMarkerClass *map_buddyClass)
{
  g_type_class_add_private (map_buddyClass, sizeof (MapBuddyMerchantMarkerPrivate));

  GObjectClass *object_class = G_OBJECT_CLASS (map_buddyClass);
  object_class->dispose = map_buddy_merchant_marker_dispose;

  ChamplainMarkerClass *markerClass = CHAMPLAIN_MARKER_CLASS (map_buddyClass);
  markerClass->draw_marker = draw_marker;

  signals[SHOW_DETAILS] =
          g_signal_new ("show-details",
                        G_TYPE_FROM_CLASS (map_buddyClass),
                        G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                        0,
                        NULL, NULL,
                        g_cclosure_marshal_VOID__VOID,
                        G_TYPE_NONE, 0,
                        G_TYPE_NONE);
}

#define RADIUS 10
#define PADDING RADIUS
/* M_PI is not defined in C99, compute it if missing */
#ifndef M_PI
#define M_PI acos(-1.0)
#endif

static void
draw_box (cairo_t *cr,
          gint width,
          gint height)
{
  cairo_move_to (cr, RADIUS, 0);
  cairo_line_to (cr, width - RADIUS, 0);
  cairo_arc (cr, width - RADIUS, RADIUS, RADIUS - 1, 3 * M_PI / 2.0, 0);
  cairo_line_to (cr, width, height - RADIUS);
  cairo_arc (cr, width - RADIUS, height - RADIUS, RADIUS - 1, 0, M_PI / 2);
  cairo_line_to (cr, RADIUS, height);
  cairo_arc (cr, RADIUS, height - RADIUS, RADIUS - 1, M_PI / 2.0, M_PI);
  cairo_line_to (cr, 0, RADIUS);
  cairo_arc (cr, RADIUS, RADIUS, RADIUS - 1, M_PI, 3 * M_PI / 2.0);
  cairo_close_path (cr);
}

static void
show_details (ClutterActor *actor,
              ClutterEvent *event,
              MapBuddyMerchantMarker *marker)
{
  g_signal_emit (marker, signals[SHOW_DETAILS], 0, NULL);
}

static void
draw_area (MapBuddyMerchantMarker *merchant_marker)
{
  MapBuddyMerchantMarkerPrivate *priv = merchant_marker->priv;
  ClutterActor *bg = NULL;
  ClutterColor color;
  ClutterColor darker_color;
  gboolean highlighted = FALSE;
  guint line_width = 1;
  cairo_t *cr;
  gfloat width, height;

  if (priv->area != NULL)
    {
      clutter_container_remove_actor (CLUTTER_CONTAINER (merchant_marker),
          priv->area);
      g_object_unref (priv->area);
      priv->area = NULL;
    }

  g_object_get (merchant_marker, "highlighted", &highlighted, NULL);
  if (!highlighted)
    {
      /* Remove all hidden actors because they are still being counted in the clickable area */
      clutter_container_remove_actor (CLUTTER_CONTAINER (merchant_marker),
          priv->text);
      clutter_container_remove_actor (CLUTTER_CONTAINER (merchant_marker),
          priv->arrow);
      return;
    }

  clutter_actor_reparent (priv->text, CLUTTER_ACTOR (merchant_marker));
  clutter_actor_reparent (priv->arrow, CLUTTER_ACTOR (merchant_marker));

  clutter_actor_show (priv->text);
  clutter_actor_show (priv->arrow);

  clutter_actor_get_size (priv->text, &width, NULL);
  height = 20 + 2 * PADDING;
  width += 2 * PADDING + 32 + 24;

  bg = clutter_cairo_texture_new (width, height);
  cr = clutter_cairo_texture_create (CLUTTER_CAIRO_TEXTURE (bg));
  color = *champlain_marker_get_color (CHAMPLAIN_MARKER (merchant_marker));

  clutter_color_darken (&color, &darker_color);
  draw_box (cr, width, height);

  cairo_set_source_rgba (cr,
      color.red / 255.0,
      color.green / 255.0,
      color.blue / 255.0,
      0.75 * color.alpha / 255.0);
  cairo_fill_preserve (cr);

  cairo_set_line_width (cr, line_width);
  cairo_set_source_rgba (cr,
      darker_color.red / 255.0,
      darker_color.green / 255.0,
      darker_color.blue / 255.0,
      darker_color.alpha / 255.0);
  cairo_stroke (cr);
  cairo_destroy (cr);

  clutter_container_add_actor (CLUTTER_CONTAINER (merchant_marker), bg);

  priv->area = g_object_ref (bg);
  clutter_actor_set_position (priv->area, 0, -PADDING);
  clutter_actor_raise (CLUTTER_ACTOR (priv->text), CLUTTER_ACTOR (priv->area));

  clutter_actor_set_reactive (CLUTTER_ACTOR (priv->area), TRUE);
  g_signal_connect (
      G_OBJECT (priv->area),
      "button-press-event",
      G_CALLBACK (show_details),
      merchant_marker);

  if (priv->point)
    clutter_actor_raise (CLUTTER_ACTOR (priv->point), CLUTTER_ACTOR (priv->area));

  if (priv->arrow)
    {
      clutter_actor_set_position (CLUTTER_ACTOR (priv->arrow), width - 24 - PADDING, 0);
      clutter_actor_raise (CLUTTER_ACTOR (priv->arrow), CLUTTER_ACTOR (priv->area));
    }
}

static void
draw_marker (ChamplainMarker *merchant_marker)
{
  MapBuddyMerchantMarkerPrivate *priv = MAP_BUDDY_MERCHANT_MARKER (merchant_marker)->priv;
  gchar *text;

  g_object_get (merchant_marker,
                "text", &text,
                NULL);

  clutter_text_set_markup (CLUTTER_TEXT (priv->text), text);
  g_free (text);

  draw_area (MAP_BUDDY_MERCHANT_MARKER (merchant_marker));

  clutter_actor_set_anchor_point (CLUTTER_ACTOR (merchant_marker), 16, 32);
}

static void
map_buddy_merchant_marker_init (MapBuddyMerchantMarker *merchant_marker)
{
  MapBuddyMerchantMarkerPrivate *priv = MAP_BUDDY_MERCHANT_MARKER_GET_PRIVATE (merchant_marker);
  merchant_marker->priv = priv;
  ClutterColor white = {0xff, 0xff, 0xff, 0xff};

  priv->text = g_object_ref (clutter_text_new ());
  clutter_actor_hide (priv->text);
  clutter_text_set_use_markup (CLUTTER_TEXT (priv->text), TRUE);
  clutter_text_set_font_name (CLUTTER_TEXT (priv->text), "Nokia Sans 14");
  clutter_container_add_actor (CLUTTER_CONTAINER (merchant_marker), priv->text);
  champlain_marker_set_color (CHAMPLAIN_MARKER (merchant_marker), &white);

  clutter_actor_set_position (priv->text, 32, 0);

  priv->point = clutter_texture_new_from_file (PIXMAPDIR "/mapbuddy-marker.png", NULL);
  if (priv->point)
   clutter_container_add_actor (CLUTTER_CONTAINER (merchant_marker), priv->point);

  priv->arrow = g_object_ref (clutter_texture_new_from_file (PIXMAPDIR "/mapbuddy-marker-arrow.png", NULL));
  if (priv->arrow)
   clutter_container_add_actor (CLUTTER_CONTAINER (merchant_marker), priv->arrow);
}

ClutterActor *
map_buddy_merchant_marker_new (void)
{
  return CLUTTER_ACTOR (g_object_new (MAP_BUDDY_TYPE_MERCHANT_MARKER, NULL));
}

