/* 
 * conf.c - GConf manager
 * Copyright (C) 2010 Collabora Ltd
 * @author Cosimo Alfarano <cosimo.alfarano@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTRCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"
#include "conf.h"

#include "mapbuddy-marshallers.h"

#include <glib.h>
#include <dbus/dbus-glib.h>
#include <gconf/gconf-client.h>

#define MB_CONF_KEY_BASE "/apps/mapbuddy"
#define MB_CONF_KEY_ICONS_DEFAULT MB_CONF_KEY_BASE "/icons/DEFAULT"
#define MB_CONF_KEY_OSM_BASE MB_CONF_KEY_BASE "/osm"
#define MB_CONF_KEY_OSM_ICONS_BASE MB_CONF_KEY_OSM_BASE "/icons"
#define MB_CONF_KEY_OSM_SHOWED_FEATURES MB_CONF_KEY_OSM_BASE "/features/show"
G_DEFINE_TYPE (MapBuddyConf, map_buddy_conf, G_TYPE_OBJECT)

#define CONF_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_TYPE_CONF, \
                               MapBuddyConfPrivate))

/* signal enum */
enum
{
    OSM_FEATURE_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

struct _MapBuddyConfPrivate {

    GConfClient *client;
    GSList *cached_shown_features;
};


static void
map_buddy_conf_dispose (GObject *obj)
{
  MapBuddyConfPrivate *priv;

  priv = CONF_GET_PRIVATE (obj);

  if (priv->client != NULL)
    {
      g_object_unref (priv->client);
      priv->client = NULL;
    }

  if (priv->cached_shown_features != NULL)
    {
      g_slist_foreach (priv->cached_shown_features, (GFunc) g_free, NULL);
      g_slist_free (priv->cached_shown_features);
      priv->cached_shown_features = NULL;
    }

  G_OBJECT_CLASS (map_buddy_conf_parent_class)->dispose (obj);
}

static MapBuddyConf G_GNUC_MAY_ALIAS *conf_singleton = NULL;

static GObject *
map_buddy_conf_constructor (GType type,
    guint n_props,
    GObjectConstructParam *props)
{
  GObject *retval;

  if (conf_singleton)
    retval = g_object_ref (conf_singleton);
  else
    {
      retval = G_OBJECT_CLASS (map_buddy_conf_parent_class)->constructor (type,
          n_props, props);
      if (retval == NULL)
        return NULL;

      conf_singleton = MAP_BUDDY_CONF (retval);
      g_object_add_weak_pointer (retval, (gpointer *) &conf_singleton);
    }
  return retval;
}

static void
map_buddy_conf_class_init (MapBuddyConfClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = map_buddy_conf_dispose;
  object_class->constructor = map_buddy_conf_constructor;

  g_type_class_add_private (object_class, sizeof (MapBuddyConfPrivate));

  signals[OSM_FEATURE_CHANGED] = g_signal_new (
      "shown-features-changed",
      G_TYPE_FROM_CLASS (object_class),
      G_SIGNAL_RUN_LAST,
      0, NULL, NULL,
      mapbuddy_marshal_VOID__POINTER_POINTER,
      G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER);
}

/** _get_shown_features:
 *
 * Internal use: fetch the current value for shown features from GConf.
 * See map_buddy_conf_get_shown_features for its public counter part.
 *
 * Return a list of feature, to be freed (list and its items) when not needed
 * anymore.
 */
static GSList *
_get_shown_features (MapBuddyConf *self,
    GError **error)
{
  MapBuddyConfPrivate *priv = CONF_GET_PRIVATE (self);
  GSList *ret = NULL;
  GError *loc_error = NULL;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);
  g_return_val_if_fail (MAP_BUDDY_IS_CONF (self), NULL);
  g_return_val_if_fail (priv->client != NULL, NULL);

  ret = gconf_client_get_list (priv->client, MB_CONF_KEY_OSM_SHOWED_FEATURES,
      GCONF_VALUE_STRING, &loc_error);
  if (loc_error != NULL)
    g_propagate_error (error, loc_error);

  return ret;
}


/* On /apps/mapbuddy/osm/features/show key changes */
static void
gconf_shown_features_update_cb (GConfClient *client,
    guint cnxn_id,
    GConfEntry *entry,
    gpointer user_data)
{
  MapBuddyConf *self = MAP_BUDDY_CONF (user_data);
  MapBuddyConfPrivate *priv = CONF_GET_PRIVATE (self);
  GSList *added = NULL, *removed = NULL;
  GSList *old_features = NULL;
  GSList *new_features, *iter;
  GError *error = NULL;

  g_return_if_fail (MAP_BUDDY_IS_CONF (self));

  /* fetch the new value */
  new_features = _get_shown_features (self, &error);
  if (error != NULL)
    {
      g_debug ("%s: Unable to obtain current disabled features list: %s",
          G_STRFUNC, error->message);
      g_error_free (error);

      goto out;
    }

  /* save the previous state */
  old_features = priv->cached_shown_features;
  /* update cached_shown_features ASAP to avoid races */
  priv->cached_shown_features = new_features;

  /* Hunt for removed items */
  for (iter = old_features;
      iter != NULL;
      iter = iter->next)
    {
      gchar *feature = iter->data;

      /* something in the former status is not present anymore in the new
       * one: add to removed items */
      if (g_slist_find_custom (new_features, feature,
            (GCompareFunc) g_strcmp0) == NULL)
        {
          g_debug ("%s: REMOVED feature %s", G_STRFUNC, feature);
          removed = g_slist_prepend (removed, feature);
        }
    }

  /* Hunt for added items */
  for (iter = new_features;
      iter != NULL;
      iter = iter->next)
    {
      gchar *feature = iter->data;

      /* something in the new status is not present in the cached one:
       * add to added items */
      if (g_slist_find_custom (old_features, feature,
            (GCompareFunc) g_strcmp0) == NULL)
        {
          g_debug ("%s ADDED feature %s", G_STRFUNC, feature);
          added = g_slist_prepend (added, feature);
        }
    }

  if (added != NULL || removed != NULL)
    g_signal_emit_by_name (self, "shown-features-changed", added, removed);

out:
  /* their items are owned by new_features */
  g_slist_free (added);
  g_slist_free (removed);

  if (old_features != NULL)
    {
      g_slist_foreach (old_features, (GFunc) g_free, NULL);
      g_slist_free (old_features);
    }

  /* do not free new_features, pointed list onwership is passed to
   * priv->cached_shown_features */
}


static void
map_buddy_conf_init (MapBuddyConf *self)
{
  MapBuddyConfPrivate *priv = CONF_GET_PRIVATE (self);
  GError *error = NULL;

  priv->client = gconf_client_get_default ();

  priv->cached_shown_features = _get_shown_features (self,
      &error);
  if (error != NULL)
    {
      g_critical ("%s: Unable to fetch shown features: %s",
          G_STRFUNC, error->message);
      goto out;
    }

  gconf_client_add_dir (priv->client, MB_CONF_KEY_OSM_SHOWED_FEATURES,
      GCONF_CLIENT_PRELOAD_ONELEVEL, &error);
  if (error != NULL)
    {
      g_critical ("%s: Unable to add dir for GConf key %s: %s",
          G_STRFUNC,
          MB_CONF_KEY_OSM_SHOWED_FEATURES, error->message);
      goto out;
    }

  gconf_client_notify_add (priv->client, MB_CONF_KEY_OSM_SHOWED_FEATURES,
      gconf_shown_features_update_cb, self, NULL, &error);
  if (error != NULL)
    {
      g_critical ("%s: Unable to hook signal for GConf key %s: %s",
          G_STRFUNC,
          MB_CONF_KEY_OSM_SHOWED_FEATURES, error->message);
      goto out;
    }

out:
  if (error != NULL)
    g_error_free (error);
}

MapBuddyConf *
map_buddy_conf_dup (void)
{
  return g_object_new (MAP_BUDDY_TYPE_CONF, NULL);
}

/** map_buddy_conf_get_osm_icon_for:
 * @self: a MapBuddyConf instance
 * @feature: the feature (string) requested
 * @feature_type: the feature type (string) requested, can be %NULL.
 *
 * It will first look for a icon path associated with the couple
 * (@feature,@feature_type), if not found it will look for a generic icon for
 * @feature, falling back to the result of #map_buddy_conf_get_default_icon()
 * if nothing is found.
 *
 * When called with @feature_type = %NULL, an generic icon for @feature will
 * be looked for.
 *
 * The icon file path can be open, for example, with
 * #clutter_texture_new_from_file()
 *
 * Return the file path of the icon, to be free when not needed anymore.
 */
gchar *
map_buddy_conf_get_osm_icon_for (MapBuddyConf *self,
    const gchar *feature,
    const gchar *feature_type,
    GError **error)
{
  MapBuddyConfPrivate *priv = CONF_GET_PRIVATE (self);
  GConfValue *value = NULL;
  GError *loc_error = NULL;
  gchar *key_esc = NULL;
  gchar *ret = NULL;
  gchar *key = NULL;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);
  if (!MAP_BUDDY_IS_CONF (self))
    {
      g_set_error (error, MAP_BUDDY_CONF_ERROR, MAP_BUDDY_CONF_ERROR_GCONF_KEY,
          "arg1 passed to %s is not MapBuddyConf instance", G_STRFUNC);

      goto out;
    }

  /* compose the gconf key, according to the passed parameters */
  if (feature_type != NULL)
    key = g_strdup_printf ("%s/%s/%s", MB_CONF_KEY_OSM_ICONS_BASE, feature,
        feature_type);
  else
    key = g_strdup_printf ("%s/%s", MB_CONF_KEY_OSM_ICONS_BASE, feature);

  /* escape key, in case feature or feature_type contains spaces or similar
   * weird chars. Assume UTF-8 (FIXME is it right?) */
  key_esc = g_uri_escape_string (key, "/", TRUE);

  value = gconf_client_get (priv->client, key_esc, &loc_error);
  if (loc_error != NULL)
    {
      g_propagate_error (error, loc_error);

      goto out;
    }

  if (value == NULL)
    {
      /* fall back result, depending on the passed parameters */
      if (feature_type == NULL)
        ret = map_buddy_conf_get_default_icon (self, &loc_error);
      else
        ret = map_buddy_conf_get_osm_icon_for (self, feature, NULL, &loc_error);

      if (loc_error != NULL)
        g_propagate_error (error, loc_error);

      goto out;
    }

  ret = g_strdup (gconf_value_get_string (value));

out:
  g_free (key);
  g_free (key_esc);

  if (value != NULL)
    gconf_value_free (value);

  return ret;
}

gchar *
map_buddy_conf_get_default_icon (MapBuddyConf *self,
    GError **error)
{
  MapBuddyConfPrivate *priv = CONF_GET_PRIVATE (self);
  GConfValue *value = NULL;
  GError *loc_error = NULL;
  gchar *ret = NULL;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);
  if (!MAP_BUDDY_IS_CONF (self))
    {
      g_set_error (error, MAP_BUDDY_CONF_ERROR,
          MAP_BUDDY_CONF_ERROR_GCONF_KEY,
          "arg1 passed to %s is not MapBuddyConf instance", G_STRFUNC);

      goto out;
    }

  value = gconf_client_get (priv->client, MB_CONF_KEY_ICONS_DEFAULT,
      &loc_error);
  if (loc_error != NULL)
    {
      g_propagate_error (error, loc_error);

      goto out;
    }

  if (value == NULL)
    {
      /* this should not be ever hit, since an error should be cought at
       * gconf_client_get() time. if you see this error, something very weird
       * is happending :) */
      g_set_error (error, MAP_BUDDY_CONF_ERROR,
          MAP_BUDDY_CONF_ERROR_GCONF_KEY,
          "not possible to retrieve the default icon for POIs - gconf key %s",
          MB_CONF_KEY_ICONS_DEFAULT);

      goto out;
    }

  ret = g_strdup (gconf_value_get_string (value));

out:
  if (value != NULL)
    gconf_value_free (value);

  return ret;
}

/** map_buddy_conf_get_shown_features::
 * @self: a MapBuddyConf instance
 * @error: address used for error reporting
 *
 * Return the cached value for the shown features, or %NULL and @error set in
 * case of error.
 * This value is valid until the configuration is changed and cannot be freed
 * The caller should make a copy of the list and its data if needed.
 */
const GSList *
map_buddy_conf_get_shown_features (MapBuddyConf *self,
    GError **error)
{
  MapBuddyConfPrivate *priv;
  GSList *ret = NULL;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);
  if (!MAP_BUDDY_IS_CONF (self))
    {
      g_set_error (error, MAP_BUDDY_CONF_ERROR,
          MAP_BUDDY_CONF_ERROR_GCONF_KEY,
          "arg1 passed to %s is not MapBuddyConf instance", G_STRFUNC);

      goto out;
    }

  priv = CONF_GET_PRIVATE (self);
  ret = priv->cached_shown_features;

out:
  return ret;
}
