#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <glib.h>
#include <gconf/gconf-client.h>

#include <dbus/dbus-glib.h>

#define QTM_MODEST_PLUGIN_SERVICE "com.nokia.Qtm.Modest.Plugin"
#define QTM_MODEST_PLUGIN_PATH "/com/nokia/Qtm/Modest/Plugin"
#define QTM_MODEST_PLUGIN_INTERFACE "com.nokia.Qtm.Modest.Plugin"
#define QTM_MODEST_PLUGIN_SEND_EMAIL "SendEmail"
#define MODESTENGINE_DEFAULT_ACCOUNT "/apps/modest/default_account"
#define MODESTENGINE_ACCOUNT_EMAIL "/apps/modest/accounts/%s/email"
void print_usage_and_exit(const char* arg_name)
{
  printf("usage: %s [OPTION...] address \n", arg_name);
  printf("\t-h, --help \t print this help\n");
  printf("\t-s, --subject=SUBJ \t use this subject\n");
  printf("\t-a, --attachement=FILE \t a file\n");
  exit(0);
}

gchar* get_default_mail_account()
{
  GError* error = NULL;
  gchar* default_email = NULL;
  GConfClient* gconfclient = gconf_client_get_default();
  gchar* default_account_id = gconf_client_get_string(gconfclient, MODESTENGINE_DEFAULT_ACCOUNT, &error);
  if(error)
  {
    printf("can not find default account\n");
    g_error_free(error);
  }
  else 
  {
    gchar* account_key = gconf_escape_key(default_account_id, -1);
    gchar* default_email_key = g_strdup_printf(MODESTENGINE_ACCOUNT_EMAIL, account_key);
    g_free(account_key);
    default_email = gconf_client_get_string(gconfclient, default_email_key, &error);
    if(error || default_email == NULL)
    { 
      fprintf(stderr, "can not get default email account (%s)\n", error->message);
      g_error_free(error);
      return NULL;
    }
    g_free(default_email_key);
    g_free(default_account_id);
  }
  g_object_unref(gconfclient);
  return default_email;
}

void send_mail(const gchar* sender, GString* receiver, const gchar* subject, GString* text, const gchar* afile)
{
  DBusGConnection* bus;
  DBusGProxy* proxy;
  GError* error = NULL;
  bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
  proxy = dbus_g_proxy_new_for_name(bus,
				    QTM_MODEST_PLUGIN_SERVICE,
				    QTM_MODEST_PLUGIN_PATH,
				    QTM_MODEST_PLUGIN_INTERFACE);

  GHashTable* sender_info = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
  g_hash_table_insert(sender_info, "from", g_strdup(sender));

  GHashTable* receiver_info = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
  g_hash_table_insert(receiver_info, "to", g_strdup(receiver->str));

  GHashTable* text_info = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
  g_hash_table_insert(text_info,"plain-body", g_strdup(text->str));

  if(subject != NULL)
  {
    g_hash_table_insert(text_info,"subject", g_strdup(subject));
  }

  GPtrArray* dummy1 = g_ptr_array_new();
  GPtrArray* dummy2 = g_ptr_array_new();
  GHashTable* dummy3 = g_hash_table_new(NULL, NULL);
  GHashTable* attachements = g_hash_table_new(NULL, NULL);
  if(afile != NULL)
    g_hash_table_insert(attachements, "filename", g_strdup(afile));
  g_ptr_array_add(dummy1, (gpointer)attachements);
  dbus_g_proxy_call_no_reply(proxy, 
			     QTM_MODEST_PLUGIN_SEND_EMAIL,
			     dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), sender_info,
			     dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), receiver_info,
			     dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), text_info,
			     dbus_g_type_get_collection("GPtrArray",
							(dbus_g_type_get_map("GHashTable", 
									     G_TYPE_STRING, 
									     G_TYPE_STRING))), dummy1,
			     dbus_g_type_get_collection("GPtrArray",
							(dbus_g_type_get_map("GHashTable", 
									     G_TYPE_STRING, 
									     G_TYPE_STRING))), dummy2,
			     G_TYPE_UINT, 0,
			     dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), dummy3,
			     G_TYPE_INVALID,
			     G_TYPE_INVALID);
  g_object_unref(proxy);
}

GString* read_message_input()
{
  char* lines = NULL;
  size_t len = 0;
  ssize_t num_char = getline(&lines, &len, stdin);
  GString* text = g_string_new(NULL);
  while(num_char > 0)
  {
    if(num_char>0)
    {
      text = g_string_append(text, lines);
      free(lines);
      lines = NULL;
    }
    num_char = getline(&lines, &len, stdin);
  }
  return text;
}

int main(int argc, char** argv)
{
  g_type_init();
  int c;
  static struct option long_options[] = {
    {"subject", 1, 0, 's'},
    {"attachement", 0, 0, 'a'},
    {"help", 0, 0, 'h'},
    {0, 0, 0, 0}
  };
  int option_index = 0;

  c = getopt_long(argc, argv, "a:s:h", long_options, &option_index);
  gchar* subject = NULL;
  gchar* afile = NULL;
  while(c != -1)
  {
    if(c == 0)
    {
      if(long_options[option_index].val == 'h')
      {
	print_usage_and_exit(argv[0]);
      }
      if(long_options[option_index].val == 's')
      {
	c = long_options[option_index].val;
      }
      if(long_options[option_index].val == 'a')
      {
	c = long_options[option_index].val;
      }
    }
    if(c != 0)
    {
      if(c == 's')
      {
	if(optarg)
	{
	  subject = g_strdup(optarg);
	}
      }
      if(c == 'a')
      {
	if(optarg)
	{
	  afile = g_strdup(optarg);
	}
      }
      else if( c == 'h')
      {
	print_usage_and_exit(argv[0]);
      }
    }
    c = getopt_long(argc, argv,"a:s:h", long_options, &option_index);
  }

  GString* receiver = g_string_new(NULL);
  if(optind<argc)
  {
    receiver = g_string_append(receiver, argv[optind++]);
  }
  for(;optind<argc;optind++)
  {
    receiver = g_string_append(receiver, ";");
    receiver = g_string_append(receiver, argv[optind]);
  }
  gchar* default_email = get_default_mail_account();

  GString* text = read_message_input();
  if(default_email)
  {
    send_mail(default_email, receiver, subject, text, afile);
    g_string_free(receiver, TRUE);
    g_string_free(text, TRUE);
    g_free(default_email);
    g_free(subject);
  }
  else
  {
    fprintf(stderr, "error no email\n");
  }
  return 0;
}
