# -*- coding: utf-8 -*-

###########################################################################
#    Maevies
#    Copyright (C) 2010 Simón Pena <spenap@gmail.com>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
###########################################################################

from dbus import glib
from dbus.exceptions import DBusException
from xml.sax import saxutils
import dbus
import gobject
import gtk

from lib import constants

gobject.threads_init()
glib.init_threads()

MINFO_PROVIDER_INTERFACE = 'com.simonpena.butaca.minfoprovider'
MINFO_PROVIDER_BUS_NAME = 'com.simonpena.butaca.minfoprovider'
TMDB_MOVIE_INTERFACE = 'com.simonpena.butaca.tmdbmovie'
TMDB_MOVIE_BUS_NAME = 'com.simonpena.butaca.tmdbmovie'
WATC_MOVIE_INTERFACE = 'com.simonpena.butaca.watcmovie'
WATC_MOVIE_BUS_NAME = 'com.simonpena.butaca.watcmovie'
TMDB = 0
WATC = 1

class MovieManager:

    _favorites = []

    def __init__(self, response_received_cb=None):
        self.response_received_cb = response_received_cb
        self._provider = None
        self._bus = dbus.SessionBus()
        self.load_favorites()

    def _get_provider(self):
        if not self._provider:
            self._provider = self._create_minfo_provider()
        return self._provider

    def _create_minfo_provider(self):
        provider_proxy = self._bus.get_object(MINFO_PROVIDER_BUS_NAME,
                                             '/MInfoProvider')
        provider_interface = dbus.Interface(provider_proxy,
                                            dbus_interface=MINFO_PROVIDER_INTERFACE)
        provider_interface.connect_to_signal('ResponseReceived',
                                             self._on_response_received)
        return provider_interface

    def query(self, movie_name='', query_type=TMDB):
        try:
            self._get_provider().Query(query_type, movie_name)
        except DBusException, ex:
            self._on_response_received(None, [], ex)

    def destroy(self):
        try:
            self._get_provider().Exit()
        except DBusException:
            pass

    def _on_response_received(self, interface, object_paths, error=None):
        movies = []

        if interface == TMDB_MOVIE_INTERFACE:
            for path in object_paths:
                movies.append(TmdbMovie.from_dbus(self._bus, path))
        elif interface == WATC_MOVIE_INTERFACE:
            for path in object_paths:
                movies.append(WatcMovie.from_dbus(self._bus, path))

        if self.response_received_cb:
            self.response_received_cb(movies, error)

    def update_favorite(self, movie, favorite):
        movie.set_favorite(favorite)
        if favorite:
            self._favorites.append(movie)
        else:
            self._favorites.remove(movie)
        self.save_favorites()

    def save_favorites(self):
        try:
            f = open(constants.MVS_FAVORITES_FILE, 'w')
            try:
                for favorite in self._favorites:
                    f.write(repr(favorite) + '\n')
            except:
                f.close()
        except:
            pass

    def load_favorites(self):
        self._favorites = []
        try:
            f = open(constants.MVS_FAVORITES_FILE, 'r')
            try:
                for line in f:
                    favorite = eval(line)
                    self._favorites.append(favorite)
            except:
                f.close()
        except:
            pass

class TmdbMovieImage:

    def __init__(self, image_id, image_url, image_type, image_size):
        self._id = image_id
        self._url = image_url
        self._type = image_type
        self._size = image_size

    @classmethod
    def from_image_struct(cls, image_struct):
        type, url, size, id = image_struct
        return cls(image_id=id, image_url=url,
                   image_type=type, image_size=size)

    def get_url(self):
        return self._url

    def get_id(self):
        return self._id

    def get_size(self):
        return self._size

    def get_type(self):
        return self._type

    def __str__(self):
        str = ''
        for field in [self._id, self._url, self._type, self._size]:
            str += '\t%s\n' % field
        return str

    def __repr__(self):
        return ('TmdbMovieImage(image_id=%(id)r,'
                ' image_url=%(url)r,'
                ' image_type=%(type)r,'
                ' image_size=%(size)r)' %
                {'id' : self._id,
                 'url' : self._url,
                 'type' : self._type,
                 'size' : self._size})

class GenericMovie(object):

    def __init__(self, dbus_interface=None):
        self._interface = dbus_interface

    @classmethod
    def from_dbus(cls, bus, object_path):
        return cls()

    @classmethod
    def interface_from_dbus(cls, bus, bus_name, bus_interface, object_path):
        proxy = bus.get_object(bus_name, object_path)
        interface = dbus.Interface(proxy,
                                   dbus_interface=bus_interface)
        return interface

    def get_title(self):
        return ''

    def get_info(self):
        return ''

    def get_value(self, field):
        return ''

    def get_image(self, type=None, size=None):
        return self.get_placeholder_image()

    def get_placeholder_image(self):
        return self._get_placeholder_pixbuf()

    def _get_placeholder_pixbuf(self):
        pixbuf = gtk.IconTheme().load_icon('general_video_file', 48, 0)
        return pixbuf

class WatcMovie(GenericMovie):

    def __init__(self, dbus_interface=None):
        super(WatcMovie, self).__init__(dbus_interface)
        self.fields = ['Name', 'Year', 'Stingers']

    @classmethod
    def from_dbus(cls, bus, path):
        interface = GenericMovie.interface_from_dbus(bus,
                                                     WATC_MOVIE_BUS_NAME,
                                                     WATC_MOVIE_INTERFACE,
                                                     path)
        return cls(dbus_interface=interface)

    def get_value(self, field):
        if field == 'Name':
            return self.get_title()
        elif field == 'Year':
            return self.get_year()
        elif field == 'Stingers':
            return self.get_stingers()
        super(WatcMovie, self).get_value()

    def get_title(self):
        return self._interface.GetName()

    def get_year(self):
        return self._interface.GetYear()

    def get_stingers(self):
        return self._interface.GetStingers()

    def get_info(self):
        stingers = self.get_stingers()
        has_stingers = ''
        if stingers == 0:
            has_stingers = 'Unknown'
        elif stingers == 1:
            has_stingers = 'Yes'
        elif stingers == 2:
            has_stingers = 'No'

        return ('<b>%(title)s</b>\n<small><i>Year: </i>%(year)s'
                '    <i>Has stingers: </i>%(stingers)s</small>' %
                {'title' : saxutils.escape(self.get_title()),
                 'year' : saxutils.escape(self.get_year()),
                 'stingers' : saxutils.escape(has_stingers)})

class TmdbMovie(GenericMovie):

    FIELD_TITLE = 'Title'
    FIELD_RELEASE = 'Release date'
    FIELD_RATING = 'Rating'
    FIELD_POPULARITY = 'Popularity'

    def __init__(self, dbus_interface=None,
                 images={},
                 title='',
                 release_date='',
                 rating=None,
                 popularity=None,
                 overview=None,
                 favorite=False):
        super(TmdbMovie, self).__init__(dbus_interface)
        self.fields = [self.FIELD_TITLE,
                       self.FIELD_RELEASE,
                       self.FIELD_RATING,
                       self.FIELD_POPULARITY]
        self._data = {}
        if dbus_interface:
            self._images = self._retrieve_images()
            self._data[self.FIELD_TITLE] = self._interface.GetTitle()
            self._data[self.FIELD_RELEASE] = self._interface.GetReleased()
            self._data[self.FIELD_RATING] = self._interface.GetRating()
            self._data[self.FIELD_POPULARITY] = self._interface.GetPopularity()
            self._overview = self._interface.GetOverview()
        else:
            self._images = images
            self._data[self.FIELD_TITLE] = title
            self._data[self.FIELD_RELEASE] = release_date
            self._data[self.FIELD_RATING] = rating
            self._data[self.FIELD_POPULARITY] = popularity
            self._overview = overview
        self._favorite = favorite

    @classmethod
    def from_dbus(cls, bus, path):
        interface = cls.interface_from_dbus(bus,
                                            TMDB_MOVIE_BUS_NAME,
                                            TMDB_MOVIE_INTERFACE,
                                            path)
        return cls(dbus_interface=interface)

    def _retrieve_images(self):
        images = {}
        dbus_images = self._interface.GetImages()
        for dbus_image in dbus_images:
            image = TmdbMovieImage.from_image_struct(dbus_image)
            if not images.has_key(image.get_type()):
                images[image.get_type()] = {}
            images[image.get_type()][image.get_size()] = image
        return images

    def get_value(self, field):
        if self._data.has_key(field):
            return self._data[field]
        super(TmdbMovie, self).get_value(field)

    def get_image(self, type, size):
        if self._images.has_key(type) and self._images[type].has_key(size):
            return self._images[type][size]
        else:
            return None

    def get_info(self):
        return ('<b>%(title)s</b>\n<small><i>Popularity: </i>%(popularity)s'
                '    <i>Rating: </i>%(rating)s    <i>Released: </i>'
                '%(released)s</small>' %
                {'title' : self._data[self.FIELD_TITLE],
                 'popularity' : self._data[self.FIELD_POPULARITY],
                 'rating' : self._data[self.FIELD_RATING],
                 'released' : self._data[self.FIELD_RELEASE]})

    def get_overview(self):
        return self._overview

    def get_favorite(self):
        return self._favorite

    def set_favorite(self, favorite):
        self._favorite = favorite

    def __repr__(self):
        return ('TmdbMovie(images=%(images)r,'
                ' title=%(title)r,'
                ' release_date=%(release)r,'
                ' rating=%(rating)r,'
                ' popularity=%(popularity)r,'
                ' overview=%(overview)r,'
                ' favorite=%(favorite)r)' %
                {'images' : self._images,
                 'title' : self._data[self.FIELD_TITLE],
                 'release' : self._data[self.FIELD_RELEASE],
                 'rating' : self._data[self.FIELD_RATING],
                 'popularity' : self._data[self.FIELD_POPULARITY],
                 'overview' : self._overview,
                 'favorite' : self._favorite})
