# -*- coding: utf-8 -*-

###########################################################################
#    Maevies
#    Copyright (C) 2010 Simón Pena <spenap@gmail.com>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
###########################################################################

from dbus import glib
from dbus.exceptions import DBusException
from xml.sax import saxutils
import dbus
import gobject
import gtk

gobject.threads_init()
glib.init_threads()

MINFO_PROVIDER_INTERFACE = 'com.simonpena.butaca.minfoprovider'
MINFO_PROVIDER_BUS_NAME = 'com.simonpena.butaca.minfoprovider'
TMDB_MOVIE_INTERFACE = 'com.simonpena.butaca.tmdbmovie'
TMDB_MOVIE_BUS_NAME = 'com.simonpena.butaca.tmdbmovie'
WATC_MOVIE_INTERFACE = 'com.simonpena.butaca.watcmovie'
WATC_MOVIE_BUS_NAME = 'com.simonpena.butaca.watcmovie'
TMDB = 0
WATC = 1

class MovieManager:

    def __init__(self, response_received_cb=None):
        self.response_received_cb = response_received_cb
        self._bus = None
        self._provider = None

        self._bus = dbus.SessionBus()

    def _get_provider(self):
        if not self._provider:
            self._provider = self._create_minfo_provider()
        return self._provider

    def _create_minfo_provider(self):
        provider_proxy = self._bus.get_object(MINFO_PROVIDER_BUS_NAME,
                                             '/MInfoProvider')
        provider_interface = dbus.Interface(provider_proxy,
                                            dbus_interface=MINFO_PROVIDER_INTERFACE)
        provider_interface.connect_to_signal('ResponseReceived',
                                             self._on_response_received)
        return provider_interface

    def query(self, movie_name='', query_type=TMDB):
        try:
            self._get_provider().Query(query_type, movie_name)
        except DBusException, ex:
            self._on_response_received(None, [], ex)

    def destroy(self):
        try:
            self._get_provider().Exit()
        except DBusException:
            pass

    def _on_response_received(self, interface, object_paths, error=None):
        movies = []

        if interface == TMDB_MOVIE_INTERFACE:
            for path in object_paths:
                movies.append(TmdbMovie(self._bus, path))
        elif interface == WATC_MOVIE_INTERFACE:
            for path in object_paths:
                movies.append(WatcMovie(self._bus, path))

        if self.response_received_cb:
            self.response_received_cb(movies, error)

class TmdbMovieImage:
    def __init__(self, image_struct):
        self._type, self._url, self._size, self._id = image_struct

    def get_url(self):
        return self._url

    def get_id(self):
        return self._id

    def get_size(self):
        return self._size

    def get_type(self):
        return self._type

    def __str__(self):
        str = ''
        for field in [self._id, self._url, self._type, self._size]:
            str += '\t%s\n' % field
        return str

class GenericMovie(object):

    def __init__(self, bus):
        self._bus = bus
        pass

    def _create_interface(self, bus_name, bus_interface, object_path):
        proxy = self._bus.get_object(bus_name,
                                     object_path)
        interface = dbus.Interface(proxy,
                                   dbus_interface=bus_interface)
        return interface

    def get_title(self):
        return ''

    def get_info(self):
        return ''

    def get_value(self, field):
        return ''

    def get_image(self, type=None, size=None):
        return self.get_placeholder_image()

    def get_placeholder_image(self):
        return self._get_placeholder_pixbuf()

    def _get_placeholder_pixbuf(self):
        pixbuf = gtk.IconTheme().load_icon('general_video_file', 48, 0)
        return pixbuf

class WatcMovie(GenericMovie):

    def __init__(self, bus, object_path):
        super(WatcMovie, self).__init__(bus)
        self.interface = self._create_interface(WATC_MOVIE_BUS_NAME,
                                                WATC_MOVIE_INTERFACE,
                                                object_path)
        self.fields = ['Name', 'Year', 'Stingers']

    def get_value(self, field):
        if field == 'Name':
            return self.get_title()
        elif field == 'Year':
            return self.get_year()
        elif field == 'Stingers':
            return self.get_stingers()
        super(WatcMovie, self).get_value()

    def get_title(self):
        return self.interface.GetName()

    def get_year(self):
        return self.interface.GetYear()

    def get_stingers(self):
        return self.interface.GetStingers()

    def get_info(self):
        stingers = self.get_stingers()
        has_stingers = ''
        if stingers == 0:
            has_stingers = 'Unknown'
        elif stingers == 1:
            has_stingers = 'Yes'
        elif stingers == 2:
            has_stingers = 'No'

        return ('<b>%(title)s</b>\n<small><i>Year: </i>%(year)s'
                '    <i>Has stingers: </i>%(stingers)s</small>' %
                {'title' : saxutils.escape(self.get_title()),
                 'year' : saxutils.escape(self.get_year()),
                 'stingers' : saxutils.escape(has_stingers)})

class TmdbMovie(GenericMovie):

    def __init__(self, bus, object_path):
        super(TmdbMovie, self).__init__(bus)
        self.interface = self._create_interface(TMDB_MOVIE_BUS_NAME,
                                                TMDB_MOVIE_INTERFACE,
                                                object_path)
        self.fields = ['Title', 'Release date', 'Rating', 'Popularity']
        self._images = self._retrieve_images()

    def _retrieve_images(self):
        images = { }
        dbus_images = self.interface.GetImages()
        for dbus_image in dbus_images:
            image = TmdbMovieImage(dbus_image)
            if not images.has_key(image.get_type()):
                images[image.get_type()] = {}
            images[image.get_type()][image.get_size()] = image
        return images

    def get_value(self, field):
        if field == 'Title':
            return self.get_title()
        elif field == 'Release date':
            return self.get_released()
        elif field == 'Rating':
            return self.get_rating()
        elif field == 'Popularity':
            return self.get_popularity()
        super(TmdbMovie, self).get_value()

    def get_title(self):
        return self.interface.GetTitle()

    def get_popularity(self):
        return self.interface.GetPopularity()

    def get_rating(self):
        return self.interface.GetRating()

    def get_image(self, type, size):
        if self._images.has_key(type) and self._images[type].has_key(size):
            return self._images[type][size]
        else:
            return None

    def get_released(self):
        return self.interface.GetReleased()

    def get_info(self):
        return ('<b>%(title)s</b>\n<small><i>Popularity: </i>%(popularity)s'
                '    <i>Rating: </i>%(rating)s    <i>Released: </i>'
                '%(released)s</small>' %
                {'title' : self.get_title(),
                 'popularity' : self.get_popularity(),
                 'rating' : self.get_rating(),
                 'released' : self.get_released()})

    def get_overview(self):
        return self.interface.GetOverview()
