/*
* This file is part of MaemoSweeper
*
* Copyright (C) 2005 INdT - Instituto Nokia de Tecnologia
* http://www.indt.org/maemo
*
* This software is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA
*
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <SDL/SDL.h>

#include "images.h"
#include "globals.h"

#ifdef MAEMO
#if HGW_FUNC
#include <hgw/hgw.h>
    HgwContext *hgw_context = NULL;
#endif
#endif

GameState gState;
SDL_Surface *screen;
int clickType;
int exchanged;
int marked;
int difficulty;
int difficulty_offset;

void new_game(void);
void click_revealed(int indexX, int indexY);
void game_loop(void);

//hildon-games-wrapper functions
/**
 * Pauses the game
 */
static int exit_callback(int errcode);
/**
 * Quit the game
 */
static int quit_callback(int errcode);
/**
 * Cleans saved game
 */
static int flush_callback(int errcode);

/******************************************************************
 * Starts a new game, according the difficulty selected by the user
 ******************************************************************/
void new_game(void){

    clickType = loadstatesaved(&marked, &difficulty);
    gState = STARTEDGAME_STATE;

    if (difficulty == EASY) {
        difficulty_offset = DIFFICULTY_REDIMENSION_EASY;
    }
    else if (difficulty == NORMAL) {
        difficulty_offset = DIFFICULTY_REDIMENSION_NORMAL;
    }
    else {
        difficulty_offset = DIFFICULTY_REDIMENSION_HARD;
    }

    if (clickType != CLICK_MARK && clickType != CLICK_SEARCH) {
        marked = 0;
        initialize_grid();
        //create a minefield with  bombs
        spread_mines(BOMBS_NUMBER, difficulty_offset);
        fill_minenumber();
        clickType = CLICK_SEARCH;
    }
}

/**
 * Shows the user adjacents cells of the one clicked.
 * If the clicked cell is a number, and the user already marked
 * that number of mines around the clicked cell,
 * it reveals the contents of the othes cells, if there is a bomb among them,
 * the user looses 
 */
void click_revealed(int indexX, int indexY){
    int revealed_number=0;
    int count = 0;
    if (minefield[indexX+indexY*COLS] > 0) {
        count = count_markedcells(indexX, indexY);
        if (count >= (int)minefield[indexX+indexY*COLS]) {
            show_cells_around(indexX, indexY);
        }
        else {
            set_cellstate(indexX, indexY);
            exchanged = true;
        }
    }
}

/**
 * Game main loop
 */
void game_loop(void) {

    int done = 0;
    int indexX = indexY = 0;
    SDL_Event event;

    //HgwError err;
    //HgwMessage msg;
    //HgwMessageFlags flags;

    gState = NEWGAME_STATE;
    exchanged = false;
    difficulty_offset = 0;

    // Main loop
    while (!done) {

        //err = hgw_msg_check_incoming(hgw_context, &msg, flags);
        //printf("\n\nhgw_msg_check_incoming: %d (%d)\n\n", msg.type, err);
        //if (err == HGW_ERR_COMMUNICATION) {
        //    printf("************** hgw_msg: %x\n\n", msg);
        //}

        if (gState == NEWGAME_STATE){
            new_game();
        }
        if (marked == (BOMBS_NUMBER - 4*difficulty_offset)){
            if (check_end_game()){
                gState = CLEAREDGAME_STATE;
            }
        }
        if ( SDL_PollEvent(&event)) {
            // Key pressed
            if (event.key.state == SDL_PRESSED) {

                // Quit
                if (event.key.keysym.sym == SDLK_ESCAPE) {
#if HGW_FUNC
                    quit_callback(0);
#endif
                    done = true;
                // Continue
                } else if (event.key.keysym.sym == SDLK_F4
                            || event.key.keysym.sym == SDLK_F5
                            || event.key.keysym.sym == SDLK_F6) {
                    statesave(clickType, marked, difficulty);
#if HGW_FUNC
                    exit_callback(0);
#endif
                    done = true;
                }
            }
            switch(event.type)
            {
                // Quit the game
                case SDL_QUIT:
#if HGW_FUNC
                    quit_callback(0);
#endif
                    done = true;
                    break;
                case SDL_KEYUP:
                    switch (clickType)
                    {
                        case CLICK_SEARCH:
                            clickType=CLICK_MARK;
                        break;
                        case CLICK_MARK:
                            clickType=CLICK_SEARCH;
                        break;
                    }
                case SDL_MOUSEBUTTONDOWN:
                        //restart game area
                        if (event.button.x >= RESTART_X
                           && event.button.x < RESTART_OFFSETX + RESTART_X
                           && event.button.y >= RESTART_Y
                           && event.button.y < RESTART_OFFSETY + RESTART_Y)
                        {

                            gState = NEWGAME_STATE;

                        }
                        //restart game area
                        if (gState == GAMEOVER_STATE || gState == CLEAREDGAME_STATE)
                        {
                            if (event.button.x >= SCREEN_BOUNDARYX/2 - DIALOG_SIZEX/2
                               && event.button.x < SCREEN_BOUNDARYX/2 - DIALOG_SIZEX/2 + DIALOG_SIZEX
                               && event.button.y >= SCREEN_BOUNDARYY/2 - DIALOG_SIZEY/2
                               && event.button.y < SCREEN_BOUNDARYY/2 - DIALOG_SIZEY/2 + DIALOG_SIZEY)
                            {

                                gState = NEWGAME_STATE;
                            }
                        }
                        if (clickType == CLICK_SEARCH) {
                           // Mines Field
                            if ((event.button.x > BOARD_OFFSETX
                               && event.button.x <= COLS * (MINE_SIZE + 1) + BOARD_OFFSETX
                               && event.button.y >= BOARD_OFFSETY
                               && event.button.y < ROWS * (MINE_SIZE + 1) + BOARD_OFFSETY)
                               && (gState != GAMEOVER_STATE)) {

                                indexX = (event.button.x - BOARD_OFFSETX) / (MINE_SIZE + 1);
                                indexY = (event.button.y - BOARD_OFFSETY) / (MINE_SIZE + 1);

                                switch((int)cellstates[indexX+indexY*COLS]) {
                                    case UNREVEALED_CELL:
                                        cellstates[indexX+indexY*COLS] = REVEALED_CELL;
                                        if ((int)minefield[indexX+indexY*COLS] == IS_BOMB){
                                            gameOver();
                                        }
                                        else if (minefield[indexX+indexY*COLS] == NO_NUMBER){
                                            show_cells_around(indexX, indexY);
                                        }
                                        break;
                                    case UNREVEALEDDOWN_CELL:
                                        break;
                                    case QUESTIONMARKED_CELL:
                                        cellstates[indexX+indexY*COLS] = REVEALED_CELL;
                                        break;
                                    case QUESTIONMARKEDDOWN_CELL:
                                        break;
                                    case MARKED_CELL:
                                        cellstates[indexX+indexY*COLS] = REVEALED_CELL;
                                        break;
                                    case REVEALED_CELL:
                                        click_revealed(indexX,indexY);
                                        break;
                                }
                            }
                            else if ((event.button.x >= CURSORFLAG_X
                               && event.button.x < CURSORFLAG_X + CURSORFLAG_OFFSETX
                               && event.button.y >= CURSORFLAG_Y
                               && event.button.y < CURSORFLAG_Y + CURSORFLAG_OFFSETY)
                               && (gState != GAMEOVER_STATE)){
                                    clickType=CLICK_MARK;
                            }
                        }
                        else if ((event.button.x >= CURSOROPEN_X
                               && event.button.x < CURSOROPEN_X + CURSOROPEN_OFFSETX
                               && event.button.y >= CURSOROPEN_Y
                               && event.button.y < CURSOROPEN_Y + CURSOROPEN_OFFSETY)
                               && (gState != GAMEOVER_STATE)) {
                            clickType=CLICK_SEARCH;
                        }
                        else {
                            if ((event.button.x > BOARD_OFFSETX
                               && event.button.x <= COLS * (MINE_SIZE + 1) + BOARD_OFFSETX
                               && event.button.y >= BOARD_OFFSETY
                               && event.button.y < ROWS * (MINE_SIZE + 1) + BOARD_OFFSETY)
                               && (gState != GAMEOVER_STATE)) {
                                    indexX = (event.button.x - BOARD_OFFSETX) / (MINE_SIZE + 1);
                                    indexY = (event.button.y - BOARD_OFFSETY) / (MINE_SIZE + 1);

                                    if ((int)cellstates[indexX+indexY*COLS] != REVEALED_CELL) {
                                        set_flag(indexX, indexY, (BOMBS_NUMBER-4*difficulty_offset));
                                    }
                                    else {
                                        click_revealed(indexX,indexY);
                                    }
                            }
                        }
                   break;
                case SDL_MOUSEBUTTONUP:
                    if (exchanged == true){
                        set_cellstate(indexX, indexY);
                        exchanged = false;
                    }
                    break;
                case SDL_MOUSEMOTION:
                    break;
            } // switch
        }//if
        draw_background(marked, clickType, difficulty_offset);
    }//while
}//game_loop

int main(int argc,char *argv[])
{
    HgwError err;
    HgwMessage msg;
    HgwMessageFlags flags;

    GConfClient *gc_client = NULL;
    g_type_init();
    gc_client = gconf_client_get_default();
    difficulty = gconf_client_get_int(gc_client, SETTINGS_DIFFICULTY, NULL);
    printf("\n difficulty plugin: %d", difficulty);


    #if HGW_FUNC
    hgw_context = hgw_context_compat_init(argc, argv);

    if(hgw_context == NULL)
    {
        return 0;
    }
    hgw_compat_set_cb_exit(hgw_context, exit_callback);
    hgw_compat_set_cb_quit(hgw_context, quit_callback);
    hgw_compat_set_cb_flush(hgw_context, flush_callback);

    if(!hgw_context_compat_check(hgw_context))
    {
        return 0;
    }

    /* hildon-games-wrapper part */
    //hgw_msg_compat_receive(hgw_context, 0);
    //err = 0;
    //msg.type = -1;
    //err = hgw_msg_check_incoming(hgw_context, &msg, flags);
    //printf("\n\nreceived %d (%d)\n\n", msg.type, err);
    usleep(100);
    #endif

    // Initialize SDL
    if (SDL_Init(SDL_INIT_VIDEO) < 0) {
        fprintf(stderr, "Couldn't initialize SDL: %s\n", SDL_GetError());
        return 0;
    }

    // Initialize video
    screen = SDL_SetVideoMode(800, 480, 16, SDL_DOUBLEBUF | SDL_HWSURFACE | SDL_FULLSCREEN);
    if (screen == NULL) {
        fprintf(stderr, "Unable to set 800x480 video: %s\n", SDL_GetError());
        return 0;
    }

    // Load images
    if (!load_surfaces()) {
        free_surfaces();
        SDL_Quit();
        fprintf(stderr, "Unable to load images!\n");
        return 0;
    }

    // Initialize game
    srand(time(NULL));

    game_loop();

    // Finalize game
    free_surfaces();
    SDL_Quit();

    /* Free memory */
#if HGW_FUNC
    hgw_context_compat_destroy_deinit(hgw_context);
#endif
    // bye
    return 0;
} // main

static int exit_callback(int errcode)
{
#if HGW_FUNC
  hgw_notify_startup(hgw_context, HGW_SUN_PAUSE);
#endif

  return 0;
}

static int quit_callback(int errcode)
{
  remove("/tmp/.maemosweeper-save");

#if HGW_FUNC
  //hgw_context_compat_destroy_quit(hgw_context);
#endif

  return 0;
}

static int flush_callback(int errcode)
{
  remove("/tmp/.maemosweeper-save");
  return 0;
}
