/*
Copyright (c) 2012, Petr Pehal, Jakub  plíchal, Rostislav Stríž,

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
    in the documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*************************************************************************************************
**
** file:    profilercontroller.cpp
** date:    10.12.2011
** authors: Petr Pehal,      xpehal00@stud.fit.vutbr.cz
**          Rostislav Striz, xstriz03@stud.fit.vutbr.cz
**          Jakub Splichal,  xsplic03@stud.fit.vutbr.cz
**
**************************************************************************************************/

#include <QTimerEvent>
#include <QSettings>
#include <cstdlib>
#include "profilercontroller.h"
#include "utils.h"
#include "qmlapplicationviewer.h"
#include <QDeclarativeContext>
#include <QDeclarativeComponent>
#include <QApplication>
#include <QDir>
#include <QPluginLoader>

#include "../config/configmanipulator.h"

// TODO
#include "../test/testconditions.h"
#include "../test/testactions.h"

/*************************************************************************************************/

const QString PluginDirectory = ".";

/*************************************************************************************************/

ProfilerController *ProfilerController::self;

/*************************************************************************************************/

ProfilerController::ProfilerController(QObject *parent) :
	QObject(parent), d(new ProfilerControllerData)
{
	if (self) {
		ERRORX("ProfilerController::ProfilerController(): There should be only one controller instance.");
	}

	self = this;

	d->config = new ConfigManipulator;
	d->autosaveInterval = ERR;
	d->currentProfile = 0;

	// default autosafe = 1min
	setAutosafeInterval(60 * 1000);
}

/*************************************************************************************************/

ProfilerController::~ProfilerController()
{
	if (!saveProfiles()) {
		ERROR("ProfilerController: Profile autosave failed.");
	}
	qDeleteAll(d->profiles);
	delete d->config;
	delete d;
}

/*************************************************************************************************/

void ProfilerController::setAutosafeInterval(int ms)
{
	if (d->autosaveInterval != ms) {
		d->autosaveInterval = ms;

		updateAutosafeIntervalChanged();
	}
}

/*************************************************************************************************/

void ProfilerController::updateAutosafeIntervalChanged()
{
	d->autosaveTimer.stop();

	if_noterr (d->autosaveInterval) {
		d->autosaveTimer.start(d->autosaveInterval, this);
	}
}

/*************************************************************************************************/

void ProfilerController::timerEvent(QTimerEvent *event)
{
	if (event->timerId() == d->autosaveTimer.timerId()) {
		if (!saveProfiles()) {
			ERROR("ProfilerController: Profile autosave failed.");
		}
		return;
	}

	QObject::timerEvent(event);
}

/*************************************************************************************************/

bool ProfilerController::loadPlugins()
{
//	TestCondition1Interface *testCondition1 = new TestCondition1Interface;
//	TestCondition2Interface *testCondition2 = new TestCondition2Interface;

//	TestAction1Interface *testAction1 = new TestAction1Interface;
//	TestAction2Interface *testAction2 = new TestAction2Interface;

//	d->conditionPlugins.append(testCondition1);
//	d->conditionPlugins.append(testCondition2);

//	d->actionPlugins.append(testAction1);
//	d->actionPlugins.append(testAction2);

	loadPluginsCore(PluginDirectory);

	return true;
}

/*************************************************************************************************/

bool ProfilerController::loadProfiles()
{
//	d->config->create();
//	d->config->save(QObjectList());
	const bool ret = d->config->loadOrCreate(d->profiles);

//	if (d->profiles.isEmpty()) {
//		Profile *profile1 = new Profile("PROFIL1");
//		Profile *profile2 = new Profile("PROFIL2");
//		Profile *profile3 = new Profile("PROFIL3");
//		Profile *profile4 = new Profile("PROFIL4");

//		d->profiles.append(profile1);
//		d->profiles.append(profile2);
//		d->profiles.append(profile3);
//		d->profiles.append(profile4);
//	}

	return ret;
}

/*************************************************************************************************/

bool ProfilerController::saveProfiles()
{
	return d->config->save(d->profiles);
}

/*************************************************************************************************/

void ProfilerController::registerQmlViewer(QmlApplicationViewer *viewer)
{
	if (d->viewer != viewer) {
		d->viewer = viewer;

		updateQmlViewerChanged();
	}
}

/*************************************************************************************************/

void ProfilerController::updateQmlViewerChanged()
{
	d->viewer->rootContext()->setContextProperty("Profiler", this);
	notifyQmlProfilesChanged();

	QObjectList Initializer;
	d->viewer->rootContext()->setContextProperty("ProfileConditions", QVariant::fromValue(Initializer));
	d->viewer->rootContext()->setContextProperty("ProfileActions", QVariant::fromValue(Initializer));

	d->viewer->rootContext()->setContextProperty("Plugins", QVariant::fromValue(Initializer));

	d->viewer->rootContext()->setContextProperty("Instance", QVariant::fromValue(Initializer));
}


/*************************************************************************************************/

void ProfilerController::setCurrentProfile(Profile *profile)
{
	if (d->currentProfile != profile) {
		d->currentProfile = profile;

		updateCurrentProfileChagned();
	}
}

/*************************************************************************************************/

void ProfilerController::updateCurrentProfileChagned()
{
	notifyQmlCurrentProfileChanged();
}

/*************************************************************************************************/

ConditionInterface *ProfilerController::conditionPlugin(const QString &name)
{
	foreach (QObject *object, d->conditionPlugins) {
		ConditionInterface *plugin = static_cast<ConditionInterface *>(object);
		if (plugin->name() == name) {
			return plugin;
		}
	}

	return 0;
}

/*************************************************************************************************/

ActionInterface *ProfilerController::actionPlugin(const QString &name)
{
	foreach (QObject *object, d->actionPlugins) {
		ActionInterface *plugin = static_cast<ActionInterface *>(object);
		if (plugin->name() == name) {
			return plugin;
		}
	}

	return 0;
}

/*************************************************************************************************/

void ProfilerController::initializeQmlPluginGUI(CoreInstance *instance)
{
	// TOTO mi nefunguje, protoze ve chvilu, kdy bych chtel inicializace provest
	// tak jeste komponenty neexistuji
//	QDeclarativeComponent *coverFlowPluginGUIComponent = d->viewer->
//			findChild<QDeclarativeComponent *>("coverFlowPluginGUI");
//	QDeclarativeComponent *directPluginGUIComponent = d->viewer->
//			findChild<QDeclarativeComponent *>("directPluginGUI");

//	if (!coverFlowPluginGUIComponent) {
//		IERROR;
//		return;
//	}

//	if (!directPluginGUIComponent) {
//		IERROR;
//		return;
//	}

//	QDeclarativeContext *coverFlowPluginGUIContext = coverFlowPluginGUIComponent->creationContext();
//	QDeclarativeContext *directPluginGUIContext = directPluginGUIComponent->creationContext();

//	if (!coverFlowPluginGUIContext || !directPluginGUIContext) {
//		IERROR;
//		return;
//	}

//	// --------------------------- //
//	// alespon zakladni bezpecnost //
//	// --------------------------- //
//	coverFlowPluginGUIContext->setContextProperty("Profiles", 0);
//	coverFlowPluginGUIContext->setContextProperty("ProfileConditions", 0);
//	coverFlowPluginGUIContext->setContextProperty("ProfileActions", 0);
//	coverFlowPluginGUIContext->setContextProperty("Plugins", 0);

//	directPluginGUIContext->setContextProperty("Profiles", 0);
//	directPluginGUIContext->setContextProperty("ProfileConditions", 0);
//	directPluginGUIContext->setContextProperty("ProfileActions", 0);
//	directPluginGUIContext->setContextProperty("Plugins", 0);

//	coverFlowPluginGUIContext->setContextProperty("Instance", instance);
//	directPluginGUIContext->setContextProperty("Instance", instance);

//	qDebug() << "registring: " << "Instance" << instance;

	d->viewer->rootContext()->setContextProperty("Instance", instance);
}

/*************************************************************************************************/
// QML NOTIFIKACE
/*************************************************************************************************/

void ProfilerController::notifyQmlProfilesChanged()
{
	d->viewer->rootContext()->setContextProperty("Profiles", QVariant::fromValue(d->profiles));
}

/*************************************************************************************************/

void ProfilerController::notifyQmlCurrentProfileChanged()
{
	d->viewer->rootContext()->setContextProperty("ProfileConditions", QVariant::fromValue(d->currentProfile->conditions()));
	d->viewer->rootContext()->setContextProperty("ProfileActions", QVariant::fromValue(d->currentProfile->actions()));
}

/*************************************************************************************************/
// QML INTERFACE
/*************************************************************************************************/

void ProfilerController::listConditionPlugins()
{
	d->viewer->rootContext()->setContextProperty("Plugins", QVariant::fromValue(d->conditionPlugins));
}

/*************************************************************************************************/

void ProfilerController::listActionPlugins()
{
	d->viewer->rootContext()->setContextProperty("Plugins", QVariant::fromValue(d->actionPlugins));
}

/*************************************************************************************************/

int ProfilerController::conditionPluginCount()
{
	return d->conditionPlugins.count();
}

/*************************************************************************************************/

int ProfilerController::actionPluginCount()
{
	return d->actionPlugins.count();
}

/*************************************************************************************************/

QUrl ProfilerController::actionPluginIcon(int actionPluginIndex)
{
	outlistIERRORS(actionPluginIndex, d->actionPlugins, QUrl());

	CoreInterface *plugin = actionPlugin(actionPluginIndex);

	return plugin->icon();
}

/*************************************************************************************************/

QUrl ProfilerController::conditionPluginIcon(int conditionPluginIndex)
{
	outlistIERRORS(conditionPluginIndex, d->conditionPlugins, QUrl());

	CoreInterface *plugin = conditionPlugin(conditionPluginIndex);

	return plugin->icon();
}

/*************************************************************************************************/

void ProfilerController::addProfile(const QString &name)
{
	Profile *profile = new Profile(name.isEmpty() ? "Profile " + QString::number(d->profiles.count())
												  : name);

	d->profiles.append(profile);

	notifyQmlProfilesChanged();
}

/*************************************************************************************************/

void ProfilerController::renameProfile(int profilesIndex, const QString &name)
{
	outlistIERROR(profilesIndex, d->profiles);

	profile(profilesIndex)->setName(name);

	notifyQmlProfilesChanged();
}

/*************************************************************************************************/

void ProfilerController::removeProfile(int profilesIndex)
{
	outlistIERROR(profilesIndex, d->profiles);

	delete static_cast<Profile *>(d->profiles.takeAt(profilesIndex));

	notifyQmlProfilesChanged();
}

/*************************************************************************************************/

void ProfilerController::selectProfile(int profilesIndex)
{
	outlistIERROR(profilesIndex, d->profiles);

	Profile *profile = this->profile(profilesIndex);

	setCurrentProfile(profile);
}

/*************************************************************************************************/

bool ProfilerController::verifyProfileRename(const QString &name)
{
	if (name.isEmpty()) {
		return false;
	}

	int count = d->profiles.count();
	for (int i = 0; i < count; i++) {
		Profile *profile = this->profile(i);
		if (profile->name() == name) {
			return false;
		}
	}

	return true;
}

/*************************************************************************************************/

QUrl ProfilerController::currentProfileConditionPluginQmlGUI(int profileConditionIndex)
{
	if (!d->currentProfile) {
		IERROR;
		return QUrl();
	}

	outlistIERRORS(profileConditionIndex, d->currentProfile->conditions(), QUrl());

	Condition *condition = d->currentProfile->condition(profileConditionIndex);
	ConditionInterface *plugin = condition->interface();

	initializeQmlPluginGUI(condition);

	QUrl url = plugin->qmlGUIFile();

//	qDebug() << "FOR INDEX: " << profileConditionIndex << "SENDING CONDTION: " << plugin->name() << url;

	return url;
}

/*************************************************************************************************/

QUrl ProfilerController::currentProfileActionPluginQmlGUI(int profileActionIndex)
{
	if (!d->currentProfile) {
		IERROR;
		return QUrl();
	}

	outlistIERRORS(profileActionIndex, d->currentProfile->actions(), QUrl());

	Action *action = d->currentProfile->action(profileActionIndex);
	ActionInterface *plugin = action->interface();

	initializeQmlPluginGUI(action);

	QUrl url = plugin->qmlGUIFile();

//	qDebug() << "FOR INDEX: " << profileActionIndex << "SENDING ACTION: " << plugin->name() << url;

	return url;
}

/*************************************************************************************************/

QUrl ProfilerController::conditionPluginQmlGUI(const QString &conditionName)
{
	ConditionInterface *plugin = conditionPlugin(conditionName);

	if (!plugin) {
		qDebug() << "NO CONDITION PLUGIN NAMED: " << conditionName;
		IERROR;
		return QUrl();
	}

//	initializeQmlPluginGUI(plugin->createCondition());

	QUrl url = plugin->qmlGUIFile();

//	qDebug() << "FOR NAME: " << conditionName << "SENDING CONDTION: " << plugin->name() << url;

	return url;
}

/*************************************************************************************************/

QUrl ProfilerController::actionPluginQmlGUI(const QString &actionName)
{
	ActionInterface *plugin = actionPlugin(actionName);

	if (!plugin) {
		qDebug() << "NO ACTION PLUGIN NAMED: " << actionName;
		IERROR;
		return QUrl();
	}

//	initializeQmlPluginGUI(plugin->createAction());

	QUrl url = plugin->qmlGUIFile();

//	qDebug() << "FOR NAME: " << actionName << "SENDING ACTION: " << plugin->name() << url;

	return url;
}

/*************************************************************************************************/

void ProfilerController::addCurrentProfileCondition(int pluginConditionIndex)
{
	if (!d->currentProfile) {
		IERROR;
		return;
	}

	outlistIERROR(pluginConditionIndex, d->conditionPlugins);

	Condition *condition = conditionPlugin(pluginConditionIndex)->createCondition();

	d->currentProfile->addCondition(condition);

	notifyQmlCurrentProfileChanged();
}

/*************************************************************************************************/

void ProfilerController::removeCurrentProfileCondition(int profileConditionIndex)
{
	if (!d->currentProfile) {
		IERROR;
		return;
	}

	outlistIERROR(profileConditionIndex, d->currentProfile->conditions());

	d->currentProfile->removeCondition(profileConditionIndex);

	notifyQmlCurrentProfileChanged();
}

/*************************************************************************************************/

void ProfilerController::addCurrentProfileAction(int pluginActionIndex)
{
	if (!d->currentProfile) {
		IERROR;
		return;
	}

	outlistIERROR(pluginActionIndex, d->actionPlugins);

	Action *action = actionPlugin(pluginActionIndex)->createAction();

	d->currentProfile->addAction(action);

	notifyQmlCurrentProfileChanged();
}

/*************************************************************************************************/

void ProfilerController::removeCurrentProfileAction(int profileActionIndex)
{
	if (!d->currentProfile) {
		IERROR;
		return;
	}

	outlistIERROR(profileActionIndex, d->currentProfile->actions());

	d->currentProfile->removeAction(profileActionIndex);

	notifyQmlCurrentProfileChanged();
}

/*************************************************************************************************/

void ProfilerController::loadPluginsCore(const QString &directory)
{
	QDir pluginsDir(directory);
    qDebug() << directory;

	foreach (QString fileName, pluginsDir.entryList(QDir::Files)) {
		QPluginLoader loader(pluginsDir.absoluteFilePath(fileName));

		qDebug() << "hasPlugin: " << fileName;

		QObject *plugin = loader.instance();

		if (plugin) {
			if (ConditionInterface *conditionPlugin = qobject_cast<ConditionInterface *>(plugin)) {
				qDebug() << "plugin is condition: " << conditionPlugin->name();
				d->conditionPlugins.append(conditionPlugin);
				continue;
			}

			if (ActionInterface *actionPlugin = qobject_cast<ActionInterface *>(plugin)) {
				qDebug() << "plugin is action" << actionPlugin->name();
				d->actionPlugins.append(actionPlugin);
				continue;
			}

			qDebug() << "unrecognized plugin";
		} else {
			qDebug() << "loader failed:" << loader.errorString();
		}
	}
}

/*************************************************************************************************/
