/*
Copyright (c) 2012, Petr Pehal, Jakub Šplíchal, Rostislav Stríž,

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
    in the documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*************************************************************************************************
**
** file:    profile.cpp
** date:    10.12.2011
** authors: Petr Pehal,      xpehal00@stud.fit.vutbr.cz
**          Rostislav Striz, xstriz03@stud.fit.vutbr.cz
**          Jakub Splichal,  xsplic03@stud.fit.vutbr.cz
**
**************************************************************************************************/

#include "profile.h"
#include "utils.h"
#include <QDebug>

/*************************************************************************************************/
// PROFILE
/*************************************************************************************************/

Profile::Profile() :
	d(new ProfileData)
{
	init();
}

/*************************************************************************************************/

Profile::Profile(const QString &name, bool active) :
	d(new ProfileData)
{
	init();

	setName(name);
	setActive(active);
}


/*************************************************************************************************/

Profile::~Profile()
{
	qDeleteAll(d->conditions);
	qDeleteAll(d->actions);

	delete d;
}

/*************************************************************************************************/

void Profile::addAction(Action *action)
{
	d->actions.append(action);

	if (d->doingActions) {
		action->doAction();
	}
}

/*************************************************************************************************/

void Profile::removeAction(int index)
{
	outlistIERROR(index, d->actions);

	Action *action = static_cast<Action *>(d->actions.takeAt(index));

	if (d->doingActions) {
		action->stopAction();
	}

	delete action;
}

/*************************************************************************************************/

void Profile::removeAction(Action *action)
{
	removeAction(d->actions.indexOf(action));
}

/*************************************************************************************************/

void Profile::addCondition(Condition *condition)
{
	d->conditions.append(condition);

	connect(condition, SIGNAL(stateChanged(bool)),
			this, SLOT(conditionStateChanged(bool)));

	update();
}

/*************************************************************************************************/

void Profile::removeCondition(int index)
{
	outlistIERROR(index, d->conditions);

	Condition *condition = static_cast<Condition *>(d->conditions.takeAt(index));

	update();

	delete condition;
}

/*************************************************************************************************/

void Profile::removeCondition(Condition *condition)
{
	removeCondition(d->conditions.indexOf(condition));
}

/*************************************************************************************************/

void Profile::setName(const QString &name)
{
	if (d->name != name) {
		d->name = name;

		updateNameChanged();
	}
}

/*************************************************************************************************/

void Profile::updateNameChanged()
{
	emit nameChanged();
}

/*************************************************************************************************/

void Profile::setActive(bool on)
{
	if (d->active != on) {
		d->active = on;

		updateActiveChanged();
	}
}

/*************************************************************************************************/

void Profile::updateActiveChanged()
{
	update();

	emit activeChanged();
}

/*************************************************************************************************/

void Profile::conditionStateChanged(bool )
{
	update();
}

/*************************************************************************************************/

void Profile::init()
{
	d->active       = false;
	d->doingActions = false;
}

/*************************************************************************************************/

void Profile::update()
{
	const bool updateDoingActions = d->active && testConditions();

	if (d->doingActions) {
		if (!updateDoingActions) {
			stopActions();
		}
	} else {
		if (updateDoingActions) {
			doActions();
		}
	}

	d->doingActions = updateDoingActions;
	emit conditionsStateChagned();
}

/*************************************************************************************************/

bool Profile::testConditions() const
{
	foreach (QObject *condition, d->conditions) {
		if (!static_cast<Condition *>(condition)->state()) {
			return false;
		}
	}

	return d->conditions.isEmpty() ? false : true;
}

/*************************************************************************************************/

void Profile::doActions() const
{
	foreach (QObject *action, d->actions) {
		static_cast<Action *>(action)->doAction();
	}
}

/*************************************************************************************************/

void Profile::stopActions() const
{
	foreach (QObject *action, d->actions) {
		static_cast<Action *>(action)->stopAction();
	}
}

/*************************************************************************************************/


