/*
Copyright (c) 2012, Petr Pehal, Jakub Šplíchal, Rostislav Stríž,

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
    in the documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*************************************************************************************************
**
** file:    configreader.cpp
** date:    10.12.2011
** authors: Petr Pehal,      xpehal00@stud.fit.vutbr.cz
**          Rostislav Striz, xstriz03@stud.fit.vutbr.cz
**          Jakub Splichal,  xsplic03@stud.fit.vutbr.cz
**
**************************************************************************************************/

#include "configreader.h"
#include "../core/profilercontroller.h"
#include <QDebug>

/*************************************************************************************************/

static const QString errorMessage = "Invalid profiler configuration file!";

/*************************************************************************************************/
// CONFIG READER
/*************************************************************************************************/

ConfigReader::ConfigReader(const QString &data) :
	QXmlStreamReader(data), d(new ConfigReaderData)
{
}

/*************************************************************************************************/

ConfigReader::~ConfigReader()
{
	delete d;
}

/*************************************************************************************************/

bool ConfigReader::read()
{
	while (!atEnd()) {
		readNext();

		if (isStartElement()) {
			if (name() == "profiles") {
				readProfiles();
			} else {
				raiseError();
			}
		}
	}

	return !hasError();
}

/*************************************************************************************************/

void ConfigReader::readProfiles()
{
	d->profiles.clear();
	d->currentActionInterface = 0;
	d->currentConditionInterface = 0;
	d->currentProfile = 0;

	while (!atEnd()) {
		readNext();

		if (isEndElement()) {
			if (name() == "profiles") {
				break;
			} else {
				raiseError();
			}
		}

		if (isStartElement()) {
			if (name() == "profile") {
				const QString name   = attributes().value("name").toString();
				const QString active = attributes().value("active").toString();

				bool ok = false;
				const bool isActive = active.toInt(&ok);

				if (name.isEmpty() || active.isEmpty() || !ok) {
					raiseError("Invalid profile name");
				} else {
					d->currentProfile = new Profile(name, isActive);
					d->profiles.append(d->currentProfile);
				}

				readProfile();
			} else {
				raiseError();
			}
		}
	}
}

/*************************************************************************************************/

void ConfigReader::readProfile()
{
	while (!atEnd()) {
		readNext();

		if (isEndElement()) {
			if (name() == "profile") {
				break;
			} else {
				raiseError();
			}
		}

		if (isStartElement()) {
			if (name() == "conditions") {
				readConditions();
			} else if (name() == "actions") {
				readActions();
			} else {
				raiseError();
			}
		}
	}
}

/*************************************************************************************************/

void ConfigReader::readConditions()
{
	while (!atEnd()) {
		readNext();

		if (isEndElement()) {
			if (name() == "conditions") {
				break;
			} else {
				raiseError();
			}
		}

		if (isStartElement()) {
			if (name() == "condition") {
				const QString name = attributes().value("name").toString();
				d->currentConditionInterface = Profiler->conditionPlugin(name);

				if (!d->currentConditionInterface) {
					raiseError(QString("Condition plugin '%1' not found!").arg(name));
				} else {
					readCondition();
				}
			} else {
				raiseError();
			}
		}
	}
}

/*************************************************************************************************/

void ConfigReader::readActions()
{
	while (!atEnd()) {
		readNext();

		if (isEndElement()) {
			if (name() == "actions") {
				break;
			} else {
				raiseError();
			}
		}

		if (isStartElement()) {
			if (name() == "action") {
				const QString name = attributes().value("name").toString();
				d->currentActionInterface = Profiler->actionPlugin(name);

				if (!d->currentActionInterface) {
					raiseError(QString("Action plugin '%1' not found!").arg(name));
				} else {
					readAction();
				}
			} else {
				raiseError();
			}
		}
	}
}

/*************************************************************************************************/

void ConfigReader::readCondition()
{
	if (!d->currentProfile || !d->currentConditionInterface) {
		raiseIError();
		return;
	}

	const QString data = readElementText();

	Condition *condition = d->currentConditionInterface->createCondition();

	condition->load(QByteArray::fromBase64(data.toLocal8Bit()));

	d->currentProfile->addCondition(condition);
}

/*************************************************************************************************/

void ConfigReader::readAction()
{
	if (!d->currentProfile || !d->currentActionInterface) {
		raiseIError();
		return;
	}

	const QString data = readElementText();

	Action *action = d->currentActionInterface->createAction();

	action->load(QByteArray::fromBase64(data.toLocal8Bit()));

	d->currentProfile->addAction(action);
}

/*************************************************************************************************/

void ConfigReader::raiseIError()
{
	raiseError("INTERNAL ERROR");
}

/*************************************************************************************************/

void ConfigReader::raiseError()
{
	raiseError(errorMessage);
}

void ConfigReader::raiseError(const QString &message)
{
	QXmlStreamReader::raiseError(message);
}

/*************************************************************************************************/
