/*  
 *  Graphics Routines -- graphics.c
 *  $Id: graphics.c,v 1.1.2.20 2004/01/14 05:18:19 sparrow_hawk Exp $
 *
 *  Copyright (C) 2003 Kevin Riggle 
 *  http://cmancala.sourcefoge.net
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details, a copy of which may be found in
 *  the file COPYING provided in the main directory of this release.
 *
 */

#include <stdio.h>

#include "SDL.h"
#include "SDL_endian.h"	/* Used for the endian-dependent 24 bpp mode */

#include "SDL_image.h"
#include "SDL_ttf.h"

#include "graphics.h"
#include "main.h"
#include "mancala.h"
#include "play.h"

static SDL_Surface *ShiftColors(SDL_Surface *old, Uint8 red, Uint8 blue, Uint8 green, Uint8 alpha);

SDL_Surface *LoadRes(char *filename) {

	return IMG_LoadPNG_RW(SDL_RWFromFile(filename, "r"));

}


SDL_Rect SurfaceToRect(SDL_Surface *src) {

	SDL_Rect dest;

	dest.x = 0;
	dest.y = 0;
	dest.w = src->w;
	dest.h = src->h;

	return dest;

}

/* create a new, blank surface with another's properties */
static SDL_Surface *NewSurfaceFrom(SDL_Surface *old, Uint32 flags) {

	return SDL_CreateRGBSurface(
			flags, old->w, old->h, old->format->BitsPerPixel,
			old->format->Rmask, old->format->Gmask, 
			old->format->Bmask, old->format->Amask
	);

}

static SDL_Surface *FillHole(int stones, TTF_Font *font, 
				SDL_Surface **stone_gfx) {

	SDL_Surface *text;
	SDL_Color font_color;
	char stone_string[STRING_MAX];
	int w, h;

	/* do we have a graphic for this number of stones? */
	if (stones <= STONE_MAX) {

		return *(stone_gfx + stones);
	}
	else {
		font_color.r = 128;
		font_color.g = 128;
		font_color.b = 128;
		/* convert the integer to text */
		if (sprintf(stone_string, "%d", stones)<=0)
			fprintf(stderr, "String conversion problem.\n");
		/* get the size of the rendered text */
		if (TTF_SizeText(font, stone_string, &w, &h)<0) 
			fprintf(stderr, "SDL_ttf: %s\n", TTF_GetError());
		/* if the text is too large, render a '?' */
		if ((w > (*stone_gfx)->w) || (h > (*stone_gfx)->h))
			sprintf(stone_string, "?");
		if (!(text = TTF_RenderText_Blended(font, stone_string, 
			font_color))) {
			fprintf(stderr, "SDL_ttf: %s\n", TTF_GetError());
			return NULL;
		}
		return text;
	}

	/* NEVER REACHED */
	return NULL;
}


SDL_Surface *DrawBoard(int *aiBoard, int *humanBoard,
			TTF_Font *board_font, TTF_Font *home_font,
			SDL_Surface *tile, SDL_Surface **stone_gfx,
			int active, int highlight) {

	SDL_Surface *board, *stones;
	SDL_Rect tile_rect, blit_rect;
	SDL_Color home_color;
	char home_string[STRING_MAX];
	int i;

	/* initialize the board surface */
	board = SDL_CreateRGBSurface(SDL_SWSURFACE, tile->w*(BOARD_MAX+2),
		tile->h*2, tile->format->BitsPerPixel, 0, 0, 0, 0);
	if (!board) {
		fprintf(stderr, "DrawBoard: %s\n", SDL_GetError());
		return NULL;
	}

	tile_rect = SurfaceToRect(tile);
	/*printf("tile is %dx%d\n", tile_rect.w, tile_rect.h);*/

	/* set the color of text in the home */
	home_color.r = 0;
	home_color.g = 0;
	home_color.b = 0;

	for (i=0; i<=BOARD_MAX+1; i++)
	{
		/* update current tile location */
		tile_rect.x = i * tile_rect.w;
		tile_rect.y = 0;
		/*printf("Currently a %dx%d rectangle at %d,%d.\n", tile_rect.w,
			tile_rect.h, tile_rect.x, tile_rect.y);*/
		blit_rect = tile_rect;
		/* is this is the first or last tile? */
		if ((i==0) || (i==BOARD_MAX+1)) {
			/* make it blank */
			if (SDL_BlitSurface(tile, NULL, board, &blit_rect)<0)
				fprintf(stderr,"DrawBoard: %s\n", 
					SDL_GetError());
			blit_rect = tile_rect;
			blit_rect.y = blit_rect.h;
			SDL_BlitSurface(tile, NULL, board, &blit_rect);
			/* render the number of stones in each home */
			if (i==0) {
				if (sprintf(home_string,"%d",humanBoard[0])<=0)
					printf("Human string problems.\n");
			}
			else {
				if (sprintf(home_string, "%d", aiBoard[0])<=0)
					printf("AI string problems.\n");
			}
                        if (!(stones = TTF_RenderText_Blended(home_font, 
				home_string, home_color)))
                                fprintf(stderr, "DrawBoard: %s\n", 
					TTF_GetError());
                        blit_rect.w = stones->w;
                        blit_rect.h = stones->h;
                        blit_rect.x += ((tile_rect.w - blit_rect.w)/2);
                        blit_rect.y = ((tile_rect.h*2 - blit_rect.h)/2);
                        SDL_BlitSurface(stones, NULL, board, &blit_rect);
		}
		/* otherwise, draw a hole and fill it */
		else {
			SDL_BlitSurface(stone_gfx[0], NULL, board, &blit_rect);
			if (humanBoard[i] > 0) {
				if (!(stones = FillHole(humanBoard[i], 
					board_font, stone_gfx)))
					fprintf(stderr,"FillHole() problems\n");
				if (i == highlight)
					stones = ShiftColors(stones,128,128,128,255);
				if (active == 1)
					stones = ShiftColors(stones,0,0,0,160);
				blit_rect.w = stones->w;
				blit_rect.h = stones->h;
				blit_rect.x += ((tile_rect.w - blit_rect.w)/2);
				blit_rect.y += ((tile_rect.h - blit_rect.h)/2);
				SDL_BlitSurface(stones,NULL,board,&blit_rect);
			}

			blit_rect = tile_rect;
			blit_rect.y = blit_rect.h;
			SDL_BlitSurface(stone_gfx[0], NULL, board, &blit_rect);
			if (aiBoard[BOARD_MAX-i+1] > 0) {
				if (!(stones = FillHole(aiBoard[BOARD_MAX-i+1], 
					board_font, stone_gfx)))
					fprintf(stderr,"FillHole() problems\n");
				if (active == 0)
					stones = ShiftColors(stones,0,0,0,160);
				blit_rect.w = stones->w;
				blit_rect.h = stones->h;
				blit_rect.x += ((tile_rect.w - blit_rect.w)/2);
				blit_rect.y += ((tile_rect.h - blit_rect.h)/2);
				SDL_BlitSurface(stones,NULL,board,&blit_rect);
			}
		}
	}

	SDL_FreeSurface(stones);

	return board;
}


/* flagrantly stol^H^H^Hborrowed from SDL Introduction */
/* passing 0 for red, green, or blue leaves them alone */
/* passing 0 for alpha makes the image transparent     */

SDL_Surface *ShiftColors(SDL_Surface *old, Uint8 red, Uint8 green, Uint8 blue, Uint8 alpha)
{
    SDL_Surface *new;
    Uint8 r, g, b, a;
    int x, y;

    new = NewSurfaceFrom(old, (SDL_SWSURFACE|SDL_SRCALPHA));

    if ( SDL_MUSTLOCK(old) ) {
	if (SDL_LockSurface(old) < 0 ) {
	    return NULL;
	}
    }

    if ( SDL_MUSTLOCK(new) ) {
        if ( SDL_LockSurface(new) < 0 ) {
            return NULL;
        }
    }

    for (x=0; x<old->w; x++)
    for (y=0; y<old->h; y++) {
 
    r = g = b = a = 0;

    switch (old->format->BytesPerPixel) {
        case 1: { /* Assuming 8-bpp */
            Uint8 *bufp, *bufq;

            bufp = (Uint8 *)old->pixels + y*old->pitch + x;
            bufq = (Uint8 *)new->pixels + y*new->pitch + x;
            SDL_GetRGBA((Uint32)*bufp, old->format, &r, &g, &b, &a);
	    a *= alpha/SDL_ALPHA_OPAQUE;
	    *bufq = (Uint8)SDL_MapRGBA(new->format, r, g, b, a);
        }
        break;

        case 2: { /* Probably 15-bpp or 16-bpp */
            Uint16 *bufp, *bufq;

            bufp = (Uint16 *)old->pixels + y*old->pitch/2 + x;
            bufq = (Uint16 *)new->pixels + y*new->pitch + x;
	    SDL_GetRGBA((Uint32)*bufp, new->format, &r, &g, &b, &a);
	    a *= alpha/SDL_ALPHA_OPAQUE;
            *bufq = (Uint8)SDL_MapRGBA(new->format, r, g, b, a);
        }
        break;

        case 3: { /* Slow 24-bpp mode, usually not used */
            Uint8 *bufp, *bufq;
	    Uint32 color;

            bufp = (Uint8 *)old->pixels + y*old->pitch + x * 3;
            bufq = (Uint8 *)new->pixels + y*new->pitch + x * 3;
	    SDL_GetRGBA((Uint32)*bufp, old->format, &r, &g, &b, &a);
	    a *= alpha/SDL_ALPHA_OPAQUE;
	    color = SDL_MapRGBA(new->format, r, g, b, a);
            if(SDL_BYTEORDER == SDL_LIL_ENDIAN) {
                bufq[0] = color;
                bufq[1] = color >> 8;
                bufq[2] = color >> 16;
            } else {
                bufq[2] = color;
                bufq[1] = color >> 8;
                bufq[0] = color >> 16;
            }
        }
        break;

        case 4: { /* Probably 32-bpp */
            Uint32 *bufp, *bufq;

            bufp = (Uint32 *)old->pixels + y*old->pitch/4 + x;
            bufq = (Uint32 *)new->pixels + y*new->pitch/4 + x;
	    SDL_GetRGBA(*bufp, old->format, &r, &g, &b, &a);
	    a = (int)((long)a * (long)alpha/SDL_ALPHA_OPAQUE);
	    r += (int)((long)(SDL_ALPHA_OPAQUE - r) * (long)red/SDL_ALPHA_OPAQUE);
	    g += (int)((long)(SDL_ALPHA_OPAQUE - g) * (long)green/SDL_ALPHA_OPAQUE);
	    b += (int)((long)(SDL_ALPHA_OPAQUE - b) * (long)blue/SDL_ALPHA_OPAQUE);
          *bufq = SDL_MapRGBA(new->format, r, g, b, a);
        }
        break;
    }
    }

    if ( SDL_MUSTLOCK(new) ) {
	SDL_UnlockSurface(new);
    }
  
    if ( SDL_MUSTLOCK(old) ) {
        SDL_UnlockSurface(old);
    }
    
    return new;
}


/*  End graphics.c  */
