/* File: main.cpp
 *
 * This file is part of MaeLyrica.
 *
 * Copyright (C) 2012 Marcin Mielniczuk.
 *
 * MaeLyrica is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, version 3 of the License.
 *
 * MaeLyrica is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 *  along with MaeLyrica.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef LYRICSDOWNLOADER_H
#define LYRICSDOWNLOADER_H

#include <sstream>
#include <curl/curl.h>
#include <QString>
#include <QRegExp>
#include <QtDeclarative>

#include "settings.h"
using namespace std;

class lyricsDownloader : public QObject
{
    Q_OBJECT
public:

    Q_PROPERTY(QString lyrics READ getLyrics WRITE setLyrics NOTIFY lyricsChanged)
    Q_PROPERTY(QString savedLyrics READ getSavedLyrics WRITE setSavedLyrics NOTIFY savedLyricsChanged)
    Q_PROPERTY(bool theseLyricsSaved READ getTheseLyricsSaved NOTIFY theseLyricsSavedSignal)
    Q_INVOKABLE virtual short perform() = 0;
    /* Do all the activities needed for the lyrics to be shown to the user
     * Return values:
     * 0 - OK
     * 1 - Error downloading
     * 2 - Error parsing
     * 3 - Lyrics absent (1 & 2)
     */
    inline void setData(const string & a, const string & t); // set artist and track
    Q_INVOKABLE inline void setData(const QString & a, const QString & t) // for QStrings
    {
        artist = a.toStdString();
        track = t.toStdString();
        emit dataChanged();
    }
    Q_INVOKABLE inline void setData(const QString & str) {setData(str.split('-')); }
    Q_INVOKABLE inline bool anyFieldEmpty() // check whether everything is filled
    {
        if ( artist.empty() || track.empty() ) return true;
        else return false;
    }

    Q_INVOKABLE inline bool lyricsExistMaeLyrica() {return progSettings.contains(savedLyricsKey);}
    Q_INVOKABLE bool lyricsExistOMP();
    Q_INVOKABLE void saveLyricsMaeLyrica();
    Q_INVOKABLE void saveLyricsOMP();
    Q_INVOKABLE inline void deleteLyricsMaeLyrica() {progSettings.remove(savedLyricsKey);}

protected:
    lyricsDownloader(const string & a, const string & t ) : artist(a), track(t)
    {
        connect(this, SIGNAL(dataChanged()), this, SLOT(cleanTitle()));
        connect(this, SIGNAL(dataChanged()), this, SLOT(findSavedLyricsKey())); // recalculate the key, so that it's always up-to-date
    }
    // settings object
    static Settings progSettings;
    // functions
    virtual CURLcode download() = 0;
    virtual string toProviderCode() const = 0;
    virtual bool parse() = 0;
    // overloaded for QML functions
    inline void setData(const QStringList& list) // for a list "artist - track"
    {
        artist = list[0].toStdString();
        track = list[1].toStdString();
        emit dataChanged();

    }
    // Q _PROPERTY fucntions
    inline void setLyrics(QString newlyrics) {lyrics = newlyrics;}
    inline void setSavedLyrics(QString newlyrics) { progSettings.setValue(savedLyricsKey, newlyrics); emit savedLyricsChanged(newlyrics); }
    inline QString getLyrics() {return lyrics;}
    inline QString getSavedLyrics() {return progSettings.value(savedLyricsKey).value<QString>();}
    inline bool getTheseLyricsSaved() {return progSettings.contains(savedLyricsKey);}
    // data
    QString lyrics;
    string* lyrics_str; // at first std::string will be used, but later QString instead
    static const string rooturl; // url for this provider
    string artist, track;
    static const QString recentSearchesUrl;
    // helper functions for curl
    static size_t write_data_to_var(char *ptr, size_t size, size_t nmemb, void *userdata);
private:
    QString savedLyricsKey;
    static const QString OMPLyricsRoot;
    QString cleanedTitle;
    QString cleanQString(QString qstr);
    QString lyricsUrlOMP;
signals:
    void dataChanged();
    void lyricsChanged(QString);
    void savedLyricsChanged(QString);
    void theseLyricsSavedSignal(QString);
protected slots:
    void findSavedLyricsKey() {savedLyricsKey = "lyrics/" + cleanedTitle;}
    void cleanTitle();
};

class AZLyricsDownloader : public lyricsDownloader
{
    Q_OBJECT
public:
    AZLyricsDownloader() : lyricsDownloader("", "") {}
    AZLyricsDownloader(const string & a, const string & t) : lyricsDownloader(a, t) {}
    Q_INVOKABLE short perform();
    //Q_INVOKABLE inline void setData(const string & a, const string & t);// set artist and track

protected:
    virtual CURLcode download();
    virtual string toProviderCode() const;
    virtual bool parse();
    /**/
    static const string rooturl;
    static const QRegExp startcomment, endcomment; // comments in the azlyrics.com website which indicate start and end of lyrics
};

#endif // LYRICSDOWNLOADER_H
