/* File: lyricsDownloader.cpp
 *
 * This file is part of MaeLyrica.
 *
 * Copyright (C) 2012 Marcin Mielniczuk.
 *
 * MaeLyrica is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, version 3 of the License.
 *
 * MaeLyrica is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 *  along with MaeLyrica.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "lyricsDownloader.h"
#include "LyricWikiDownloader.h"
#include "AZLyricsDownloader.h"
#include <curl/curl.h>
#include <QDebug>
#include <iostream>
#include <QFile>
#include <QSettings>
#include <QtNetwork/QtNetwork>
#include <iostream>

using namespace std;

/////////////////////////////////////////////////////////////////////////////

short lyricsDownloader::perform(const string& a, const string& t)
{
    CURLcode dlret; // return value
    dlret = download(a, t); // it doesn't fail yet to recognize

    if (dlret == -2) return -1; // no internet
    else if (dlret != 0) return 1; // error downloading
    //else continues!

    if ( parse() == false)
    {
        qDebug() << "Error while parsing lyrics\n";
        return 2;
    }

    return 0; // ok
}

CURLcode lyricsDownloader::download(const string& a, const string& t)
{
    QNetworkConfigurationManager manager;
    QNetworkConfiguration cfg = manager.defaultConfiguration();
    /*QNetworkSession* session = new QNetworkSession(cfg);
    session->setSessionProperty("ConnectInBackground", true);
    session->open();*/

    CURL * handle;
    CURLcode err;
    handle = curl_easy_init();
    if (! handle) return static_cast<CURLcode>(-1);
    // set verbose if debug on
    curl_easy_setopt( handle, CURLOPT_VERBOSE, DEBUG );
    curl_easy_setopt( handle, CURLOPT_URL, toProviderCode(a, t).c_str() ); // set the download url to the generated one
    curl_easy_setopt(handle, CURLOPT_WRITEDATA, &buff);
    curl_easy_setopt(handle, CURLOPT_WRITEFUNCTION, &lyricsDownloader::write_data_to_var); // set the function writing to ostringstream
    curl_easy_setopt(handle, CURLOPT_FOLLOWLOCATION, 1); // curl needs to follow redirects with this provider
    curl_easy_setopt(handle, CURLOPT_FAILONERROR, 1);
    if (manager.isOnline() == false) return static_cast<CURLcode>(-2);
    err = curl_easy_perform(handle);
    curl_easy_cleanup(handle);
    return err;
}

size_t lyricsDownloader::write_data_to_var(char *ptr, size_t size, size_t nmemb, void *userdata)
{
    ostringstream * stream = (ostringstream*) userdata;
    size_t count = size * nmemb;
    stream->write(ptr, count);
    return count;
}

/////////////////////////////////////////////////////////////////////////

Settings LyricsDownloaderManager::progSettings;

LyricsDownloaderManager::LyricsDownloaderManager(LyricsDownloaderClass lyricsDownloaderName)
{
    connect(this, SIGNAL(dataChanged()), this, SLOT(cleanTitle()));
    connect(this, SIGNAL(dataChanged()), this, SLOT(findSavedLyricsKey())); // recalculate the key, so that it's always up-to-date

    currentClass = lyricsDownloaderName;
    downloader = newDownloader(lyricsDownloaderName);
}

LyricsDownloaderManager::~LyricsDownloaderManager()
{
    delete downloader;
}

void LyricsDownloaderManager::changeLyricsDownloader(LyricsDownloaderClass lyricsDownloaderName)
{
    qDebug() << "Changing lyricsDownloader to " << lyricsDownloaderName << "\n";
    if (lyricsDownloaderName != currentClass) // we don't need to do the same again if it's the same downloader
    {
        lyricsDownloader * oldDownloader = downloader;
        downloader = newDownloader(lyricsDownloaderName);
        delete oldDownloader;
    }
}

lyricsDownloader * LyricsDownloaderManager::newDownloader(LyricsDownloaderClass & name)
{
    switch (name)
    {
    case AZLyricsDownloaderClass:
        return new AZLyricsDownloader;
        break;
    case LyricWikiDownloaderClass:
        return new LyricWikiDownloader;
        break;
    }
}

short LyricsDownloaderManager::perform()
{
    short ret = downloader -> perform(artist, track);
    lyrics = downloader -> lyrics_qstr; // clean it too!
    downloader -> lyrics_qstr.clear();
    emit lyricsChanged(lyrics);
    return ret;
}

void LyricsDownloaderManager::cleanTitle()
{
    cleanedTitle = cleanQString( QString::fromStdString(artist));
    cleanedTitle += "-";
    cleanedTitle += cleanQString(QString::fromStdString(track));
}

QString LyricsDownloaderManager::cleanQString(const QString qstr)
{
    return qstr.toLower().replace("&", "and").remove(QRegExp("\\([^)]*\\)")).remove(QRegExp("[\\W_]"));
}

void LyricsDownloaderManager::saveLyricsMaeLyrica()
{
    Settings().setValue("lyrics/" + cleanedTitle, lyrics);
}
