###############################################################################
# copyright 2010 Edwin Marshall (aspidites) <aspidties@inbox.com>             #
#                                                                             #
# This file is part of MaeGym                                                 #
#                                                                             #
# MaeGym is free software: you can redistribute it and/or                     #
# modify it under the terms of the GNU General Public License as published by #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# MaeGym is distributed in the hope that it will be useful,                   #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with MaeGym.                                                          # 
# If not, see <http://www.gnu.org/licenses/>.                                 #
###############################################################################

# pyqt modules
from PyQt4.QtGui import (QAbstractItemView, QApplication, QLabel, QListView, 
                         QMenu, QProgressDialog, QPushButton, QMainWindow, 
                         QVBoxLayout, QWidget)
from PyQt4.QtMaemo5 import QMaemo5EditBar
from PyQt4.QtCore import (pyqtSignature, pyqtSignal, Qt)

# third-party modules
import config as c

# program modules 
from ui.ui_exercises_window import Ui_ExercisesWindow
import routines_window as ww
import exercise_wizard as ew
import target_areas_window as taw

from delegates import ExercisesDelegate

class ExercisesWindow(QMainWindow, Ui_ExercisesWindow):

    def __init__(self, parent=None, database=None, t_name=None):
        super(ExercisesWindow, self).__init__(parent)
        self.setAttribute(Qt.WA_Maemo5StackedWindow, True)

        self.setupUi(self)
        self.setupEditBar()
        self.config = c.Config()
        self.db = database
        self.setupListView(t_name)

        self.customContextMenuRequested.connect(self.openMenu)

    def setupEditBar(self):
        self.delete_button = QPushButton('Delete')
        self.edit_bar = QMaemo5EditBar('Select Exercises', self)
        self.edit_bar.addButton(self.delete_button)
        self.edit_bar.setVisible(False)

        self.central_widget.layout().insertWidget(0, self.edit_bar)

    def closeEvent(self, event):
        if self.edit_bar.isVisible():
            self.exitFullScreen()
            event.ignore()
        else:
            event.accept()

    def refreshModel(self):
        self.model.select()

    def setupListView(self, t_name):
        self.model = self.db.exercises_model(t_name)

        if t_name:    
            self.target_areas_action.setEnabled(False)
            self.routines_action.setEnabled(False)
            self.new_exercise_button.hide()
            self.setWindowTitle(t_name)

        self.exercises_list_view.setModel(self.model)
        self.exercises_list_view.setModelColumn(1)
        self.exercises_list_view.setItemDelegate(ExercisesDelegate())

    @pyqtSignature('bool')
    def on_routines_action_triggered(self, checked):
        self.config.interface['last-fitness'] = 'routines'
        self.parent().setFitnessWindow()
        routines_window = ww.RoutinesWindow(self.parent(), self.db)
        routines_window.show()
        self.hide()

    @pyqtSignature('bool')
    def on_target_areas_action_triggered(self, checked):
        self.config.interface['last-fitness'] = 'target_areas'
        self.parent().setFitnessWindow()
        target_areas_window = taw.TargetAreasWindow(self.parent(), self.db)
        target_areas_window.show()
        self.hide()

    @pyqtSignature('bool')
    def on_delete_exercises_action_triggered(self, checked):
        def deleteExercises():
            indexes = []
            for index in self.exercises_list_view.selectedIndexes():
                try:
                    if index.row() > indexes[-1].row():
                        indexes.insert(0, index)
                    else:
                        indexes.append(index)
                except IndexError:
                    indexes.append(index)

            for index in indexes:
                self.model.removeRow(index.row())

            self.exitFullScreen()

        self.exercises_list_view.setSelectionMode(QAbstractItemView.MultiSelection)
        self.edit_bar.setVisible(True)
        self.new_exercise_button.setVisible(False)

        self.showFullScreen()

        self.delete_button.clicked.connect(deleteExercises)

    def on_new_exercise_button_pressed(self):
        exercise_wizard = ew.ExerciseWizard(self, self.db)
        exercise_wizard.exercisesUpdated.connect(self.refreshModel)
        exercise_wizard.show()


    @pyqtSignature('QModelIndex')
    def on_exercises_list_view_clicked(self, index):
        pass

    def exitFullScreen(self):
        self.exercises_list_view.clearSelection()
        self.exercises_list_view.setSelectionMode(QAbstractItemView.SingleSelection)
        self.new_exercise_button.setVisible(True)
        self.edit_bar.setVisible(False)
        self.showNormal()

    def openMenu(self, position):
        menu = QMenu(self)
        edit_action = menu.addAction('Edit')
        delete_action = menu.addAction('Delete')

        action = menu.exec_(self.mapToGlobal(position))

        if action == edit_action:
            index = self.exercises_list_view.currentIndex()
            name = self.model.record(index.row()).value('ex_name')

            exercise_wizard = ew.ExerciseWizard(self, self.db, name)
            exercise_wizard.exercisesUpdated.connect(self.refreshModel)
            exercise_wizard.setStartId(1)
            exercise_wizard.setWindowTitle('Edit Exercise')

            description = self.model.record(index.row()).value('ex_description')

            t_name = self.model.record(index.row()).value('t_name')
            t_id = self.db.getRow('t_name', t_name)

            f_name = self.model.record(index.row()).value('f_name')
            f_id = self.db.getRow('f_name', f_name)

            d_name = self.model.record(index.row()).value('d_name')
            d_id = self.db.getRow('d_name', d_name)

            i_name = self.model.record(index.row()).value('i_name')
            i_id = self.db.getRow('i_name', i_name)

            e_name = self.model.record(index.row()).value('e_name')
            e_id = self.db.getRow('e_name', e_name)
            

            media = self.model.index(index.row(),8).data()
            exercise_wizard.name_line_edit.setText(name.toString())
            exercise_wizard.description_text_edit.setHtml(description.toString())
            exercise_wizard.target_area_selector.setCurrentIndex(t_id)
            exercise_wizard.frequency_selector.setCurrentIndex(f_id)
            exercise_wizard.duration_selector.setCurrentIndex(d_id)
            exercise_wizard.intensity_selector.setCurrentIndex(i_id)
            exercise_wizard.equipment_selector.setCurrentIndex(e_id)
            exercise_wizard.media_line_edit.setText(media.toString())

            exercise_wizard.show()


        elif action == delete_action:
            self.model.removeRow(self.exercises_list_view.currentIndex().row())
