#include <QGridLayout>
#include <QStackedWidget>
#include <QLineEdit>
#include <QLabel>
#include <QPushButton>
#include <QString>
#include <QStringList>
#include <QValidator>
#include <QCheckBox>
#include <QComboBox>

#include "mainwindow.h"
#include "editinterface.h"
#include "processrunner.h"

static const char readonly_color[] = "background-color: #333";
static const char default_color[] = "background-color: white";

class IP4Validator : public QValidator
{
public:
	IP4Validator(QObject *parent = 0) : QValidator(parent) {}
	State validate(QString &input, int &pos) const
		{
			(void)pos;
			if (input.isEmpty())
				return Acceptable;

			QStringList octets = input.split('.');
			int s = octets.size();
			if(s > 4)
				return Invalid;

			bool empty = false;
			for (int i=0; i < s; i++) {
				if (octets[i].isEmpty()) {
					empty = true;
					continue;
				}

				bool ok;
				int val = octets[i].toInt(&ok);
				if (!ok || val < 0 || val > 255)
					return Invalid;
			}

			if (s < 4 || empty) return Intermediate;
			return Acceptable;
		}
};

class IP4MaskValidator : public QValidator
{
public:
	IP4MaskValidator(QObject *parent = 0) : QValidator(parent) {}
	State validate(QString &input, int &pos) const
		{
			(void)pos;
			if (input.isEmpty())
				return Acceptable;

			QStringList octets = input.split('.');
			int s = octets.size();
			if(s > 4)
				return Invalid;

			bool empty = false;
			bool zero = false;
			for (int i=0; i < s; i++) {

				if (octets[i].isEmpty()) {
					empty = true;
					continue;
				}
				bool ok;
				int val = octets[i].toInt(&ok);
				if (!ok || (zero && val != 0))
					return Invalid;

				switch (val) {
				case 1:
				case 2:
				case 25:
				case 24:
				case 22:
				case 19:
				case 12:
					if (i < s - 1)
						return Invalid;
					continue;

				case 255:
					continue;
				case 254:
				case 252:
				case 248:
				case 240:
				case 224:
				case 192:
				case 128:
				case 0:
					zero = true;
					continue;
				default:
					return Invalid;
				}
			}

			if (s < 4 || empty) return Intermediate;
			return Acceptable;
		}
};

EditInterface::EditInterface(MainWindow *main, ProcessRunner *runner)
	: m_main(main), m_runner(runner)
{
	m_layout = new QGridLayout;
	setLayout(m_layout);

	IP4Validator *ipvtor = new IP4Validator;
	IP4MaskValidator *maskvtor = new IP4MaskValidator;

	m_ipaddress = new QLabel(tr("IP Address:"));
	m_iptxt = new QLineEdit(tr("192.168.2.15"));
	m_iptxt->setFrame(false);
	m_iptxt->setValidator(ipvtor);
	m_iptxt->setMaxLength(15);
	m_iptxt->setStyleSheet(default_color);

	m_netmask = new QLabel(tr("Netmask:"));
	m_masktxt = new QLineEdit(tr("255.255.255.0"));
	m_masktxt->setFrame(false);
	m_masktxt->setValidator(maskvtor);
	m_masktxt->setMaxLength(15);
	m_masktxt->setStyleSheet(default_color);

	m_stack = new QStackedWidget();
	/*
	 * Extra spaces because of txt cannot been fully seen on N900
	 */
	m_route = new QCheckBox(tr("Default Route:  "));
	m_route->setTristate(false);

	m_peerip = new QLabel(tr("Peer IP:"));
	m_defroute = new QLabel(tr("Default Route:"));

	m_stack->addWidget(m_route);
	m_stack->addWidget(m_peerip);
	m_stack->addWidget(m_defroute);
	m_stack->setCurrentIndex(1);

	m_gwtxt = new QLineEdit(tr("192.168.2.14"));
	m_gwtxt->setFrame(false);
	m_gwtxt->setValidator(ipvtor);
	m_gwtxt->setMaxLength(15);
	m_gwtxt->setReadOnly(false);
	m_gwtxt->setStyleSheet(default_color);

	m_dhcp = new QComboBox();
	m_dhcp->addItem("DHCP server");
	m_dhcp->addItem("DHCP client");
	m_dhcp->addItem("Static");
	m_dhcp->setMaxCount(3);

	m_configure = new QPushButton(tr("Configure"));
	m_cancel = new QPushButton(tr("Cancel"));


	m_layout->addWidget(m_ipaddress, 0, 0, Qt::AlignLeft);
	m_layout->addWidget(m_iptxt,     0, 1, Qt::AlignLeft);

	m_layout->addWidget(m_netmask,   1, 0, Qt::AlignLeft);
	m_layout->addWidget(m_masktxt,   1, 1, Qt::AlignLeft);

	m_layout->addWidget(m_stack,     2, 0, Qt::AlignLeft);
	//m_layout->addWidget(m_route,     2, 0, Qt::AlignLeft);
	m_layout->addWidget(m_gwtxt,     2, 1, Qt::AlignLeft);

	m_layout->addWidget(m_dhcp,      0, 2, Qt::AlignLeft);

	m_layout->addWidget(m_cancel,    1, 2);
	m_layout->addWidget(m_configure, 2, 2);

	m_layout->setRowStretch(2, 1);

	connect(m_dhcp, SIGNAL(activated(int)),
		this, SLOT(toggleDhcp(int)));
	connect(m_route, SIGNAL(stateChanged(int)),
		this, SLOT(toggleRoute(int)));
	connect(m_cancel, SIGNAL(pressed()), m_main, SLOT(showLauncher()));
	connect(m_configure, SIGNAL(pressed()), this, SLOT(configure()));
}

void
EditInterface::configure0()
{
	QString name("./interfaces.sh");
	QStringList args(QString("usb0"));

	if (m_dhcp->currentIndex() == DHCP_CLIENT) {
		args.append("dhcp");
	} else if (m_dhcp->currentIndex() == DHCP_STATIC) {
		if (!m_iptxt->hasAcceptableInput()) {
			// TODO popup
			return ;
		}

		if (!m_masktxt->hasAcceptableInput()) {
			// TODO popup
			return ;
		}

		args.append("static");
		args.append(m_iptxt->text());
		args.append(m_masktxt->text());

		if (m_route->checkState() == Qt::Checked) {
			if (!m_gwtxt->hasAcceptableInput()) {
				// TODO popup check if valid with IP also ???
				return ;
			}
			args.append(m_gwtxt->text());
		}
	} else if (m_dhcp->currentIndex() == DHCP_SERVER) {
		if (!m_iptxt->hasAcceptableInput()) {
			// TODO popup
			return ;
		}

		if (!m_masktxt->hasAcceptableInput()) {
			// TODO popup
			return ;
		}

		args.append("dhcpd");
		args.append(m_iptxt->text());
		args.append(m_masktxt->text());
		args.append(m_gwtxt->text());
	}

	m_runner->start(name, args);
}

void
EditInterface::configure()
{
	configure0();
	m_main->showLauncher();
}

void
EditInterface::toggleDhcp(int state)
{
	if (state == DHCP_STATIC) {
		m_iptxt->setReadOnly(false);
		m_masktxt->setReadOnly(false);
		m_iptxt->setStyleSheet(default_color);
		m_masktxt->setStyleSheet(default_color);
		if (m_route->checkState() == Qt::Checked) {
			m_gwtxt->setStyleSheet(default_color);
			m_gwtxt->setReadOnly(false);
		} else {
			m_gwtxt->setStyleSheet(readonly_color);
			m_gwtxt->setReadOnly(true);
		}
		m_stack->setCurrentIndex(0);
	} else if (state == DHCP_SERVER) {
		m_iptxt->setReadOnly(false);
		m_masktxt->setReadOnly(false);
		m_iptxt->setStyleSheet(default_color);
		m_masktxt->setStyleSheet(default_color);
		m_gwtxt->setStyleSheet(default_color);
		m_gwtxt->setReadOnly(false);
		m_stack->setCurrentIndex(1);
	} else if (state == DHCP_CLIENT) {
		m_iptxt->setReadOnly(true);
		m_masktxt->setReadOnly(true);
		m_gwtxt->setReadOnly(true);
		m_iptxt->setStyleSheet(readonly_color);
		m_masktxt->setStyleSheet(readonly_color);
		m_gwtxt->setStyleSheet(readonly_color);
		m_stack->setCurrentIndex(2);
	}
}

void
EditInterface::toggleRoute(int state)
{
	if (m_dhcp->currentIndex() != DHCP_STATIC)
		return ;

	if (state == Qt::Checked) {

		m_gwtxt->setReadOnly(false);
		m_gwtxt->setStyleSheet(default_color);
	} else {
  		m_gwtxt->setReadOnly(true);
		m_gwtxt->setStyleSheet(readonly_color);
	}
}
