/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of libzeemote.
 *
 * libzeemote is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * libzeemote is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libzeemote.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "zeemote.h"

#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

/* Byte order conversions */
#if __BYTE_ORDER == __LITTLE_ENDIAN
#define htozs(d)  bswap_16(d)
#define ztohs(d)  bswap_16(d)
#elif __BYTE_ORDER == __BIG_ENDIAN
#define htozs(d)  (d)
#define ztohs(d)  (d)
#else
#error "Unknown byte order"
#endif

/* supported zeemote reports */
#define ZEEMOTE_BUTTONS      7
#define ZEEMOTE_STICK        8 
#define ZEEMOTE_BATTERY     17

/* unused entries in button report */
#define ZEEMOTE_BUTTON_NONE  (0xfe)

/* Peripheral, Pointing device/Joystick */
#define ZEEMOTE_CLASS  0x000584

/* internal list of all connected zeemotes */
static zeemote_t *zeemote_list = NULL;

static int rfcomm_connect(bdaddr_t *bdaddr, int channel) {
  struct sockaddr_rc rem_addr;
  int bt = -1;

  // connect to zeemote
  if( (bt = socket(PF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM)) < 0 ) {
    fprintf(stderr, "libzeemote: Can't create socket. %s(%d)\n", 
	    strerror(errno), errno);
    return -1;
  }

  /* connect on rfcomm */
  memset(&rem_addr, 0, sizeof(rem_addr));
  rem_addr.rc_family = AF_BLUETOOTH;
  rem_addr.rc_bdaddr = *bdaddr;
  rem_addr.rc_channel = channel;
  if( connect(bt, (struct sockaddr *)&rem_addr, sizeof(rem_addr)) < 0 ){
    fprintf(stderr, "libzeemote: Can't connect. %s(%d)\n", 
	    strerror(errno), errno);

    close(bt);
    return -1;
  }

  return bt;
}

static int zeemote_device_compare(const void *a, const void *b) {
  return bacmp(&((zeemote_device_t*)a)->bdaddr, 
	       &((zeemote_device_t*)b)->bdaddr);
}


/* scan for zeemotes in vicinity */
zeemote_scan_result_t *zeemote_scan(void) {
  zeemote_scan_result_t *result = malloc(sizeof(zeemote_scan_result_t));
  memset(result, 0, sizeof(zeemote_scan_result_t));

  inquiry_info *info = NULL;
  uint8_t lap[3] = { 0x33, 0x8b, 0x9e };
  int num_rsp, i;
  char name[16];

  int dev_id = hci_get_route(NULL);
  if(dev_id < 0)
    return result;

  num_rsp = hci_inquiry(-1, 8, 0, lap, &info, IREQ_CACHE_FLUSH);
  if (num_rsp < 0) {
    fprintf(stderr, "libzeemote: Inquiry failed: %s(%d)\n", 
	    strerror(errno), errno);
    return result;
  }

  int dd = hci_open_dev(dev_id);
  if (dd < 0) {
    fprintf(stderr, "libzeemote: HCI device open failed: %s(%d)\n", 
	    strerror(errno), errno);
    free(info);
    return result;
  }

  /* search list of all bluetooth devices found */
  for (i = 0; i < num_rsp; i++) {

    /* check for matching device class */
    if((info+i)->dev_class[0] == ((ZEEMOTE_CLASS >> 0) & 0xff) &&
       (info+i)->dev_class[1] == ((ZEEMOTE_CLASS >> 8) & 0xff) &&
       (info+i)->dev_class[2] == ((ZEEMOTE_CLASS >> 16) & 0xff)) {

      /* verify device name */
      if(hci_read_remote_name_with_clock_offset(dd,
		 &(info+i)->bdaddr,
		 (info+i)->pscan_rep_mode,
		 (info+i)->clock_offset | 0x8000,
		sizeof(name), name, 100000) >= 0) {

	if(strcmp(name, "Zeemote JS1") == 0) {
	  /* found a zeemote */

	  /* increase list size by one */
	  result->number_of_devices++;
	  result = realloc(result, sizeof(zeemote_scan_result_t) +
		   result->number_of_devices * sizeof(zeemote_device_t));

	  /* and save its type and address */
	  result->device[result->number_of_devices-1].type = 
	    ZEEMOTE_JS1; 
	  result->device[result->number_of_devices-1].bdaddr =
	    (info+i)->bdaddr; 
	}
      } else 
	fprintf(stderr, "libzeemote: read remote name failed: %s(%d)\n", 
		strerror(errno), errno);
    }
  }
  
  free(info);
  hci_close_dev(dd);

  /* return a list sorted by bdaddr, so the results are reproducable */
  qsort(result->device, result->number_of_devices, 
	sizeof(zeemote_device_t), zeemote_device_compare);

  return result;
}

#define ZEEMOTE_MAGIC  0xa1

typedef struct {
  unsigned char length;
  unsigned char magic;
  unsigned char type;
} zeemote_hdr_t;

void zeemote_set_state(zeemote_t *zeemote, int state, int req_lock) {
  if(req_lock)
    pthread_mutex_lock(&zeemote->mutex);

  zeemote->state[0].state = state;

  if(req_lock)
    pthread_mutex_unlock(&zeemote->mutex);
}

/* read a fixed number of bytes and don't accept less */
static ssize_t 
read_num(zeemote_t *zeemote, void *data, size_t count, int lock) {
  ssize_t total = 0;

  while(count) {
    ssize_t rd = read(zeemote->fd, data, count);
    if(rd < 0) {
      fprintf(stderr, "libzeemote: read failed: %s(%d)\n", 
	      strerror(errno), errno);

      zeemote_set_state(zeemote, ZEEMOTE_STATE_CONNECTION_LOST, lock);

      if(zeemote->fd >= 0) {
	close(zeemote->fd);
	zeemote->fd = -1;
      }

      return rd;
    } else {
      count -= rd;
      data += rd;
      total += rd;
    }
  }
  return total;
}

/* Byte order conversions */
#if __BYTE_ORDER == __LITTLE_ENDIAN
#define htozs(d)  bswap_16(d)
#define ztohs(d)  bswap_16(d)
#elif __BYTE_ORDER == __BIG_ENDIAN
#define htozs(d)  (d)
#define ztohs(d)  (d)
#else
#error "Unknown byte order"
#endif

#define ZEEMOTE_AXIS_UNKNOWN 0
#define ZEEMOTE_AXIS_X       1
#define ZEEMOTE_AXIS_Y       2

#define ZEEMOTE_BUTTONS 7
#define ZEEMOTE_STICK   8
#define ZEEMOTE_BATTERY 17

#define ZEEMOTE_BUTTON_NONE  (0xfe)

#if 0
/* hexdump for debug */
static void hexdump(void *buf, int size) {
  int n = 0, i, b2c;
  unsigned char *ptr = buf;
  
  if(!size) return;
  
  while(size>0) {
    printf("%04x: ", n);

    b2c = (size>16)?16:size;
    
    for(i=0;i<b2c;i++)
      printf("%02x ", 0xff & ptr[i]);
    
    printf("\n");
 
    ptr  += b2c;
    size -= b2c;
    n    += b2c;
 }
}
#endif

static void *zeemote_thread(void *data) {
  zeemote_t *zeemote = (zeemote_t*)data;

  zeemote_set_state(zeemote, ZEEMOTE_STATE_CONNECTING, 1);

  /* connect to zeemote */
  zeemote->fd = rfcomm_connect(&zeemote->bdaddr, 1);
  if(zeemote->fd < 0) {
    zeemote_set_state(zeemote, ZEEMOTE_STATE_CONNECTION_FAILED, 1);
    return NULL;
  }

  zeemote_set_state(zeemote, ZEEMOTE_STATE_CONNECTED, 1);

  while(zeemote->refcount && zeemote->fd >= 0) {
    zeemote_hdr_t hdr;

    union {
      signed char axis[3];
      unsigned char buttons[6];
      unsigned short voltage;
      char dummy[48];
    } data;

    fd_set rfds;
    FD_ZERO(&rfds);
    FD_SET(zeemote->fd, &rfds);
    struct timeval tv;
    tv.tv_sec = 0;
    tv.tv_usec = 100000;

    if(select(FD_SETSIZE, &rfds, NULL, NULL, &tv) < 0) {
      fprintf(stderr, "libzeemote: select() failed: %s(%d)\n", 
	      strerror(errno), errno);
    } else {
      if(FD_ISSET(zeemote->fd, &rfds)) {

	/* read any evaluate header */
	int rd = read_num(zeemote, &hdr, sizeof(hdr), 1);
	if(rd == sizeof(hdr)) {

	  if(hdr.magic == ZEEMOTE_MAGIC && hdr.length >= 2) {
	    pthread_mutex_lock(&zeemote->mutex);

	    switch(hdr.type) {
	      
	    case ZEEMOTE_STICK:
	      if(hdr.length-2 == sizeof(data.axis)) {
		if(read_num(zeemote, data.axis, sizeof(data.axis),0)) {
		  if(data.axis[ZEEMOTE_AXIS_UNKNOWN])
		    fprintf(stderr, "libzeemote: WARNING, ZEEMOTE_STICK "
			    "axis UNKNOWN != 0!\n");

		  zeemote->state[0].axis[0] = data.axis[ZEEMOTE_AXIS_X] * 256;
		  zeemote->state[0].axis[1] = data.axis[ZEEMOTE_AXIS_Y] * 256;
		} else 
		  fprintf(stderr, "libzeemote: reading ZEEMOTE_STICK "
			  "payload failed\n");
	      } else {
		fprintf(stderr, "libzeemote: unexpected length %d in "
			"ZEEMOTE_STICK\n", hdr.length);
		read_num(zeemote, data.dummy, hdr.length - 2, 0);
	      }
	      break;
	      
	    case ZEEMOTE_BATTERY:
	      if(hdr.length-2 == sizeof(data.voltage)) {
		if(read_num(zeemote, &data.voltage, sizeof(data.voltage), 0))
		  zeemote->state[0].battery = ztohs(data.voltage);
		else 
		  fprintf(stderr, "libzeemote: reading ZEEMOTE_BATTERY "
			  "payload failed\n");
	      } else {
		fprintf(stderr, "libzeemote: unexpected length %d "
			"in ZEEMOTE_BATTERY\n", hdr.length);
		read_num(zeemote, data.dummy, hdr.length - 2, 0);
	      }
	      break;
	      
	    case ZEEMOTE_BUTTONS:
	      if(hdr.length-2 == sizeof(data.buttons)) {
		if(read_num(zeemote, data.buttons, sizeof(data.buttons), 0)) {
		  unsigned int i;

		  zeemote->state[0].buttons = 0;		  
		  for(i=0;i<sizeof(data.buttons);i++)
		    if(data.buttons[i] != ZEEMOTE_BUTTON_NONE)
		      zeemote->state[0].buttons |= 1<<data.buttons[i];
		} else 
		  fprintf(stderr, "libzeemote: reading ZEEMOTE_BUTTONS "
			  "payload failed\n");
	      } else {
		fprintf(stderr, "libzeemote: unexpected length %d"
			" in ZEEMOTE_BUTTONS\n", 
			hdr.length);
		read_num(zeemote, data.dummy, hdr.length - 2, 0);
	      }
	      break;
	      
	    default:
	      if((size_t)hdr.length - 2 > sizeof(data.dummy)) 
		fprintf(stderr, "libzeemote: data length %d too big\n", 
			hdr.length - 2);
	      
#if 0
	      printf("libzeemote: skipping %d bytes of unknown command %d\n", 
		     hdr.length-2, hdr.type);
	      read_num(zeemote, data.dummy, hdr.length - 2, 0);
	      hexdump(data.dummy, hdr.length - 2);
#else
	      read_num(zeemote, data.dummy, hdr.length - 2, 0);
#endif
	      break;
	    }
	    pthread_mutex_unlock(&zeemote->mutex);
	  }
	}
      }
    }
  }

  return NULL;
}

zeemote_t *zeemote_connect(bdaddr_t *bdaddr) {
  zeemote_t *zeemote = NULL, **zeemoteP = &zeemote_list;
  while(*zeemoteP && bacmp(&(*zeemoteP)->bdaddr, bdaddr))
    zeemoteP = &(*zeemoteP)->next;

  /* use existing zeemote entry if present */
  if(*zeemoteP) {
    zeemote = *zeemoteP;
    zeemote->refcount++;
    return zeemote;
  }

  zeemote = *zeemoteP = malloc(sizeof(zeemote_t));
  memset(zeemote, 0, sizeof(zeemote_t));
  zeemote->bdaddr = *bdaddr;
  zeemote->fd = -1;
  zeemote->refcount = 1;

  /* currently only the zeemote js1 is supported */
  zeemote->info.type = ZEEMOTE_JS1;
  zeemote->info.num_axes = 2;
  zeemote->info.num_buttons = 4;  

  zeemote->state[0].state = ZEEMOTE_STATE_UNKNOWN;

  pthread_mutex_init(&zeemote->mutex, NULL);

  /* fork a handler thread */
  if ( pthread_create(&zeemote->thread, NULL, 
		      zeemote_thread, (void *)zeemote) != 0 ) {
    fprintf(stderr, "libzeemote: Creation of thread failed: %s(%d)\n", 
	    strerror(errno), errno);
    free(zeemote);
    *zeemoteP = NULL;
    return NULL;
  }

  return zeemote;
}

void zeemote_disconnect(zeemote_t *zeemote) {
  zeemote_t **zeemoteP = &zeemote_list;
  while(*zeemoteP && *zeemoteP != zeemote)
    zeemoteP = &(*zeemoteP)->next;

  if(!*zeemoteP) {
    fprintf(stderr, "libzeemote: illegal reference\n");
    return;
  }
  
  zeemote->refcount--;
  if(!zeemote->refcount) {

    /* ... wait for thread to end ... */
    if( pthread_join(zeemote->thread, NULL) < 0)
      fprintf(stderr, "libzeemote: joining of thread failed: %s(%d)\n", 
	      strerror(errno), errno);

    /* close connection ... */
    if(zeemote->fd >= 0) {
      close(zeemote->fd);
      zeemote->fd = -1;
    }

    /* ... and remove entry from chain */
    *zeemoteP = zeemote->next;

    free(zeemote);
  }
}

zeemote_state_t *zeemote_get_state(zeemote_t *zeemote) {
  /* fetch state from thread area into application area */
  pthread_mutex_lock(&zeemote->mutex);
  memcpy(&zeemote->state[1], &zeemote->state[0], sizeof(zeemote_state_t));
  pthread_mutex_unlock(&zeemote->mutex);

  return &zeemote->state[1];
}
