/*
 * Copyright (C) 2009 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <utime.h>
#include "thumbtest.h"
#include <QFile>
#include <QUrl>
#include <QDebug>
#include <QDir>
#include <QFile>
#include <QSignalSpy>
#include <QTimer>

using namespace MediaArt;
using namespace Thumbnails;

/*
 wait until spy will reach target number of signals; after that number is reached
 we can continue with verifying test's results. Thanks to that we do not need to wait always
 long time to be sure that tumbler replied - just give there finished spy and that should be
 working.

 maxTime is the maximum time we want to wait.

 NOTE: that macro could be placed only once in every block of the code and it is supposed
       to be put at max. only once in every test slot!
 NOTE: as a maxTime You need to put positive number
 */ 
#define WAIT_FOR_FINISH(spy,target,maxTime) static int _counter = 0; \
                                            while(spy.count()<target && _counter <= maxTime/50) { \
                                                  ++_counter; QTest::qWait(50); \
                                            }

static bool createFile(QString fileName) {
	if(QFile::exists(fileName)) return true;
	QFile file(fileName);
	bool created = file.open(QIODevice::WriteOnly | QIODevice::Text);
	if(created) {
		file.write("test-data");
		file.close();
	}
	return created;
}
static bool deleteFile(QString fileName) {
	return QFile::remove(fileName);
}

void ThumbTest::initTestCase() {
	qRegisterMetaType<MediaArt::Info>();
}

void ThumbTest::emptyTest_checkIfQTestLibIsWorking() {
	QCOMPARE(0, 0);
	QCOMPARE(NULL, NULL);
	QVERIFY(1 + 1 == 2);
}

void ThumbTest::Thumbnailer_getThumbnailPath_direct_data() {
	QTest::addColumn<QString>("flavor");
	QTest::addColumn<QString>("pattern");
	QTest::addColumn<QUrl>("image");

	QTest::newRow("data #1") << QString("normal") << QString("/.*/.thumbnails/normal/3c648bdaf1b59ec7b205b4900317fa73.jpeg") << QUrl("file:///home/user/tmp.jpeg");
	QTest::newRow("data #2") << QString("preview") << QString("/.*/.thumbnails/preview/3c648bdaf1b59ec7b205b4900317fa73.jpeg") << QUrl("file:///home/user/tmp.jpeg");
	QTest::newRow("data #3") << QString("normal") << QString("/.*/.thumbnails/normal/283b1c0489ac5a3ac8f1044c371a81d7.jpeg") << QUrl("file:///home/user/different.jpeg");
	QTest::newRow("data #4") << QString("preview") << QString("/.*/.thumbnails/preview/283b1c0489ac5a3ac8f1044c371a81d7.jpeg") << QUrl("file:///home/user/different.jpeg");
	QTest::newRow("data #5") << QString("fullscreen") << QString("/.*/.thumbnails/fullscreen/3f548d80f8bb27d373a0510ff27a77f2.jpeg") << QUrl("file:///home/bubu.avi");
	QTest::newRow("data #6") << QString("small") << QString("/.*/.thumbnails/small/3f548d80f8bb27d373a0510ff27a77f2.jpeg") << QUrl("file:///home/bubu.avi");
}

void ThumbTest::Thumbnailer_getThumbnailPath_direct() {
	QFETCH(QString, flavor);
	QFETCH(QString, pattern);
	QFETCH(QUrl,    image);

	QUrl path = Thumbnailer::getThumbnailPath(image, flavor);
	QVERIFY( path.toString().contains(QRegExp(pattern)) );
}

void
ThumbTest::Thumbnailer_objectCreation() {
	Thumbnailer thumb;
	// check if we can access anything from inside
	QCOMPARE(thumb.defaultURI(), QUrl());
}

void
ThumbTest::Thumbnailer_objectCreation_new() {
	Thumbnailer* thumb;
	thumb = new Thumbnailer();
	QVERIFY(NULL != thumb);
	QCOMPARE(thumb->defaultURI(), QUrl());

	delete thumb;
}

#ifndef DATA_PREFIX
#define DATA_PREFIX "/usr/share/libthumbnailer0-test/"
#endif

void
ThumbTest::Thumbnailer_objectDestruction() {
	Thumbnailer* thumb = new Thumbnailer();

	QList<QUrl> images; images << QString("file://") + DATA_PREFIX "test.jpg";;
	QStringList mimes;  mimes  << QString("image/jpeg");

	thumb->request (images, mimes);
	thumb->request (images, mimes);
	thumb->request (images, mimes);

	delete thumb;
	// have to find out how to test if everything was deleted properly
	// for this moment test passes when there is no segmentation fault
	// ort memmory corruption during normal destructing the object
}


void
ThumbTest::Thumbnailer_objectCreation_defaultUrl() {
	const QUrl image("file:///home/user/image.jpeg");
	Thumbnailer thumb(image);

	QCOMPARE(thumb.defaultURI(), image);
}

void
ThumbTest::Thumbnailer_objectCreation_defaultParameters() {
	const QUrl    image  ("file:///home/user/image.jpeg");
	const QPixmap pixmap (QSize(43, 85));

	Thumbnailer thumb(image, pixmap);

	QCOMPARE(thumb.defaultURI(), image);
	QCOMPARE(thumb.defaultPixmap().size(), pixmap.size());
}

void
ThumbTest::setDefaultPixmap_setting() {
	Thumbnailer thumb;
	const QPixmap pixmap (QSize(43, 85));

	thumb.setDefaultPixmap(pixmap);

	QCOMPARE(thumb.getPrivateData()->defaultPixmap.size(), pixmap.size());
}

void
ThumbTest::defaultPixmap_getting() {
	Thumbnailer thumb;
	const QPixmap pixmap (QSize(43, 85));

	thumb.getPrivateData()->defaultPixmap = pixmap;

	QCOMPARE(thumb.defaultPixmap().size(), pixmap.size());
}

void
ThumbTest::setDefaultURI_setting() {
	Thumbnailer thumb;
	const QUrl url ("http://something/path");

	thumb.setDefaultURI(url);

	QCOMPARE(thumb.getPrivateData()->defaultUri, url);
}

void
ThumbTest::defaultURI_getting() {
	Thumbnailer thumb;
	const QUrl url ("http://something/path");

	thumb.getPrivateData()->defaultUri = url;

	QCOMPARE(thumb.defaultURI(), url);
}

/****************************************************************************
 *     TESTS - DUI MEDIA ART
 ****************************************************************************/

	void ThumbTest::MediaArt_basicStaticType() {
		QCOMPARE(Info::Type, QString("unknown"));
	}
	void ThumbTest::MediaArt_basicConstructor() {
		Info a();
	}
	void ThumbTest::MediaArt_basicGettingDefaultType() {
		Info a("A", "B");
		QCOMPARE(a.type(), MediaArt::Album::Type);
	}
	void ThumbTest::MediaArt_basicGettingType() {
		Info a("A", "B", "other");
		QCOMPARE(a.type(), QString("other"));
	}
	/*******************************************
	  all target md5 sums calculation done via http://md5online.110mb.com/
	  ******************************************/
	void ThumbTest::MediaArt_basicCalculatingPath_data() {
		QTest::addColumn<QString>("first");
		QTest::addColumn<QString>("second");
		QTest::addColumn<QString>("type");
		QTest::addColumn<QString>("result");

		QTest::newRow("data #1") << QString("a") << QString("b") << QString("other") <<
		QString("other-7215ee9c7d9dc229d2921a40e899ec5f-92eb5ffee6ae2fec3ad71c777531578f.jpeg");
		QTest::newRow("data #2") << QString("A") << QString("B") << QString("test") <<
		QString("test-7215ee9c7d9dc229d2921a40e899ec5f-92eb5ffee6ae2fec3ad71c777531578f.jpeg");
		QTest::newRow("data #3") << QString("first") << QString() << QString("ala") <<
		QString("ala-7215ee9c7d9dc229d2921a40e899ec5f-7215ee9c7d9dc229d2921a40e899ec5f.jpeg");
		QTest::newRow("data #4") << QString("") << QString() << QString() <<
		QString("album-7215ee9c7d9dc229d2921a40e899ec5f-7215ee9c7d9dc229d2921a40e899ec5f.jpeg");
	}
	void ThumbTest::MediaArt_basicCalculatingPath() {
		QFETCH(QString, first);
		QFETCH(QString, second);
		QFETCH(QString, type);
		QFETCH(QString, result);

		Info a(first, second, type);
		//qDebug() << "Potential:" << a.potentialPath() << ", current:" << result;
		QVERIFY(a.potentialPath().toString().endsWith(result));
	}
	/*******************************************
	  check http://live.gnome.org/MediaArtStorageSpec for info about what should be removed
	  ******************************************/
	void ThumbTest::MediaArt_basicStrippingOutInvalidEntities_data() {
		QTest::addColumn<QString>("src");
		QTest::addColumn<QString>("result");

		QTest::newRow("data #01") << QString("") << QString(" ");
		QTest::newRow("data #02") << QString() << QString(" ");
		QTest::newRow("data #03") << QString(" ") << QString(" ");
		QTest::newRow("data #04") << QString("a") << QString("a");
		QTest::newRow("data #05") << QString("AAA") << QString("aaa");
		QTest::newRow("data #06") << QString("AbbB") << QString("abbb");
		QTest::newRow("data #07") << QString("  alka ") << QString("alka");
		QTest::newRow("data #08") << QString("long    spaCe") << QString("long space");
		QTest::newRow("data #09") << QString("\t\ttabs\t  ") << QString("tabs");
		QTest::newRow("data #10") << QString(")(][><}{_!@#$^&*+=|\\/\"'?~`") << QString(" ");
		QTest::newRow("data #11") << QString("brackets(ala) a") << QString("brackets a");
		QTest::newRow("data #12") << QString("brackets{ala} a") << QString("brackets a");
		QTest::newRow("data #13") << QString("brackets[ala] a") << QString("brackets a");
		QTest::newRow("data #14") << QString("brackets<ala> a") << QString("brackets a");
		QTest::newRow("data #15") << QString(")a(b]c[d>e<f}g{h_i!j@k#l$m^n&o*p+r=s|t\\u/v\"x'y?z~q`") << QString("abcdefghijklmnoprstuvxyzq");
	}
	void ThumbTest::MediaArt_basicStrippingOutInvalidEntities() {
		QFETCH(QString, src);
		QFETCH(QString, result);

		QCOMPARE(Info::stripInvalidEntities(src), result);
	}
	void ThumbTest::MediaArt_basicEqualOperator_data() {
		QTest::addColumn<Info>("left");
		QTest::addColumn<Info>("right");
		QTest::addColumn<bool>("equals");

		QTest::newRow("data #01") << Info() << Info() << true;
		QTest::newRow("data #02") << Info() << Info("","") << true;
		QTest::newRow("data #03") << Info() << Info(" ","  (to-strip-anyway){}[]<>#$") << true;
		QTest::newRow("data #04") << Info("abb") << Info("AbB") << true;
		QTest::newRow("data #05") << Info("a","b","test") << Info("A","B","test") << true;
		QTest::newRow("data #06") << Info("a","b") << Info("b","a") << false;
	}
	void ThumbTest::MediaArt_basicEqualOperator() {
		QFETCH(Info, left);
		QFETCH(Info, right);
		QFETCH(bool, equals);

		QCOMPARE(left == right, equals);
		QCOMPARE(right == left, equals);
	}
	void ThumbTest::MediaArt_basicCooperationWithQHash() {
		// create couple of Info objects (we will insert them into the hash)
		Info      obj1("B", "C", "other");
		MediaArt::Album  obj2("B", "C");
		MediaArt::Artist obj3("B");
		// non inserted later into the hash
		Info      obj4("B", "C", "other");
		Info      obj5("  B", "c", "album");
		Info      obj6("B", "d", "album");
		Info      obj7("B", "", "artist");

		QHash<Info, int> hash;
		hash.insert(obj1, 1);
		hash[obj2] = 2;
		hash[obj3] = 3;

		QCOMPARE(hash.contains(obj1), true);
		QCOMPARE(hash.contains(obj2), true);
		QCOMPARE(hash.contains(obj3), true);
		QCOMPARE(hash.contains(obj4), true);
		QCOMPARE(hash.contains(obj5), true);
		QCOMPARE(hash.contains(obj6), false);
		QCOMPARE(hash.contains(obj7), true);

		QCOMPARE(hash[obj4], 1);
		QCOMPARE(hash[obj5], 2);
		QCOMPARE(hash[obj6], 0);
		QCOMPARE(hash[obj7], 3);
	}
	void ThumbTest::MediaArt_basicCooperationWithQList() {
		// create couple of Info objects (we will insert them into the hash)
		Info      obj1("B", "C", "other");
		MediaArt::Album  obj2("B", "C");
		MediaArt::Artist obj3("B");
		// non inserted later into the hash
		Info      obj4("B", "C", "other");
		Info      obj5("  B", "c", "album");
		Info      obj6("B", "d", "album");
		Info      obj7("B", "", "artist");

		QList<Info> list;
		list << obj1;
		list.append(obj2);
		list.insert(0, obj3);

		QCOMPARE(list.count(), 3);
		QCOMPARE(list.contains(obj1), QBool(true));
		QCOMPARE(list.contains(obj2), QBool(true));
		QCOMPARE(list.contains(obj3), QBool(true));
		QCOMPARE(list.contains(obj4), QBool(true));
		QCOMPARE(list.contains(obj5), QBool(true));
		QCOMPARE(list.contains(obj6), QBool(false));
		QCOMPARE(list.contains(obj7), QBool(true));

		QVERIFY(list.at(0) == obj3);
		QVERIFY(list.at(1) == obj1);
		QVERIFY(list.at(2) == obj2);
	}

	void ThumbTest::MediaArt_basicSettingContentPath() {
		QString fileName("example.jpeg");
		QUrl uri(QDir::currentPath() + QDir::separator() + fileName);
		uri.setScheme("file");
		createFile(fileName);

		MediaArt::Album a("foobar", "foo-bar-foo");
		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);

		bool success = a.setMediaArtImage(uri, false);
		QCOMPARE(success, true);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);

		success = a.setMediaArtImage(uri, false);
		QCOMPARE(success, false);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);

		deleteFile(fileName);
		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);
	}
	void ThumbTest::MediaArt_basicSettingContentPath_wrongSchema() {
		QUrl uri("http://www.foobar.com");
		MediaArt::Album a("foobar", "foo-bar-foo");

		a.remove();
		bool success = a.setMediaArtImage(uri, true);
		QCOMPARE(success, false);
		QCOMPARE(a.exists(), false);
	}
	void ThumbTest::MediaArt_basicSettingContentPath_wrongExtension() {
		QString fileName("example.jpeg.png");
		QUrl uri(QDir::currentPath() + QDir::separator() + fileName);
		uri.setScheme("file");
		createFile(fileName);

		MediaArt::Album a("foobar", "foo-bar-foo");

		a.remove();
		bool success = a.setMediaArtImage(uri, true);
		QCOMPARE(success, false);
		QCOMPARE(a.exists(), false);
	}
	void ThumbTest::MediaArt_basicSettingContentPath_overwrite() {
		QString fileName("example.jpeg");
		QUrl uri(QDir::currentPath() + QDir::separator() + fileName);
		uri.setScheme("file");
		createFile(fileName);

		MediaArt::Album a("foobar", "foo-bar-foo");
		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);

		bool success = a.setMediaArtImage(uri, true);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);
		QCOMPARE(success, true);

		success = a.setMediaArtImage(uri, true);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);
		QCOMPARE(success, true);

		deleteFile(fileName);
		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);
	}
	void ThumbTest::MediaArt_basicSettingContentPixmap() {
		QPixmap pixmap(QSize(53,48));

		MediaArt::Album a("foobar", "foo-bar-foo");
		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);

		bool success = a.setMediaArtImage(pixmap, false);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);
		QCOMPARE(success, true);

		success = a.setMediaArtImage(pixmap, false);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);
		QCOMPARE(success, false);

		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);
	}
	void ThumbTest::MediaArt_basicSettingContentPixmap_overwrite() {
		QPixmap pixmap(QSize(53,48));

		MediaArt::Album a("foobar", "foo-bar-foo");
		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);

		bool success = a.setMediaArtImage(pixmap, true);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);
		QCOMPARE(success, true);

		success = a.setMediaArtImage(pixmap, true);
		QCOMPARE(QFile::exists(a.potentialPath().path()), true);
		QCOMPARE(success, true);

		deleteFile(a.potentialPath().path());
		QCOMPARE(QFile::exists(a.potentialPath().path()), false);
	}
	void ThumbTest::MediaArt_basicRemoveingContent() {
		MediaArt::Album a("foobar", "foo-bar-foo");
		a.setMediaArtImage(QPixmap(QSize(100,50)));
		QCOMPARE(a.exists(), true);
		a.remove();
		QCOMPARE(a.exists(), false);
	}

	void ThumbTest::MediaArt_albumConstructor() {
		MediaArt::Album a("Queen", "Greatest Hits vol. 1");
	}
	void ThumbTest::MediaArt_albumConstructorBasicCasting() {
		Info a = MediaArt::Album("Queen", "Greatest Hits vol. 1");

		//QCOMPARE(a._b, QString("Queen"));
		//QCOMPARE(a._c, QString("Greatest Hits vol. 1"));
		QCOMPARE(a.type(), MediaArt::Album::Type);
	}
	void ThumbTest::MediaArt_albumStaticType() {
		QCOMPARE(MediaArt::Album::Type, QString("album"));
	}
	void ThumbTest::MediaArt_albumGettingAlbum() {
		MediaArt::Album a("Queen", "Greatest Hits vol. 1");
		QCOMPARE(a.album(), QString("Greatest Hits vol. 1"));
	}
	void ThumbTest::MediaArt_albumGettingArtist() {
		MediaArt::Album a("Queen", "Greatest Hits vol. 1");
		QCOMPARE(a.artist(), QString("Queen"));
	}
	void ThumbTest::MediaArt_albumGettingType() {
		MediaArt::Album a("A", "B");
		QCOMPARE(a.type(), MediaArt::Album::Type);
	}
	void ThumbTest::MediaArt_albumCalculatingPotentialPath() {
		MediaArt::Album a("A", "B");
		QVERIFY(a.potentialPath().toString().contains(QRegExp(".*album-[a-f,0-9]{32}-[a-f,0-9]{32}.jpeg")));
	}

	void ThumbTest::MediaArt_artistConstructor() {
		MediaArt::Artist a("Queen");
	}
	void ThumbTest::MediaArt_artistStaticType() {
		QCOMPARE(MediaArt::Artist::Type, QString("artist"));
	}
	void ThumbTest::MediaArt_artistGettingArtist() {
		MediaArt::Artist a("Queen");
		QCOMPARE(a.artist(), QString("Queen"));
	}
	void ThumbTest::MediaArt_artistGettingAlbum() {
		MediaArt::Artist a("Queen", "Greatest Hits vol. 1");
		QCOMPARE(a.album(), QString("Greatest Hits vol. 1"));
	}
	void ThumbTest::MediaArt_artistGettingType() {
		MediaArt::Artist a("2Pac");
		QCOMPARE(a.type(), MediaArt::Artist::Type);
	}
	void ThumbTest::MediaArt_artistCalculatingPotentialPath() {
		MediaArt::Artist a("2Pac");
		QVERIFY(a.potentialPath().toString().contains(QRegExp(".*artist-[a-f,0-9]{32}-[a-f,0-9]{32}.jpeg")));
	}

	void ThumbTest::MediaArt_trackCalculatingPotentialPath() {
		MediaArt::Track t("Queen", "Greatest hits vol.1", "Want to break free");
		QVERIFY(t.potentialPath().toString().contains(QRegExp(".*track-[a-f,0-9]{32}-[a-f,0-9]{32}-[a-f,0-9]{32}.jpeg")));
	}

	void ThumbTest::MediaArt_requesterSimpleRequestOneExistingMediaArt() {
		MediaArt::Requester* requester = new MediaArt::Requester();

		QSignalSpy ready(requester, SIGNAL(mediaArt(MediaArt::Info,QUrl,QPixmap)));
		QSignalSpy readyDefault(requester, SIGNAL(defaultMediaArt(MediaArt::Info,QUrl,QPixmap)));
		QSignalSpy error(requester, SIGNAL(error(QString,MediaArt::Info)));
		QSignalSpy stop(requester, SIGNAL(finished()));
		QSignalSpy start(requester, SIGNAL(started()));

		MediaArt::Album album("Queen", "Greatest hits vol.1");
		QList<Info> media; media << album;

		// make sure that file exists
		album.setMediaArtImage(QPixmap(QSize(100,100)), true);
		bool noDBusTraffic = requester->request(media);

		WAIT_FOR_FINISH(stop, 1, 100);
		QCOMPARE(noDBusTraffic, true);
		QCOMPARE(error.count(), 0);
		QCOMPARE(readyDefault.count(), 0);
		QCOMPARE(ready.count(), 1);
		QCOMPARE(stop.count(), 1);
		QCOMPARE(start.count(), 1);

		deleteFile(album.potentialPath().path());
		delete requester;
	}
	void ThumbTest::MediaArt_requesterSimpleRequestOneNonExistingMediaArt() {
		MediaArt::Requester* requester = new MediaArt::Requester();

		QSignalSpy ready(requester, SIGNAL(mediaArt(const MediaArt::Info&,const QUrl&,const QPixmap&)));
		QSignalSpy readyDefault(requester, SIGNAL(defaultMediaArt(MediaArt::Info,QUrl,QPixmap)));
		QSignalSpy error(requester, SIGNAL(error(QString,MediaArt::Info)));
		QSignalSpy stop(requester, SIGNAL(finished()));
		QSignalSpy start(requester, SIGNAL(started()));

		MediaArt::Album album("Queen", "Greatest hits vol.1");
		QList<Info> media; media << album;

		// make sure that file does not exist
		deleteFile(album.potentialPath().path());
		bool noDBusTraffic = requester->request(media);

		WAIT_FOR_FINISH(stop, 1, 100);
		// TODO - need to update condition after some Media Art Service become available
		QCOMPARE(noDBusTraffic, false);
		QCOMPARE(error.count(), 1);
		QCOMPARE(readyDefault.count(), 1);
		QCOMPARE(ready.count(), 0);
		QCOMPARE(stop.count(), 1);
		QCOMPARE(start.count(), 1);

		deleteFile(album.potentialPath().path());
	}
	void ThumbTest::MediaArt_requesterSimpleRequestCoupleMixedMediaArt() {
		MediaArt::Requester* requester = new MediaArt::Requester();

		QSignalSpy ready(requester, SIGNAL(mediaArt(MediaArt::Info,QUrl,QPixmap)));
		QSignalSpy readyDefault(requester, SIGNAL(defaultMediaArt(MediaArt::Info,QUrl,QPixmap)));
		QSignalSpy error(requester, SIGNAL(error(QString,MediaArt::Info)));
		QSignalSpy stop(requester, SIGNAL(finished()));
		QSignalSpy start(requester, SIGNAL(started()));

		MediaArt::Album   album("Queen", "Greatest hits vol.1");
		MediaArt::Artist  artist("Queen", "Greatest hits vol.1");
		MediaArt::Podcast podcast("Funky London House");
		MediaArt::Radio   radio("Radio Zet");
		MediaArt::Track   track("Queen", "Greatest hits vol.1", "I want to break free");

		QList<Info> media;
		media << album << artist << podcast << radio << track;

		// make sure that some files exists and some not (3 exists, 2 not)
		album.setMediaArtImage(QPixmap(QSize(100,100)), true);
		podcast.setMediaArtImage(QPixmap(QSize(100,100)), true);
		track.setMediaArtImage(QPixmap(QSize(100,100)), true);
		radio.remove();
		artist.remove();
		// make a request
		bool noDBusTraffic = requester->request(media);

		WAIT_FOR_FINISH(stop, 1, 500);
		QCOMPARE(noDBusTraffic, false);
		QCOMPARE(error.count(), 2);
		QCOMPARE(readyDefault.count(), 2);
		QCOMPARE(ready.count(), 3);
		QCOMPARE(stop.count(), 1);
		QCOMPARE(start.count(), 1);

		deleteFile(album.potentialPath().path());
		deleteFile(artist.potentialPath().path());
		deleteFile(radio.potentialPath().path());
		deleteFile(podcast.potentialPath().path());
		deleteFile(track.potentialPath().path());
		delete requester;
	}

/****************************************************************************
 *     TESTS - DUI MEDIA ART THUMBNAILER
 ****************************************************************************/
// execute all tests
QTEST_MAIN(ThumbTest)

