/*
 * Copyright (C) 2009 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/*
 * thumbnailer_p.h
 *
 *  Created on: Oct 27, 2009
 *     Authors: iridian, wiecheck
 */

#ifndef THUMBNAILER_P_H_
#define THUMBNAILER_P_H_

#include "include/thumbnailer.h"
#include "dbus/thumbnailer_generic.h"
#include "dbus/tumbler_cache.h"
#include <QReadWriteLock>
#include <QReadLocker>
#include <QWriteLocker>
#include <QMutex>
#include <QMutexLocker>

using namespace Thumbnails;
using namespace MediaArt;

// internal global data and default values
static QString _defaultScheduler        = QString("foreground");
static QString _defaultFlavor           = QString("normal");
static QString _thumbnailsCacheDir      = QDir::separator() +
                                          QString(".thumbnails") +
                                          QDir::separator();

// define how often timer will increase 
static const int __intervalDuration = 3000;
// define how many times it is possible to increase intervalsSinceLastAction
// counter before thumbnailer will treat particular request to be timeouted
// after that it will be removed from requests list
static const int __maxIntervalsBeforeTimeout = 5;

/* this structure will be created for every single request() call and will be
 * connected with one dbus call to tumbler. it contains all uris and
 * parameters of request
 */
struct ServiceRequest {
	Thumbnailer* parent;
	bool         sendPixmap;
	QString      flavor;
	QStringList  uris;
	QList<Info>  media;
	QStringList  mimes;
	long         intervalsSinceLastAction;

	ServiceRequest (Thumbnailer* parent,
	                bool         sendPixmap,
	                QString      flavor) :
	                    parent(parent),
	                    sendPixmap(sendPixmap),
	                    flavor(flavor),
	                    uris(QStringList()),
	                    media(QList<Info>()),
	                    mimes(QStringList()),
	                    intervalsSinceLastAction(0) {};
};

namespace Thumbnails {

/* 
 *
 */
class ThumbnailerPrivate : public QObject {
Q_OBJECT
public:
	ThumbnailerPrivate() : QObject(), timer(0) {};
	~ThumbnailerPrivate() {};
	Thumbnailer* parent;

	QUrl    defaultUri;
	QPixmap defaultPixmap;

	// list of all requests
	QHash<uint, ServiceRequest*> requests;
	QHash<QDBusPendingCallWatcher*, ServiceRequest*> waitingForReply;

	// starts timer if there is any request running
	void timerStart ();
	// stops the timer if there is no request running (no need for timer)
	void timerStop  ();
	// makes a request directly to the tumbler
	bool makeRequest ( ServiceRequest* request,
	                   const QString& flavor,
	                   const QString& scheduler );
	void timerForcedStop  ();

public Q_SLOTS:
	// slots called when dbus connection had to be reseted - need to connect
	// signals to proper GenericProxy object
	void genericProxyChanged();
	// watcher for dbus call reply for request() function
	void requestCallFinished(QDBusPendingCallWatcher* watcher);

	// slots which should be connected to Thumbnailer Proxy object's signals
	void StartedHandler (uint handle);
	void ReadyHandler   (uint handle, const QStringList &uris);
	void ErrorHandler   ( uint handle,
	                      const QStringList &failedUris,
	                      int errorCode,
	                      const QString &message );
	void FinishedHandler(uint handle);
protected:
	// keeps id of the timer
	int timer;
	// check if there is no request which is 'timeout-ed'
	void timerEvent (QTimerEvent* event);
	// check particular request if it did not 'hang'
	bool processServiceRequest ( ServiceRequest* request,
	                             bool forceTimeout = false );
private:
	// checks if particular signal is dedicated for this instance of 
	// ThumbnailerPrivate object
	inline bool isThisForMe(uint handle);

#ifdef TESTING_CODE
Q_SIGNALS:
	void remoteStarted (uint handle);
	friend class ThumbTest;
#endif
};

} // end of namespace

#endif
