#include "include/mediaartinfo.h"
#include <QString>
#include <QDir>
#include <QUrl>
#include <QByteArray>
#include <QCryptographicHash>
#include <QDebug>
#include <QPixmap>
#include <QTemporaryFile>

using namespace MediaArt;

static const QString separator = "-";
static const QString fileExtension = ".jpeg";
static const QString basePath = QDir::homePath() + QDir::separator() +
				".cache"    + QDir::separator() +
				"media-art" + QDir::separator();
static const QString fileScheme = "file";


static QString _calculateMD5(const QString& src) {
	QByteArray md5 = QCryptographicHash::hash ( src.toUtf8 (),
						    QCryptographicHash::Md5 );

	return QString(md5.toHex());
}

const QString Info::Type = "unknown";
Info::Info ( const QString& b,
             const QString& c,
             const QString& type) :
             _b(b), _c(c), _d(QString()), _type(type)
{
	if(_type.isEmpty()) _type = Album::Type;
	updatePath();
}

Info::Info(const MediaArt::Info& src) :
      _b(src._b), _c(src._c), _d(src._d), _type(src._type), _path(src._path)
{ }

Info::~Info() {}
uint MediaArt::qHash(MediaArt::Info const &mai) {
	return qHash(mai._path.toString());
}
bool MediaArt::operator==(const MediaArt::Info &mai1, const MediaArt::Info &mai2)
{
	return MediaArt::qHash(mai1) == MediaArt::qHash(mai2);
}
QDebug MediaArt::operator<<(QDebug dbg, const MediaArt::Info& mai) {
	dbg.nospace() << "\n";
	dbg.nospace() << "  Media Art Information (type = " << mai._type << "):\n";
	dbg.nospace() << "       Main data: " << mai._b << "\n";
	dbg.nospace() << "  Secondary data: " << mai._c << "\n";
	dbg.nospace() << " Calculated path: " << mai._path.toString() << "\n";
	return dbg;
}

QString Info::md5(const QString& src) {
	return _calculateMD5(src);
}

QString Info::stripInvalidEntities(const QString& src) {
	QString sourcei(src);
	QString source = sourcei.normalized (QString::NormalizationForm_KD);
	if(source.isEmpty()) return QString(" ");
	// http://live.gnome.org/MediaArtStorageSpec
	// remove blocks and special characters
	QRegExp removeblocks( "(\\([^\\)]*\\))"
			     "|(\\[[^\\]]*\\])"
			     "|(\\{[^\\}]*\\})"
			     "|(\\<[^\\>]*\\>)");
	source.replace(removeblocks, "");

	// remove special characters
	source.remove(QRegExp("[\\(\\)\\_\\{\\}\\[\\]\\!\\@\\#\\$\\^\\&\\*\\+\\=\\|\\\\\\/\\\"\\'\\?\\<\\>\\~\\`]+"));

	// replace tabs with spaces
	source.replace('\t', " ");

	// replace space sequences with single spaces
	source.replace(QRegExp("\\s+"), " ");

	// remove leading and trailing spaces
	source.remove(QRegExp("(^\\s+)|(\\s+$)"));

	if(source.isEmpty()) return QString(" ");
	return source.toLower();
}

//void Info::updatePath() {
//	_path = QUrl( basePath + _type + separator +
//		      md5(stripInvalidEntities(_b)) + separator +
//		      md5(stripInvalidEntities(_c)) + fileExtension );
//	_path.setScheme(fileScheme);
//}

void Info::updatePath() {
	_path = QUrl( basePath + _type + separator +
		      md5(QString (" ")) + separator + // Always use " " for Artist for NB#275719
		      md5(stripInvalidEntities(_c)) + fileExtension );
	_path.setScheme(fileScheme);
}

const QString& Info::type (void) const {
	return _type;
}
const QUrl& Info::potentialPath() const {
	return _path;
}
bool Info::exists() const {
	// file must exist and be readable
	return QFile::exists( this->_path.path()) &&
	       QFileInfo(this->_path.path()).isReadable();
}
bool Info::remove() const {
	return QFile::remove(_path.path());
}
bool Info::setMediaArtImage ( const QUrl& source,
                                           bool overwrite) const
{
	// we support only local files
	if(fileScheme != source.scheme()) return false;
	// we support only jpeg files
	QRegExp regexp(".(jpg|jpeg)$", Qt::CaseInsensitive);
	if(!source.path().contains(regexp)) return false;

	if(overwrite) QFile::remove(_path.path());
	// make sure that proper directories are there
	QString mediaArtPath = QString(".cache") + QDir::separator() + QString("media-art");
	if( !QDir::home().exists(mediaArtPath) &&
	    !QDir::home().mkpath(mediaArtPath) )
	{
		return false;
	}
	return QFile::copy(source.path(), _path.path());
}
bool Info::setMediaArtImage ( const QPixmap& source,
                                           bool overwrite ) const
{
	// if overwrite - delet the old image
	if(overwrite) {
		QFile::remove(_path.path());
	}
	// make sure that proper directories are there
	QString mediaArtPath = QString(".cache") + QDir::separator() + QString("media-art");
	if( !QDir::home().exists(mediaArtPath) &&
	    !QDir::home().mkpath(mediaArtPath) )
	{
		return false;
	}
	// create new temporary file
	QTemporaryFile temp(basePath + QString("temp-"));
	// we need to open and close the file in order to create it and have file name
	if(!temp.open()) {
		return false;
	}
	temp.close();
	// try to save the image into the new file (in jpeg file format)
	if(!source.save(temp.fileName(), "JPEG")) {
		return false;
	}

	return QFile::rename(temp.fileName(), _path.path());
}

const QString Album::Type = "album";
Album::Album(const QString& artist, const QString& album) :
		Info(artist, album, Album::Type)
{}
const QString& Album::artist() const {
	return _b;
}
const QString& Album::album() const {
	return _c;
}

const QString Artist::Type = "artist";
Artist::Artist(const QString& artist, const QString& album) :
		Info(artist, album, Artist::Type)
{}
const QString& Artist::artist() const {
	return _b;
}
const QString& Artist::album() const {
	return _c;
}


const QString Radio::Type = "radio";
Radio::Radio(const QString& radio) :
		Info(radio, QString(), Radio::Type)
{}
const QString& Radio::radio() const {
	return _b;
}

const QString Podcast::Type = "podcast";
Podcast::Podcast(const QString& podcast) :
		Info(podcast, QString(), Podcast::Type)
{}
const QString& Podcast::podcast() const {
	return _b;
}

const QString Track::Type = "track";
Track::Track(const QString& artist, const QString& album, const QString& track) :
		Info(artist, album, Track::Type)
{
	_d = track;
	updatePath();
}
const QString& Track::artist() const {
	return _b;
}
const QString& Track::album() const {
	return _c;
}
const QString& Track::track() const {
	return _d;
}

//void Track::updatePath() {
//	_path = basePath + _type + separator +
//		md5(stripInvalidEntities(_b)) + separator +
//		md5(stripInvalidEntities(_c)) + separator +
//		md5(stripInvalidEntities(_d)) + fileExtension;
//	_path.setScheme(fileScheme);
//}

void Track::updatePath() {
	_path = basePath + _type + separator +
		md5(QString (" ")) + separator + // Always use " " for Artist for NB#275719
		md5(stripInvalidEntities(_c)) + separator +
		md5(stripInvalidEntities(_d)) + fileExtension;
	_path.setScheme(fileScheme);
}

