#ifndef MEDIAARTSREQUESTER_H
#define MEDIAARTSREQUESTER_H
#include "mediaartinfo.h"
#include <QUrl>
#include <QPixmap>
#include <QList>
#include <QObject>

namespace MediaArt {
class MediaArtRequesterPrivate;

/** \brief Requests for media art content.
 *
 * \see MediaArt::Info documentation to get know what are media arts.\n
 *
 * \see Thumbnails::Thumbnailer class documentation to get know how
 * requesting for thumbnails is working - requesting for media arts is very
 * similar.\n \n
 * 
 * Media arts requester is working in very similar way to Thumbnailer when it
 * is requesting thumbnails. You are creating Requester object with default
 * values which will be returned if content for particular media art was not found.
 * Then You are connecting signals from Requester: mediaArt(), defaultMediaArt(),
 * error(), unqueued() - see documentation for each of them to get know
 * in what situation each of them is emitted.
 * Request is made by calling request() method. \n
 * Media art You are requesting for need to be described as a MediaArt::Info
 * objects. \n \n
 *
 * Simplest example how to use MediaArt::Requester:
 * \code
 * // we have some list of MediaArt::Info objects:
 * MediaArt::Info album = MediaArt::Album("Artist A", "Album A");
 * MediaArt::Info track = MediaArt::Track("Artist A", "Album A", "Track 01");
 * MediaArt::Info podcast = MediaArt::Podcast("Funky Londonstyle");
 * 
 * QList<MediaArt::Info> listOfMedia;
 * listOfMedia << album << track << podcast
 * 
 * // now we are creating requester object (and we do not care about default values):
 * MediaArt::Requester* requester = new MediaArt::Requester();
 * 
 * // connecting requester's signal to our slot:
 * QObject::connect ( requester, SIGNAL(mediaArt(MediaArt::Info,QUrl,QPixmap)),
 *                    this,        SLOT(mediaArt(MediaArt::Info,QUrl,QPixmap)) );
 * 
 * // and makes a request:
 * requester->request(listOfMedia);
 * 
 * // You should have a mediaArt slot in You class which could look like this:
 * void mediaArt(const MediaArt::Info& mai, const QUrl& url, const QPixmap& pixmap) {
 * 	// it means there is a new media art's content - url for that content
 * 	// is in url parameter. If You requested also for loading pixmaps
 * 	// the pixmap with the content is in pixmap parameter.
 * 	qDebug() << "There is a new media art's content in path:" << url.path();
 * 	
 * 	// if You requested also for pixmap (requester->request(list, true)):
 * 	someLabel->setPixmap(pixmap);
 * }
 * \endcode
 * \see mediaArt(), defaultMediaArt(), error(), unqueued()
 */
class Requester : public QObject {
Q_OBJECT
	public:
		/** \brief Create a new Requester object.
		 *
		 * Builds a new Requester object. After object is built
		 * developer can start making requests stright away.
		 * \param defaultPath url which should be used as a default
		 *        url returned in defaultMediaArt signal
		 * \param defaultPixmap pixmap which should be used as a default
		 *        pixmap returned in defaultMediaArt signals
		 * \see defaultMediaArt
		 */
		Requester ( QUrl    defaultPath   = QUrl(),
			    QPixmap defaultPixmap = QPixmap() );

		/** \brief Default destructor for requester. */
		~Requester ();

		/** \brief Sets current default URL.
		 *
		 * When there is no real media art content for particular media
		 * art, Requester will send defaultMediaArt signal with default
		 * values: url and pixmap.
		 * \param path url to be used as a default one
		 */
		void setDefaultPath (const QUrl& path);

		/** \brief Sets current default pixmap.
		 *
		 * When there is no real media art content for particular media
		 * art, Requester will send defaultMediaArt signal with default
		 * values: url and pixmap.
		 * \param pixmap image to be used as a default one
		 */
		void setDefaultPixmap (const QPixmap& pixmap);

		/** \brief Returns current default URL.
		 *
		 * \return url which is current default url
		 */
		const QUrl defaultPath () const;

		/** \brief Gives current default pixmap.
		 *
		 * \return pixmap which is current default pixmap
		 */
		const QPixmap defaultPixmap () const;
	public:
		/** \brief Cancel all currently running requests.
		 *
		 * When there are some requests running and You are not
		 * interested anymore in the results You should call this method
		 * to reduce CPU usage. It will stop processing all requests.
		 * \param sendRemainingSignals if true Requester will send 
		 *        dequeueud signal for every not processed media art;
		 *        otherwise You will receive only finished signal.
		 */
		void cancel (bool sendRemainingSignals = false);

		/** \brief Make a request for media arts.
		 *
		 * Starts a request for content of list of media arts. In one
		 * request there could be mixed all kind of media arts (so it
		 * does not need to be a list with only albyum arts or artist
		 * arts).\n \n
		 * Please note, that Requester returns You a media art itself,
		 * not a thumbnail of it. For thumbnails please use
		 * Thumbnails::Thumbnailer class.
		 * \param list a list of media art we want to request for
		 * \param sendPixmap set to true if You want Requester to load
		 *        every image and send it in mediaArt signal. Setting
		 *        this to true makes Requester to work much longer.
		 *        When setting this to true You need to remember that
		 *        You do not have any influence how images are loaded.
		 * \return true if all media arts were already processed and
		 *         there will not be any other signals later. false mean
		 *         that some media art had to be requested and they
		 *         will be processed later (there will be some further
		 *         signals connected with that request).
		 * \see Thumbnails::Thumbnailer::request()
		 */
		bool request ( QList<Info>& list,
			       bool sendPixmap = false   );
	Q_SIGNALS:
		/** \brief Signals the processing hte request has started.
		 *
		 * This signal is sent when Requester start processing some
		 * request. It is automatically sent at the beginning of
		 * request() method.
		 */
		void started ();

		/** \brief Signals that request is finished.
		 *
		 * Signals that some request is finished and there will not be
		 * any other signal connected to that media art request.
		 */
		void finished ();

		/** \brief Signals that media art content is available.
		 *
		 * This signal is emitted when some media art's content is
		 * available.
		 * \param mai media art for which content was found
		 * \param path url to the media art content (image)
		 * \param pixmap pixmap with the image (it is filled only
		 *        when you called request() with sendPixmap set to true)
		 */
		void mediaArt (const MediaArt::Info& mai, const QUrl& path, const QPixmap& pixmap);

		/** \brief Signal with default content for media art.
		 * 
		 * This signal is emitted when media art's content was not
		 * available at the call time and we need to spend some time
		 * trying to find this media art. It means that after some time
		 * You will receive for the same MediaArt::Info object one of:\n
		 * mediaArt() if content will be found \n
		 * error() if Requester will not able to find any content for
		 *         media art.
		 * \param mai media art for which default content is signaled
		 * \param path default url for the media art
		 * \param pixmap default pixmap with the image for media art
		 *        (it is filled only when you called request() with
		 *         sendPixmap set to true)
		 */
		void defaultMediaArt (const MediaArt::Info& mai, const QUrl& path, const QPixmap& pixmap);

		/** \brief Signals that there was some problem with media art.
		 *
		 * This signals that the content for particular media art
		 * is not possible to be retrieved.
		 * \param message information with the reason why media art's
		 *        content is not possible to be returned
		 * \param mai media art with wich that error is connected
		 */
		void error (const QString& message, const MediaArt::Info& mai);

		/** \brief Signals that some media art request was unqueued.
		 *
		 * Signals that particular media art was uqueued from the
		 * request. It means that there will not be any other signals
		 * for that media art.
		 * \param mai media art which is abandoned in the request
		 */
		void unqueued (const MediaArt::Info& mai);
	private:
		friend class MediaArtRequesterPrivate;
		MediaArtRequesterPrivate* priv;
};}


#endif // MEDIAARTSREQUESTER_H
