#ifndef __MEDIAARTINFO_H
#define __MEDIAARTINFO_H

#include <QMetaType>
#include <QString>
#include <QUrl>

class QDebug;
class QPixmap;

namespace MediaArt {

/** \brief Describes particular media art.
 * 
 * Every resource (file, web page, internet radio) can have some additional
 * data attached. For example for album You can have an album cover or image
 * of the CD. For video file You can have a DVD cover or producer's logo. All
 * of those are called Media Arts. It some kind of art connected with particular
 * media source.\n
 * There are lots of media arts types: album art, artist art, podcast art,
 * radio art, track art etc. You can even create Your own type of media art.
 * In Thumbnailer library we are supporting only those mentioned above.
 * Supporting means we have some special version of MediaArt::Info class which
 * make it easier to work with particular media art type. So You can use:\n
 * MediaArt::Album - for ablum arts \n
 * MediaArt::Artist - for artist arts \n
 * MediaArt::Track - for track arts \n
 * MediaArt::Radio - for radio arts \n
 * MediaArt::Podcast - for podcast arts \n
 * See specified subclass documentation to check how to use it. Supporting DOES
 * NOT mean that You will always get something for that type of media art.
 * The content of media art cache depends on much more components (mainly 
 * tracker, but also every other application which will put something into the
 * cache - see setMediaArtImage()).
 * \n
 * MediaArt::Info (base) class allows You to represent every kind of media art
 * and to check if there is such a media art available:
 * \code
 * MediaArt::Info album = MediaArt::Album("Queen", "Greatest Hits vol.1");
 * if(album.exists()) {
 * 	// here we know that there is a media art of type album for artist Queen
 * 	// and album "Greatest Hits vol.1". You can get the media art from Url:
 *	doSomethingWithMediaArtFromURL(album.potentialPath());
 * }
 * \endcode
 * You can also put into that media art Your own data. Later on when we will
 * be asking for that album art we will get this content. ALSO someone else
 * who will be asking for that will receive it. This could be usefull if You
 * know some source for possible media art of particular media, e.g. for
 * podcast which You have already downloaded:
 * \code
 * MediaArt::Info podcast = MediaArt::Podcast("Funky House Londondstyle");
 * if(!podcast.exists()) {
 * 	// here we know that there is NO media art of type album for that podcast.
 * 	// Lets put there some own data, e.g. image downloaded with the podcast
 * 	podcast.setMediaArtImage(qpixmapWithMediaArtGotFromTheServer);
 * }
 * \endcode
 * \n
 * Media art path is defined by the type of media art and media itself. If You
 * know that info You would be able to check Yourself if that media art exists but
 * MediaArt::Info is doing that for You - You do not need to worry about how files
 * with media art are named, where they are placed and what kind of prefix they
 * have:\n
 * \code
 * MediaArt::Info podcast = MediaArt::Podcast("Funky House Londondstyle");
 * // proper URL to the media art file is...
 * QUrl mediaArtUrl = podcast.potentialPath();
 * // .. so file should be placed here:
 * QString pathToMediaArt = mediaArtUrl.path();
 * \endcode
 * In that example You do not care if You have created album, artist or podcast art.
 * As far as you have MediaArt::Info object You can get path to media art (which could
 * or could not exists - you can check it by calling exists()) \n
 * You can also clear cache with media art by calling remove() function:
 * \code
 * MediaArt::Info podcast = MediaArt::Podcast("Funky House Londondstyle");
 * podcast.remove()
 * // from now on there is no media art with type podcast for "Funky House Londonstyle".
 * \endcode
 */
class Info {
	friend uint   qHash (MediaArt::Info const &mai);
	friend QDebug operator<< (QDebug dbg, const MediaArt::Info& mai);
protected:
	/** \brief Private field for media art description - 1st part */
	QString _b;
	/** \brief Private field for media art description - 2nd part. */
	QString _c;
	/** \brief Private field for media art description - 3rd part. */
	QString _d;
	/** \brief Type of media art. */
	QString _type;
	/** \brief Potential path to image with media art content. */
	QUrl    _path;
protected:
	/** \brief Function which calculates MD5 of the string.
	 * 
	 * \param src string for which we want to get the checksum
	 * \return new string with checksum formatted properly
	 */
	static QString md5 (const QString& src);

	/** \brief Cleans up the string from invalid entities.
	 * 
	 * Specification \see http://live.gnome.org/MediaArtStorageSpec
	 * specifies what shopuld be removed from strings in order to be a proper
	 * description of media art.
	 * \param src string to be cleanuped
	 * \return new string which is clean version of src
	 */
	static QString stripInvalidEntities (const QString& src);

	/** \brief Builds a proper potential path.
	 *
	 * If you are building Your own type of media art it is possible that
	 * You need to overwritte that method in order to build proper
	 * potential path.
	 */
	void updatePath ();
public:
	/** \brief Type of undefined media arts. */
	static const QString Type;
public:
	/** \brief Returns type of particular media art.
	 *
	 * \return string with type of media art
	 */
	const QString& type () const;
	/** \brief Returns potential path of image with media art.
	 *
	 * \return url of place where potential image with media art is placed
	 */
	const QUrl&    potentialPath () const;
	/** \brief Checks if media art exists.
	 *
	 * \return true if that media art exists and there is some image connected
	 * with it. False if there is no content for that media art.
	 */
	bool exists() const;
	/** \brief Remove the content of media art.
	 *
	 * \return true if the content was deleted properly, false otherwise.
	 */
	bool remove() const;
	/** \brief Constructs basic media art object.
	 *
	 * You should NOT use that constructor in Your code. You should work
	 * with dedicated classes for particular types of media arts!
	 * \param b 1st part of media art description
	 * \param c 2st part of media art description
	 * \param type type of media art
	 */
	Info( const QString& b    = QString(),
	      const QString& c    = QString(),
	      const QString& type = QString() );
	/** \brief Creates new MediaArt::Info object which is a copy of src. */
	Info(const MediaArt::Info& src);
	/** \brief Destructs media art object. */
	~Info();
public:
	/** \brief Sets the content of media art image.
	 *
	 * At some cases You will have some image that You will know that it is
	 * a media art for some particular media. You should put then that image
	 * into the media arts cache, so other application (and even You later)
	 * can received it by MediaArt::Info API.
	 * \param source url to the file which should become the content of that media art
	 * \param overwrite false mean that if there is already some content it will not be
	 * overwritten. true will force to do that - old content will be deleted and replaced
	 * with the source.
	 * \return true if operation was successful, false otherwise
	 */
	bool setMediaArtImage(const QUrl& source,    bool overwrite = false) const;

	/** \brief Sets the content of media art image.
	 *
	 * At some cases You will have some image that You will know that it is
	 * a media art for some particular media. You should put then that image
	 * into the media arts cache, so other application (and even You later)
	 * can received it by MediaArt::Info API.
	 * \param source pixmap with image which should become the content of that media art
	 * \param overwrite false mean that if there is already some content it will not be
	 * overwritten. true will force to do that - old content will be deleted and replaced
	 * with the source.
	 * \return true if operation was successful, false otherwise
	 */
	bool setMediaArtImage(const QPixmap& source, bool overwrite = false) const;
};

	/** \brief Album arts.
	 *
	 * This helper class make it easier to build proper MediaArt::Info
	 * objects describing album arts.
	 */
	class Album   : public MediaArt::Info {
	public:
		/** \brief String identifying Media Art of Album types. */
		static const QString Type;
		/** \brief Constructs proper Album art object. */
		Album (const QString& artist, const QString& album);
		/** \brief Returns album artist's name. */
		const QString& artist () const;
		/** \brief Return album's name. */
		const QString& album () const;
	};

	/** \brief Artist arts.
	 *
	 * This helper class make it easier to build proper MediaArt::Info
	 * objects describing artist arts.\n
	 * This is different from MediaArt;:Album because it is concentrated
	 * on the artist, not particular album.
	 */
	class Artist  : public MediaArt::Info {
	public:
		/** \brief String identifying Media Art of Artist types */
		static const QString Type;
		/** \brief Constructs proper Artist art object. */
		Artist (const QString& artist, const QString& album = QString());
		/** \brief Returns artist's name. */
		const QString& artist () const;
		/** \brief Returns album for which we want to get the artist art. */
		const QString& album () const;
	};

	/** \brief Radio arts.
	 *
	 * This helper class make it easier to build proper MediaArt::Info
	 * objects describing radio arts.
	 */
	class Radio   : public MediaArt::Info {
	public:
		/** \brief String identifying Media Art of Radio types */
		static const QString Type;
		/** \brief Constructs proper Radio art object. */
		Radio (const QString& radio);
		/** \brief Returns radio's name. */
		const QString& radio () const;
	};

	/** \brief Podcast arts.
	 *
	 * This helper class make it easier to build proper MediaArt::Info
	 * objects describing podcast arts.
	 */
	class Podcast : public MediaArt::Info {
	public:
		/** \brief String identifying Media Art of Podcast types */
		static const QString Type;
		/** \brief Constructs proper Podcast media art object. */
		Podcast (const QString& podcast);
		/** \brief Returns podcast's name/description. */
		const QString& podcast () const;
	};

	/** \brief Track arts.
	 *
	 * This helper class make it easier to build proper MediaArt::Info
	 * objects describing track arts.
	 */
	class Track   : public MediaArt::Info {
	protected:
		void updatePath();
	public:
		/** \brief String identifying Media Art of Track types */
		static const QString Type;
		/** \brief Constructs Track media art object. */
		Track (const QString& artist, const QString& album, const QString& track);
		/** \brief Returns track artist's name. */
		const QString& artist () const;
		/** \brief Returns track album's name. */
		const QString& album () const;
		/** \brief Returns track's title. */
		const QString& track () const;
	};

/** \brief Hashing function for MediaArt::Info objects. */
uint   qHash      (MediaArt::Info const &mai);
/** \brief Simple operator to prints out MediaArt::Info objects. */
QDebug operator<< (QDebug dbg, const MediaArt::Info& mai);
/** \brief Checks if two MediaArt::Info are equal. */
bool   operator== (const MediaArt::Info &mai1, const MediaArt::Info &mai2);

} // end of namespace

Q_DECLARE_METATYPE(MediaArt::Info)

#endif // __MEDIAARTINFO_H
