#include <QApplication>
#include <QPixmap>
#include <QPainter>
#include <QSignalSpy>
#include <QTest>
#include "classes.h"

static QPixmap pixmapWithString(QString string);

int
main(int argc, char* argv[])
{
	QApplication app(argc, argv);
	qDebug() << "Application started...\n===========\n\n";

	/*
	   First we need to know what we are looking for (e.g. album arts for
	   particular albums). Then we need to create a proper MediaArt::Info
	   object for each of those album using helper classes (album art ->
	   MediaArt::Album):
	*/
	MediaArt::Info   album = MediaArt::Album("Queen", "The Greatest Hits vol.1");
	/*
	   For other media art types use different helper clasees:
	   WARNING: currently there is only one type of media art supported by the whole
	   platform: Album arts. Any other media arts do not have any content unless
	   some third part (or Your own) application will set the content for other
	   type of media art. For example some media player can set content
	   for media art of podcast type - then it will be available also for other
	   application. But only album art are extracted automatically.

	   If You are getting some media arts from different places and You need
	   to cache them on the disk, You can then use particular helper class
	   and setMediaArtImage() functions to store the content of that media art.
	   Thanks to that others will be able to use that image also.
	*/
	MediaArt::Info podcast = MediaArt::Podcast("Funky House Londonstyle");
	MediaArt::Info   radio = MediaArt::Radio("RMF Internet");
	/*
	   Lets add all of them to one QList:
	*/
	QList<MediaArt::Info> mediaList;
	mediaList << album;
	mediaList << podcast;
	mediaList << radio;
	/*
	   Now You are able to check if there is already some content for any of the media arts.
	   You can also always check what is the type of particular MediaArt::Info
	   object and what is the potential path of the content for that media art:
	*/
	qDebug() << "1. print information about media arts:";
	foreach(MediaArt::Info media, mediaList) {
		qDebug( QString("   * type: %1, path: %2, exists: %3")
		        .arg(media.type())
		        .arg(media.potentialPath().toString())
		        .arg(bool2str(media.exists()))
		        .toLatin1() );
	}
	/*
	   for debugging purposes You can also use simple << operator for qDebug()
	*/
	qDebug() << "\n2. print information about media arts (using built in << operator):";
	foreach(MediaArt::Info media, mediaList) {
		qDebug() << media;
	}

	/*
	   As You can see You have straight away the path to the media art content.
	   You can use it to access it directly from MediaArt::Info objects after checking
	   if there is any content (exists() functions). If You have some other image You want
	   to be a content for that media art You can set it directly through
	   MediaArt::Info API.
	   
	   Lets set a content for album art:
	   NOTE: we are giving the second parameter as a true - it will force
	   to overwritte any previous (if any) content for that media art
	*/
	qDebug() << "\n4. setting the content of media art:";
	bool set = album.setMediaArtImage(pixmapWithString("album art"), true);
	/* now we should see this as a content of that media art: */
	qDebug(QString("   * Was setting the content successful? %1").arg(bool2str(set)).toLatin1());
	qDebug(QString("   * Does content for album art exist? %1").arg(bool2str(album.exists())).toLatin1());
	/* try to set this again but do not overwrite already existing content */
	set = album.setMediaArtImage(pixmapWithString("album art"), false);
	qDebug(QString("   * Was overwritting the content successful (it should not be)? %1").arg(bool2str(set)).toLatin1());
	/* but with true it should work */
	set = album.setMediaArtImage(pixmapWithString("album art new"), true);
	qDebug(QString("   * Was overwritting the content successful (now it should be)? %1").arg(bool2str(set)).toLatin1());

	/* if You know that the content of particular media art is wrong You
	   can remove the content easily (but before doing that be sure that
	   nobody would like to use it anytime later):
	*/
	qDebug() << "\n5. deleting the content of media art:";
	bool deleted = album.remove();
	qDebug(QString("   * Was deleting existing content successful? %1").arg(bool2str(deleted)).toLatin1());
	qDebug(QString("   * Does content for album art exist (it shouldn't)? %1").arg(bool2str(album.exists())).toLatin1());
	deleted = album.remove();
	qDebug(QString("   * Was deleting non existing content successful (it shouldn't)? %1").arg(bool2str(deleted)).toLatin1());
	/* lets create back some content for the album */
	album.setMediaArtImage(pixmapWithString("album art latest"), true);

	/* MediaArt::Info class is a basic class You have to use while
	   working with media arts. It identifies every particular media art and
	   point where should be the content for that media art.
	   
	   You have to use this class together with MediaArt::Requester
	   and/or Thumbnails::Thumbnailer. The first one allows You to ask for a contents
	   of many media arts. In the future it could be possible to download
	   content from the internet - then MediaArt::Requester will do
	   this for You if there will not be an already existing content. The second
	   one (Thumbnails::Thumbnailer) allows You for retrieveing thumbnails
	   of media arts.
	   
	   Lets ask MediaArt::Requester for a contents of our media list
	*/
	qDebug() << "\n6. Requesting for media art contents through MediaArt::Requester API:";
	MediaArt::Requester requester( QUrl("file:///home/user/MyDocs/.images/test.jpeg") );
	/*
	    This will just receive and counts signals from Requester. Please, see the code
	    of that class in classes.h file - it is quite simple and easy to understand.
	    
	    For every request You will get exactly one started and finished signals.
	    - mediaArt signal tells You that there is a content for one media art
	    - defaultMediaArt signals tells You that requester was not able to find the content
	      but it will try to find it (You will get later error if that failed or
	      mediaArt signals if content was found)
	    - unqueued - signals that You canceled some request and requester was not
	      working on that particular media art at all
	    - error means that for that particular media art there is no content available
	*/
 	RequesterDebugger debugger (&requester);
	 debugger.clearCounters ();
	requester.request (mediaList, false);
	 debugger.printCounters ();
	/*
	   You can also see that if media art's content is found and You requested
	   the pixmap, pixmap will be loaded.
	   Lets check if pixmap for album (this is set by us before, size 250x250)
	   will be loaded successfuly (we will test the size of the pixmap)
	   
	   Similar You can check that setting default pixmap is working - what You set
	   there You will get in defaultMediaArt signals:
	*/
	QSignalSpy mediaArtSpy(&requester, SIGNAL(mediaArt(MediaArt::Info,QUrl,QPixmap)));
	QSignalSpy defaultMediaArtSpy(&requester, SIGNAL(defaultMediaArt(MediaArt::Info,QUrl,QPixmap)));
	requester.setDefaultPixmap ( QPixmap(125, 124) );
	requester.request (mediaList, true);
	// we should get only one mediaArt signal
	qDebug(QString("   * Number of mediaArt signal was (should be 1): %1").arg(mediaArtSpy.count()).toLatin1());
	QSize size = (qvariant_cast<QPixmap>(mediaArtSpy.at(0).at(2))).size();
	qDebug() << "   * Size of the pixmap was (should be QSize(250, 250)):" << size;
	// and there should be two defaultMediaArt signals (as there are two media arts without the content)
	qDebug(QString("   * Number of defaultMediaArt signal was (should be 2): %1").arg(defaultMediaArtSpy.count()).toLatin1());
	size = (qvariant_cast<QPixmap>(defaultMediaArtSpy.at(0).at(2))).size();
	qDebug() << "   * Size of the pixmap #1 was (should be QSize(125, 124)):" << size;
	size = (qvariant_cast<QPixmap>(defaultMediaArtSpy.at(1).at(2))).size();
	qDebug() << "   * Size of the pixmap #2 was (should be QSize(125, 124)):" << size;
	/* exactly as above You can set default url which You will get in defaultMediaArt signal*/
	requester.setDefaultPath(QUrl("file:///home/user/MyDocs/.images/test.jpeg"));
	defaultMediaArtSpy.clear();
	requester.request (mediaList, false);

	qDebug(QString("   * Number of defaultMediaArt signal was (should be 2): %1").arg(defaultMediaArtSpy.count()).toLatin1());
	QUrl url = qvariant_cast<QUrl>(defaultMediaArtSpy.at(0).at(1));
	qDebug() << "   * URL in signal #1 (should be \'file:///home/user/MyDocs/.images/test.jpeg\'):" << url;
	url = qvariant_cast<QUrl>(defaultMediaArtSpy.at(1).at(1));
	qDebug() << "   * URL in signal #2 (should be as above):" << url;

	/* And after all You can use Thumbnails::Thumbnailer class to get thumbnails of
	   list of media arts. You do not need to create MediaArt::Requester
	   to request for a media arts contents - Thumbnailer will do this for You.
	   
	   below the simplest example how to get a thumbnails of some media arts.
	   In that example only one media art has a content so we will get only
	   one thumnbail - for rest media arts we will get defaultThumbnail
	   signal followed by error signal. Error signals means that there was
	   no content of media art so there was no source to be thumnbailed.
	   
	   Please notice that thumbnail for existing media art will need to be
	   always recreated because above we overwritte the file so if there
	   was already a thumbnail for that media art (e.g. You have been running this app
	   before so You have already a thumbnail of media art's content) it becomes
	   out of date (different creation date than the source).
	*/
	qDebug() << "\n7. Requesting for media art's THUMBNAILS through Thumbnails::Thumbnailer API:";
	Thumbnails::Thumbnailer thumbnailer( QUrl("file:///home/user/test.jpeg"),
	                                     QPixmap(99, 98) );
	QSignalSpy thumbnailSpy(&thumbnailer, SIGNAL(thumbnail(MediaArt::Info,QUrl,QPixmap,QString) ));
	QSignalSpy defaultThumbnailSpy(&thumbnailer, SIGNAL( defaultThumbnail(MediaArt::Info,QUrl,QPixmap,QString) ));

	thumbnailer.request(mediaList, true);
	// here we need to wait, because thumbnailing one media art can take some time
	QTest::qWait(1000);
	qDebug(QString("   * Number of thumbnail signal was (should be 1): %1").arg(thumbnailSpy.count()).toLatin1());
	url = qvariant_cast<QUrl>(thumbnailSpy.at(0).at(1));
	qDebug() << "   * URL in signal:" << url << "That is the place where thumbnail of media art is placed.";

	qDebug(QString("   * Number of defaultThumbnail signal was (should be 3): %1").arg(defaultThumbnailSpy.count()).toLatin1());
	url = qvariant_cast<QUrl>(defaultThumbnailSpy.at(0).at(1));
	qDebug() << "   * URL in signal #1 (should be \'file:///home/user/test.jpeg\'):" << url;
	size = (qvariant_cast<QPixmap>(defaultThumbnailSpy.at(0).at(2))).size();
	qDebug() << "   * Size of the default pixmap in signal #1 was (should be QSize(99, 98)):" << size;


	qDebug() << "\n\n===========\n...Application finished.";
	return 0;
}


// just create some example pixmap and put a string there
QPixmap pixmapWithString(QString string) {
	QPixmap pixmap(250, 250);
	pixmap.fill(Qt::yellow);

	QPainter painter(&pixmap);
        painter.drawText(5, 130, string);

	return pixmap;
}
