/*
 * tangle-opacity-effect.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-opacity-effect.h"
#include <cogl/cogl.h>

G_DEFINE_TYPE(TangleOpacityEffect, tangle_opacity_effect, CLUTTER_TYPE_OFFSCREEN_EFFECT);

enum {
	PROP_0,
	PROP_OPACITY
};

struct _TangleOpacityEffectPrivate {
	guchar opacity;
	
	CoglHandle program;
	gint tex_uniform;
	gint opacity_uniform;
};

static const char* opacity_glsl_shader =
	"uniform sampler2D tex;\n"
	"uniform float opacity;\n"
	"\n"
	"void main() {\n"
	"	vec4 color = gl_Color * texture2D(tex, vec2(gl_TexCoord[0].xy));\n"
	"	gl_FragColor = vec4(color.r, color.g, color.b, opacity * color.a);\n"
	"}}n";

ClutterEffect* tangle_opacity_effect_new() {

	return CLUTTER_EFFECT(g_object_new(TANGLE_TYPE_OPACITY_EFFECT, NULL));
}

static gboolean tangle_opacity_effect_pre_paint(ClutterEffect *effect) {
	gboolean retvalue = FALSE;
	TangleOpacityEffect* opacity_effect;
	CoglHandle shader;
	
	opacity_effect = TANGLE_OPACITY_EFFECT(effect);
#if 0	
	if (clutter_actor_meta_get_enabled(CLUTTER_ACTOR_META(opacity_effect))) {
		g_return_val_if_fail(clutter_feature_available(CLUTTER_FEATURE_SHADERS_GLSL), FALSE);
				
		if (opacity_effect->priv->program == COGL_INVALID_HANDLE) {
			shader = cogl_create_shader(COGL_SHADER_TYPE_FRAGMENT);
			cogl_shader_source(shader, opacity_glsl_shader);		
			cogl_shader_compile(shader);
			if (!cogl_shader_is_compiled(shader)) {
				g_print("ERROR: %s\n", cogl_shader_get_info_log(shader));
			}
			
			opacity_effect->priv->program = cogl_create_program();
			cogl_program_attach_shader(opacity_effect->priv->program, shader);
			cogl_program_link(opacity_effect->priv->program);
			
			cogl_handle_unref(shader);
			
			opacity_effect->priv->tex_uniform = cogl_program_get_uniform_location(opacity_effect->priv->program, "tex");
			opacity_effect->priv->opacity_uniform = cogl_program_get_uniform_location(opacity_effect->priv->program, "opacity");
		}
	
		retvalue = CLUTTER_EFFECT_CLASS(tangle_opacity_effect_parent_class)->pre_paint(effect);
	}
#endif
	return retvalue;
}

static void tangle_opacity_effect_paint_target(ClutterOffscreenEffect* effect) {
	TangleOpacityEffect* opacity_effect;
	CoglHandle material;
	CoglHandle shader;

	opacity_effect = TANGLE_OPACITY_EFFECT(effect);
#if 0
	if (opacity_effect->priv->program != COGL_INVALID_HANDLE) {
		cogl_program_set_uniform_1i(opacity_effect->priv->program, opacity_effect->priv->tex_uniform, 0);
		cogl_program_set_uniform_1f(opacity_effect->priv->program, opacity_effect->priv->opacity_uniform, opacity_effect->priv->opacity / 255.0);
		cogl_material_set_user_program(clutter_offscreen_effect_get_target(effect), opacity_effect->priv->program);
	}
#endif
	CLUTTER_OFFSCREEN_EFFECT_CLASS(tangle_opacity_effect_parent_class)->paint_target(effect);
}

static void tangle_opacity_effect_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleOpacityEffect* opacity_effect;
	
	opacity_effect = TANGLE_OPACITY_EFFECT(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_opacity_effect_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleOpacityEffect* opacity_effect;

	opacity_effect = TANGLE_OPACITY_EFFECT(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_opacity_effect_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_opacity_effect_parent_class)->finalize(object);
}

static void tangle_opacity_effect_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_opacity_effect_parent_class)->dispose(object);
}

static void tangle_opacity_effect_class_init(TangleOpacityEffectClass* opacity_effect_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(opacity_effect_class);
	ClutterEffectClass* effect_class = CLUTTER_EFFECT_CLASS(opacity_effect_class);
	ClutterOffscreenEffectClass* offscreen_effect_class = CLUTTER_OFFSCREEN_EFFECT_CLASS(opacity_effect_class);

	gobject_class->finalize = tangle_opacity_effect_finalize;
	gobject_class->dispose = tangle_opacity_effect_dispose;
	gobject_class->set_property = tangle_opacity_effect_set_property;
	gobject_class->get_property = tangle_opacity_effect_get_property;

	effect_class->pre_paint = tangle_opacity_effect_pre_paint;

	offscreen_effect_class->paint_target = tangle_opacity_effect_paint_target;

	g_type_class_add_private (gobject_class, sizeof (TangleOpacityEffectPrivate));
}

static void tangle_opacity_effect_init(TangleOpacityEffect* opacity_effect) {
	opacity_effect->priv = G_TYPE_INSTANCE_GET_PRIVATE(opacity_effect, TANGLE_TYPE_OPACITY_EFFECT, TangleOpacityEffectPrivate);
}

