/*
 * tangle-grid.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-grid.h"

/**
 * SECTION:tangle-grid
 * @Short_description: An actor that displays grid lines
 * @Title: TangleGrid
 */
 
G_DEFINE_TYPE(TangleGrid, tangle_grid, TANGLE_TYPE_ACTOR);

enum {
	PROP_0,
	PROP_GRID_SPACING_X,
	PROP_GRID_SPACING_Y,
	PROP_BACKGROUND_COLOR,
	PROP_LINE_COLOR_X,
	PROP_LINE_COLOR_Y,
	PROP_LINE_WIDTH_X,
	PROP_LINE_WIDTH_Y
};

struct _TangleGridPrivate {
	gfloat grid_spacing_x;
	gfloat grid_spacing_y;
	guint line_width_x;
	guint line_width_y;
	ClutterColor background_color;
	ClutterColor line_color_x;
	ClutterColor line_color_y;
	guint background_color_set : 1;
	guint line_color_x_set : 1;
	guint line_color_y_set : 1;
};

static ClutterColor black = { 0, 0, 0, 255 };

ClutterActor* tangle_grid_new(gfloat grid_spacing_x, gfloat grid_spacing_y) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_GRID, "grid-spacing-x", grid_spacing_x, "grid-spacing-y", grid_spacing_y, NULL));
}

ClutterActor* tangle_grid_new_with_line_color(gfloat grid_spacing_x, gfloat grid_spacing_y, ClutterColor* line_color) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_GRID, "grid-spacing-x", grid_spacing_x, "grid-spacing-y", grid_spacing_y, "line-color-x", line_color, "line-color-y", line_color, NULL));
}

gfloat tangle_grid_get_grid_spacing_x(TangleGrid* grid) {
	g_return_if_fail(TANGLE_IS_GRID(grid));

	return grid->priv->grid_spacing_x;
}

void tangle_grid_set_grid_spacing_x(TangleGrid* grid, gfloat grid_spacing_x) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(grid_spacing_x >= 0.0);

	if (grid->priv->grid_spacing_x != grid_spacing_x) {
		grid->priv->grid_spacing_x = grid_spacing_x;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "grid-spacing-x");
	}
}

gfloat tangle_grid_get_grid_spacing_y(TangleGrid* grid) {
	g_return_val_if_fail(TANGLE_IS_GRID(grid), 0.0);

	return grid->priv->grid_spacing_y;
}

void tangle_grid_set_grid_spacing_y(TangleGrid* grid, gfloat grid_spacing_y) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(grid_spacing_y >= 0.0);

	if (grid->priv->grid_spacing_y != grid_spacing_y) {
		grid->priv->grid_spacing_y = grid_spacing_y;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "grid-spacing-y");
	}
}

guint tangle_grid_get_line_width_x(TangleGrid* grid) {
	g_return_val_if_fail(TANGLE_IS_GRID(grid), 0);

	return grid->priv->line_width_x;
}

void tangle_grid_set_line_width_x(TangleGrid* grid, guint line_width_x) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(line_width_x >= 1);

	if (grid->priv->line_width_x != line_width_x) {
		grid->priv->line_width_x = line_width_x;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "line-width-x");
	}
}

guint tangle_grid_get_line_width_y(TangleGrid* grid) {
	g_return_val_if_fail(TANGLE_IS_GRID(grid), 0);

	return grid->priv->line_width_y;
}

void tangle_grid_set_line_width_y(TangleGrid* grid, guint line_width_y) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(line_width_y >= 1);

	if (grid->priv->line_width_y != line_width_y) {
		grid->priv->line_width_y = line_width_y;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "line-width-y");
	}
}

void tangle_grid_get_background_color(TangleGrid* grid, ClutterColor* color_return) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(color_return != NULL);

	*color_return = grid->priv->background_color;
}

void tangle_grid_set_background_color(TangleGrid* grid, const ClutterColor* background_color) {
	g_return_if_fail(TANGLE_IS_GRID(grid));

	if (!clutter_color_equal(&grid->priv->background_color, background_color)) {
		if (background_color) {
			grid->priv->background_color = *background_color;
			grid->priv->background_color_set = TRUE;
		} else {
			grid->priv->background_color_set = FALSE;
		}
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "background-color");
	}
}

void tangle_grid_get_line_color_x(TangleGrid* grid, ClutterColor* color_return) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(color_return != NULL);

	*color_return = grid->priv->line_color_x;
}

void tangle_grid_set_line_color_x(TangleGrid* grid, const ClutterColor* line_color_x) {
	g_return_if_fail(TANGLE_IS_GRID(grid));

	if (!clutter_color_equal(&grid->priv->line_color_x, line_color_x)) {
		if (line_color_x) {
			grid->priv->line_color_x = *line_color_x;
			grid->priv->line_color_x_set = TRUE;
		} else {
			grid->priv->line_color_x_set = FALSE;
		}
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "line-color-x");
	}
}

void tangle_grid_get_line_color_y(TangleGrid* grid, ClutterColor* color_return) {
	g_return_if_fail(TANGLE_IS_GRID(grid));
	g_return_if_fail(color_return != NULL);

	*color_return = grid->priv->line_color_y;
}

void tangle_grid_set_line_color_y(TangleGrid* grid, const ClutterColor* line_color_y) {
	g_return_if_fail(TANGLE_IS_GRID(grid));

	if (!clutter_color_equal(&grid->priv->line_color_y, line_color_y)) {
		if (line_color_y) {
			grid->priv->line_color_y = *line_color_y;
			grid->priv->line_color_y_set = TRUE;
		} else {
			grid->priv->line_color_y_set = FALSE;
		}
		clutter_actor_queue_redraw(CLUTTER_ACTOR(grid));
		g_object_notify(G_OBJECT(grid), "line-color-y");
	}
}

static void tangle_grid_paint_aligned(TangleActor* actor, gfloat width, gfloat height) {
	TangleGrid* grid;
	gfloat x, y;
	guint8 alpha;
	guint u;
	
	grid = TANGLE_GRID(actor);
	
	if (grid->priv->background_color_set) {
		alpha = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(actor)) * grid->priv->background_color.alpha / 255;
		cogl_set_source_color4ub(grid->priv->background_color.red,
			 		 grid->priv->background_color.green,
			 		 grid->priv->background_color.blue,
					 alpha);
		cogl_rectangle(0.0, 0.0, width, height);
	}

	if (grid->priv->line_color_x_set) {
		alpha = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(actor)) * grid->priv->line_color_y.alpha / 255;
		cogl_set_source_color4ub(grid->priv->line_color_y.red,
			 	 grid->priv->line_color_y.green,
			 	 grid->priv->line_color_y.blue,
				 alpha);

		if (grid->priv->grid_spacing_x > 0.0) {
			for (u = 0; u < grid->priv->line_width_y; u++) {
				for (x = 1.0; x < width; x += grid->priv->grid_spacing_x) { /* TODO: Why 1.0? */
					cogl_path_move_to(x + u, 0.0);
					cogl_path_line_to(x + u, height);
				}
				cogl_path_stroke();
			}
		}
	}
	
	if (grid->priv->line_color_y_set) {
		alpha = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(actor)) * grid->priv->line_color_x.alpha / 255;
		cogl_set_source_color4ub(grid->priv->line_color_x.red,
			 	 grid->priv->line_color_x.green,
			 	 grid->priv->line_color_x.blue,
				 alpha);

		if (grid->priv->grid_spacing_y > 0.0) {
			for (u = 0; u < grid->priv->line_width_x; u++) {
				for (y = 0.0; y < height; y += grid->priv->grid_spacing_y) {
					cogl_path_move_to(0.0, y + u);
					cogl_path_line_to(width + 1, y + u); /* TODO: Why + 1? */
				}
				cogl_path_stroke();
			}
		}
	}
}

static void tangle_grid_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleGrid* grid;
	
	grid = TANGLE_GRID(object);

	switch (prop_id) {
		case PROP_GRID_SPACING_X:
			tangle_grid_set_grid_spacing_x(grid, g_value_get_float(value));
			break;
		case PROP_GRID_SPACING_Y:
			tangle_grid_set_grid_spacing_y(grid, g_value_get_float(value));
			break;
		case PROP_LINE_WIDTH_X:
			tangle_grid_set_line_width_x(grid, g_value_get_uint(value));
			break;
		case PROP_LINE_WIDTH_Y:
			tangle_grid_set_line_width_y(grid, g_value_get_uint(value));
			break;
		case PROP_BACKGROUND_COLOR:
			tangle_grid_set_background_color(grid, clutter_value_get_color(value));
			break;
		case PROP_LINE_COLOR_X:
			tangle_grid_set_line_color_x(grid, clutter_value_get_color(value));
			break;
		case PROP_LINE_COLOR_Y:
			tangle_grid_set_line_color_y(grid, clutter_value_get_color(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_grid_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleGrid* grid;

	grid = TANGLE_GRID(object);

        switch (prop_id) {
		case PROP_GRID_SPACING_X:
			g_value_set_float(value, grid->priv->grid_spacing_x);
			break;
		case PROP_GRID_SPACING_Y:
			g_value_set_float(value, grid->priv->grid_spacing_y);
			break;
		case PROP_LINE_WIDTH_X:
			g_value_set_uint(value, grid->priv->line_width_x);
			break;
		case PROP_LINE_WIDTH_Y:
			g_value_set_uint(value, grid->priv->line_width_y);
			break;
		case PROP_BACKGROUND_COLOR:
			clutter_value_set_color(value, &grid->priv->background_color);
			break;
		case PROP_LINE_COLOR_X:
			clutter_value_set_color(value, &grid->priv->line_color_x);
			break;
		case PROP_LINE_COLOR_Y:
			clutter_value_set_color(value, &grid->priv->line_color_y);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_grid_class_init(TangleGridClass* grid_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(grid_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(grid_class);

	gobject_class->set_property = tangle_grid_set_property;
	gobject_class->get_property = tangle_grid_get_property;

	actor_class->paint_aligned = tangle_grid_paint_aligned;

	/**
	 * TangleGrid:grid-spacing-x:
	 */
	g_object_class_install_property(gobject_class, PROP_GRID_SPACING_X,
	                                g_param_spec_float("grid-spacing-x",
	                                                   "Grid spacing x",
	                                                   "The horizontal spacing of grid lines",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleGrid:grid-spacing-y:
	 */
	g_object_class_install_property(gobject_class, PROP_GRID_SPACING_Y,
	                                g_param_spec_float("grid-spacing-y",
	                                                   "Grid spacing y",
	                                                   "The vertical spacing of grid lines",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleGrid:line-width-x:
	 */
	g_object_class_install_property(gobject_class, PROP_LINE_WIDTH_X,
	                                g_param_spec_uint("line-width-x",
	                                                  "Line width x",
	                                                  "The horizontal width of grid lines",
	                                                  1, G_MAXUINT, 1,
	                                                  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleGrid:line-width-y:
	 */
	g_object_class_install_property(gobject_class, PROP_LINE_WIDTH_Y,
	                                g_param_spec_uint("line-width-y",
	                                                  "Line width y",
	                                                  "The vertical width of grid lines",
	                                                  1, G_MAXUINT, 1,
	                                                  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleGrid:background-color:
	 */
	g_object_class_install_property(gobject_class, PROP_BACKGROUND_COLOR,
	                                clutter_param_spec_color("background-color",
	                                                	 "Background color",
	                                                	 "The background color of the grid",
	                                                	 NULL,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleGrid:line-color-x:
	 */
	g_object_class_install_property(gobject_class, PROP_LINE_COLOR_X,
	                                clutter_param_spec_color("line-color-x",
	                                                	 "Line color x",
	                                                	 "The horizontal line stroke color of the grid",
	                                                	 &black,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleGrid:line-color-y:
	 */
	g_object_class_install_property(gobject_class, PROP_LINE_COLOR_Y,
	                                clutter_param_spec_color("line-color-y",
	                                                	 "Line color y",
	                                                	 "The vertical line stroke color of the grid",
	                                                	 &black,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));


	g_type_class_add_private (gobject_class, sizeof (TangleGridPrivate));
}

static void tangle_grid_init(TangleGrid* grid) {
	grid->priv = G_TYPE_INSTANCE_GET_PRIVATE(grid, TANGLE_TYPE_GRID, TangleGridPrivate);
	grid->priv->line_width_x = grid->priv->line_width_y = 1;
}

