/*
 * tangle-easing-transition.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-easing-transition.h"

/**
 * SECTION:tangle-easing-transition
 * @Short_description: A transition that is based on easing modes.
 * @Title: TangleEasingTransition
 */

G_DEFINE_TYPE(TangleEasingTransition, tangle_easing_transition, TANGLE_TYPE_TRANSITION);

enum {
	PROP_0,
	PROP_MODE,
	PROP_DURATION,
	PROP_SHOW_FROM_SCALE,
	PROP_HIDE_TO_SCALE,
	PROP_USE_OPACITY,
	PROP_SHOW_FROM_OPACITY,
	PROP_SHOW_TO_OPACITY,
	PROP_HIDE_TO_OPACITY,	
};

struct _TangleEasingTransitionPrivate {
	gulong mode;
	guint duration;
	gdouble show_from_scale;
	gdouble hide_to_scale;
	guint show_from_opacity;
	guint show_to_opacity;
	guint hide_to_opacity;

	guint use_opacity : 1;
};

TangleTransition* tangle_easing_transition_new(gulong mode, guint duration) {

	return TANGLE_TRANSITION(g_object_new(TANGLE_TYPE_EASING_TRANSITION, "mode", mode, "duration", duration, NULL));
}

gulong tangle_easing_transition_get_mode(TangleEasingTransition* easing_transition) {
	g_return_val_if_fail(TANGLE_IS_EASING_TRANSITION(easing_transition), 0);

	return easing_transition->priv->mode;
}

void tangle_easing_transition_set_mode(TangleEasingTransition* easing_transition, gulong mode) {
	g_return_if_fail(TANGLE_IS_EASING_TRANSITION(easing_transition));

	if (easing_transition->priv->mode != mode) {
		easing_transition->priv->mode = mode;
		g_object_notify(G_OBJECT(easing_transition), "mode");
	}
}

guint tangle_easing_transition_get_duration(TangleEasingTransition* easing_transition) {
	g_return_val_if_fail(TANGLE_IS_EASING_TRANSITION(easing_transition), 0);

	return easing_transition->priv->duration;
}

void tangle_easing_transition_set_duration(TangleEasingTransition* easing_transition, guint duration) {
	g_return_if_fail(TANGLE_IS_EASING_TRANSITION(easing_transition));

	if (easing_transition->priv->duration != duration) {
		easing_transition->priv->duration = duration;
		g_object_notify(G_OBJECT(easing_transition), "duration");
	}
}

void tangle_easing_transition_set_mode_duration(TangleEasingTransition* easing_transition, glong mode, guint duration) {
	g_return_if_fail(TANGLE_IS_EASING_TRANSITION(easing_transition));

	g_object_freeze_notify(G_OBJECT(easing_transition));

	tangle_easing_transition_set_mode(easing_transition, mode);
	tangle_easing_transition_set_duration(easing_transition, duration);

	g_object_thaw_notify(G_OBJECT(easing_transition));
}

static ClutterTimeline* tangle_easing_transition_animate_actor(TangleTransition* transition, TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box) {
	ClutterAnimation* animation = NULL;
	TangleEasingTransition* easing_transition;
	gfloat move_x, move_y;
	gdouble scale_x, scale_y;
	ClutterActor* parent;
	GValue value = { 0 };
	
	easing_transition = TANGLE_EASING_TRANSITION(transition);

	if (easing_transition->priv->duration && easing_transition->priv->mode) {
		tangle_actor_get_transition_move(actor, &move_x, &move_y);
		tangle_actor_get_transition_scale(actor, &scale_x, &scale_y);
		if (current_box && new_box) {
			move_x += current_box->x1 - new_box->x1;
			move_y += current_box->y1 - new_box->y1;
			scale_x *= (current_box->x2 - current_box->x1) / (new_box->x2 - new_box->x1);
			scale_y *= (current_box->y2 - current_box->y1) / (new_box->y2 - new_box->y1);
			g_object_set(actor, "transition-move-x", move_x, "transition-move-y", move_y, "transition-scale-x", scale_x, "transition-scale-y", scale_y, NULL);

			animation = tangle_actor_animate(actor, easing_transition->priv->mode, easing_transition->priv->duration, "transition-move-x", 0.0, "transition-move-y", 0.0, "transition-scale-x", 1.0, "transition-scale-y", 1.0, NULL);
		} else if (current_box) {
			animation = tangle_actor_animate(actor, easing_transition->priv->mode, easing_transition->priv->duration,
							 "transition-scale-x", easing_transition->priv->hide_to_scale,
							 "transition-scale-y", easing_transition->priv->hide_to_scale,
							 NULL);
			if (easing_transition->priv->use_opacity) {
				if (CLUTTER_CHECK_VERSION(1, 2, 0)) {
					g_value_init(&value, G_TYPE_UINT);
					g_value_set_uint(&value, easing_transition->priv->hide_to_opacity);
				} else {
					g_value_init(&value, G_TYPE_UCHAR);
					g_value_set_uchar(&value, easing_transition->priv->hide_to_opacity);
				}
				clutter_animation_bind(animation, "opacity", &value);
				g_value_unset(&value);
			}
		} else if ((parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor))) &&
		           (!TANGLE_IS_ACTOR(parent) || TANGLE_ACTOR_IS_ALLOCATED(parent))) {
			/* If the parent is not also a new TangleActor (and thus, doing transition animation),
			 * animate self.
			 */
			g_object_set(actor,
			             "transition-scale-x", easing_transition->priv->show_from_scale,
				     "transition-scale-y", easing_transition->priv->show_from_scale,
				     (easing_transition->priv->use_opacity ? "opacity" : NULL), easing_transition->priv->show_from_opacity,
				     NULL);

			animation = tangle_actor_animate(actor, easing_transition->priv->mode, easing_transition->priv->duration, "transition-scale-x", 1.0, "transition-scale-y", 1.0, NULL);
			if (easing_transition->priv->use_opacity) {
				if (CLUTTER_CHECK_VERSION(1, 2, 0)) {
					g_value_init(&value, G_TYPE_UINT);
					g_value_set_uint(&value, 255);
				} else {
					g_value_init(&value, G_TYPE_UCHAR);
					g_value_set_uchar(&value, 255);
				}
				clutter_animation_bind(animation, "opacity", &value);
				g_value_unset(&value);
			}
		}
	}

	return (animation ? clutter_animation_get_timeline(animation) : NULL);
}

static void tangle_easing_transition_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleEasingTransition* easing_transition;
	
	easing_transition = TANGLE_EASING_TRANSITION(object);

	switch (prop_id) {
		case PROP_MODE:
			tangle_easing_transition_set_mode(easing_transition, g_value_get_ulong(value));
			break;
		case PROP_DURATION:
			tangle_easing_transition_set_duration(easing_transition, g_value_get_uint(value));
			break;
		case PROP_SHOW_FROM_SCALE:
			easing_transition->priv->show_from_scale = g_value_get_double(value);
			break;
		case PROP_HIDE_TO_SCALE:
			easing_transition->priv->hide_to_scale = g_value_get_double(value);
			break;
		case PROP_USE_OPACITY:
			easing_transition->priv->use_opacity = g_value_get_boolean(value);
			break;
		case PROP_SHOW_FROM_OPACITY:
			easing_transition->priv->show_from_opacity = g_value_get_uint(value);
			break;
		case PROP_SHOW_TO_OPACITY:
			easing_transition->priv->show_to_opacity = g_value_get_uint(value);
			break;
		case PROP_HIDE_TO_OPACITY:
			easing_transition->priv->hide_to_opacity = g_value_get_uint(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_easing_transition_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleEasingTransition* easing_transition;

	easing_transition = TANGLE_EASING_TRANSITION(object);

        switch (prop_id) {
		case PROP_MODE:
			g_value_set_ulong(value, easing_transition->priv->mode);
			break;
		case PROP_DURATION:
			g_value_set_uint(value, easing_transition->priv->duration);
			break;
		case PROP_SHOW_FROM_SCALE:
			g_value_set_double(value, easing_transition->priv->show_from_scale);
			break;
		case PROP_HIDE_TO_SCALE:
			g_value_set_double(value, easing_transition->priv->hide_to_scale);
			break;
		case PROP_USE_OPACITY:
			g_value_set_boolean(value, easing_transition->priv->use_opacity);
			break;
		case PROP_SHOW_FROM_OPACITY:
			g_value_set_uint(value, easing_transition->priv->show_from_opacity);
			break;
		case PROP_SHOW_TO_OPACITY:
			g_value_set_uint(value, easing_transition->priv->show_to_opacity);
			break;
		case PROP_HIDE_TO_OPACITY:
			g_value_set_uint(value, easing_transition->priv->hide_to_opacity);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_easing_transition_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_easing_transition_parent_class)->finalize(object);
}

static void tangle_easing_transition_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_easing_transition_parent_class)->dispose(object);
}

static void tangle_easing_transition_class_init(TangleEasingTransitionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	TangleTransitionClass* transition_class = TANGLE_TRANSITION_CLASS(klass);

	gobject_class->finalize = tangle_easing_transition_finalize;
	gobject_class->dispose = tangle_easing_transition_dispose;
	gobject_class->set_property = tangle_easing_transition_set_property;
	gobject_class->get_property = tangle_easing_transition_get_property;

	transition_class->animate_actor = tangle_easing_transition_animate_actor;

	/**
	 * TangleEasingTransition:mode:
	 *
	 * The easing mode used in the transition animation.
	 */
	g_object_class_install_property(gobject_class, PROP_MODE,
	                                g_param_spec_ulong("mode",
	                                "Mode",
	                                "The easing mode used in transition animation",
	                                0, G_MAXULONG, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:duration:
	 *
	 * The duration of the transition animation.
	 */
	g_object_class_install_property(gobject_class, PROP_DURATION,
	                                g_param_spec_uint("duration",
	                                "Duration",
	                                "The duration of the transition animation",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:show-from-scale:
	 *
	 * A value that is used as a starting scale when showing an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_SHOW_FROM_SCALE,
	                                g_param_spec_double("show-from-scale",
	                                "Show from scale",
	                                "A value that is used as a starting scale when showing an actor",
	                                0, G_MAXDOUBLE, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:hide-to-scale:
	 *
	 * A value that is used as a target scale when hiding an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_HIDE_TO_SCALE,
	                                g_param_spec_double("hide-to-scale",
	                                "Hide to scale",
	                                "A value that is used as a target scale when hiding an actor",
	                                0, G_MAXDOUBLE, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:use-opacity:
	 *
	 * Whether to animate opacity of an actor during show or hide transition.
	 */
	g_object_class_install_property(gobject_class, PROP_USE_OPACITY,
	                                g_param_spec_boolean("use-opacity",
	                                "Use opacity",
	                                "Whether to animate opacity of an actor",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:show-from-opacity:
	 *
	 * A value that is used as a starting opacity when showing an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_SHOW_FROM_OPACITY,
	                                g_param_spec_uint("show-from-opacity",
	                                "Show from opacity",
	                                "A value that is used as a starting opacity when showing an actor",
	                                0, 255, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:show-to-opacity:
	 *
	 * A value that is used as a target opacity when showing an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_SHOW_TO_OPACITY,
	                                g_param_spec_uint("show-to-opacity",
	                                "Show to opacity",
	                                "A value that is used as a target opacity when showing an actor",
	                                0, 255, 255,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleEasingTransition:hide-to-opacity:
	 *
	 * A value that is used as a target opacity when hiding an actor.
	 */
	g_object_class_install_property(gobject_class, PROP_HIDE_TO_OPACITY,
	                                g_param_spec_uint("hide-to-opacity",
	                                "Hide to opacity",
	                                "A value that is used as a target opacity when hiding an actor",
	                                0, 255, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));


	g_type_class_add_private (gobject_class, sizeof (TangleEasingTransitionPrivate));
}

static void tangle_easing_transition_init(TangleEasingTransition* easing_transition) {
	easing_transition->priv = G_TYPE_INSTANCE_GET_PRIVATE(easing_transition, TANGLE_TYPE_EASING_TRANSITION, TangleEasingTransitionPrivate);
	
	easing_transition->priv->show_to_opacity = 255;
}
