/*
 * tangle-highlight-effect.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2011 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-highlight-effect.h"

G_DEFINE_TYPE(TangleHighlightEffect, tangle_highlight_effect, CLUTTER_TYPE_OFFSCREEN_EFFECT);

enum {
	PROP_0,
	PROP_COLOR,
	PROP_DISTANCE
};

struct _TangleHighlightEffectPrivate {
	ClutterColor color;
	gfloat distance;
	
	CoglHandle program;
	gint tex_uniform;
	gint color_uniform;
	gint x_offset_uniform;
	gint y_offset_uniform;
};

static const gchar* shader_source =
#ifdef COGL_HAS_GLES2
	"precision mediump float;\n"
	"varying vec2 tex_coord[1];\n"
#define TEX_COORD "tex_coord[0]"
#else
#define TEX_COORD "gl_TexCoord[0]"
#endif
	"uniform sampler2D tex;\n"
	"uniform vec4 color;\n"
	"uniform float x_offset;\n"
	"uniform float y_offset;\n"
	"void main () {\n"
	"	float x_offset_half = y_offset * 0.5;\n"
	"	float y_offset_half = y_offset * 0.5;\n"
	"	vec4 frag_color = color;\n"
	"	float alpha = \n"
	"		texture2D(tex, vec2(" TEX_COORD ".x, " TEX_COORD ".y - y_offset_half)).a * 0.125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x, " TEX_COORD ".y + y_offset_half)).a * 0.125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x - x_offset_half, " TEX_COORD ".y)).a * 0.125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x - x_offset_half, " TEX_COORD ".y)).a * 0.125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x - x_offset_half, " TEX_COORD ".y - y_offset_half)).a * 0.09375 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x + x_offset_half, " TEX_COORD ".y + y_offset_half)).a * 0.09375 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x - x_offset_half, " TEX_COORD ".y + y_offset_half)).a * 0.09375 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x + x_offset_half, " TEX_COORD ".y - y_offset_half)).a * 0.09375 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x, " TEX_COORD ".y - y_offset)).a * 0.03125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x, " TEX_COORD ".y + y_offset)).a * 0.03125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x - x_offset, " TEX_COORD ".y)).a * 0.03125 +\n"
	"		texture2D(tex, vec2(" TEX_COORD ".x + x_offset, " TEX_COORD ".y)).a * 0.03125;\n"
	"	alpha = 1.0 - alpha;\n"
	"	alpha = alpha * alpha * alpha;\n"
	"	alpha = 1.0 - alpha;\n"
	"	frag_color = frag_color * alpha;\n"
	"	gl_FragColor = frag_color;\n"
	"}\n";
	
static const ClutterColor white = { 255, 255, 255, 255 };

ClutterEffect* tangle_highlight_effect_new() {

	return CLUTTER_EFFECT(g_object_new(TANGLE_TYPE_HIGHLIGHT_EFFECT, NULL));
}

void tangle_highlight_effect_get_color(TangleHighlightEffect* highlight_effect, ClutterColor* color) {
	g_return_if_fail(TANGLE_IS_HIGHLIGHT_EFFECT(highlight_effect));
	g_return_if_fail(color != NULL);

	*color = highlight_effect->priv->color;
}

void tangle_highlight_effect_set_color(TangleHighlightEffect* highlight_effect, const ClutterColor* color) {
	ClutterActor* actor;

	g_return_if_fail(TANGLE_IS_HIGHLIGHT_EFFECT(highlight_effect));
	g_return_if_fail(color != NULL);
	
	if (!clutter_color_equal(&highlight_effect->priv->color, color)) {
		highlight_effect->priv->color = *color;
		g_object_notify(G_OBJECT(highlight_effect), "color");
		if ((actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(highlight_effect)))) {
			clutter_actor_queue_redraw(actor);
		}
	}
}

gfloat tangle_highlight_effect_get_distance(TangleHighlightEffect* highlight_effect) {
	g_return_val_if_fail(TANGLE_IS_HIGHLIGHT_EFFECT(highlight_effect), 0.0);

	return highlight_effect->priv->distance;
}

void tangle_highlight_effect_set_distance(TangleHighlightEffect* highlight_effect, gfloat distance) {
	ClutterActor* actor;
	
	g_return_if_fail(TANGLE_IS_HIGHLIGHT_EFFECT(highlight_effect));

	if (highlight_effect->priv->distance != distance) {
		highlight_effect->priv->distance = distance;
		
		if ((actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(highlight_effect)))) {
			clutter_actor_queue_redraw(actor);
		}
		
		g_object_notify(G_OBJECT(highlight_effect), "distance");
	}
}

static CoglHandle tangle_highlight_effect_create_texture(ClutterOffscreenEffect* offscreen_effect, gfloat min_width, gfloat min_height) {
	gfloat distance;
	
	distance = TANGLE_HIGHLIGHT_EFFECT(offscreen_effect)->priv->distance;

	return cogl_texture_new_with_size(min_width + 2 * (distance + 1), min_height + 2 * (distance + 1), COGL_TEXTURE_NO_SLICING, COGL_PIXEL_FORMAT_RGBA_8888_PRE);
}

static gboolean tangle_highlight_effect_pre_paint(ClutterEffect* effect) {
	gboolean success = FALSE;
	TangleHighlightEffect* highlight_effect;
	ClutterActor* actor;
	CoglHandle shader;
	gchar* info_log;
	gint distance;
	
	highlight_effect = TANGLE_HIGHLIGHT_EFFECT(effect);

	if (highlight_effect->priv->distance > 0.0 &&
	    clutter_actor_meta_get_enabled(CLUTTER_ACTOR_META(effect)) &&
	    (actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(effect))) &&
	    clutter_feature_available(CLUTTER_FEATURE_SHADERS_GLSL)) {
		if (highlight_effect->priv->program == COGL_INVALID_HANDLE) {
			shader = cogl_create_shader(COGL_SHADER_TYPE_FRAGMENT);
			cogl_shader_source(shader, shader_source);
			cogl_shader_compile(shader);
			if (!cogl_shader_is_compiled(shader)) {
				info_log = cogl_shader_get_info_log(shader);
				g_warning(G_STRLOC ": Unable to compile the shader: %s", info_log);
				g_free(info_log);
			} else {
				highlight_effect->priv->program = cogl_create_program();
				cogl_program_attach_shader(highlight_effect->priv->program, shader);
				cogl_program_link(highlight_effect->priv->program);
				
				highlight_effect->priv->tex_uniform = cogl_program_get_uniform_location(highlight_effect->priv->program, "tex");
				highlight_effect->priv->color_uniform = cogl_program_get_uniform_location(highlight_effect->priv->program, "color");
				highlight_effect->priv->x_offset_uniform = cogl_program_get_uniform_location(highlight_effect->priv->program, "x_offset");
				highlight_effect->priv->y_offset_uniform = cogl_program_get_uniform_location(highlight_effect->priv->program, "y_offset");

				success = CLUTTER_EFFECT_CLASS(tangle_highlight_effect_parent_class)->pre_paint(effect);
			}
			cogl_handle_unref(shader);
		} else {
			success = CLUTTER_EFFECT_CLASS(tangle_highlight_effect_parent_class)->pre_paint(effect);
		}

	}

	if (success) {
		distance = highlight_effect->priv->distance + 1;
		cogl_translate(distance, distance, 0.0);
	}

	return success;
}

static void tangle_highlight_effect_paint_target(ClutterOffscreenEffect* offscreen_effect) {
	TangleHighlightEffect* highlight_effect;
	gint distance;
	gfloat color[4];
	ClutterActorBox box;
	CoglHandle material;
	CoglHandle user_program;

	highlight_effect = TANGLE_HIGHLIGHT_EFFECT(offscreen_effect);

	if (highlight_effect->priv->program) {
		distance = highlight_effect->priv->distance + 1;
		cogl_translate(-distance, -distance, 0.0);

		cogl_program_set_uniform_1i(highlight_effect->priv->program, highlight_effect->priv->tex_uniform, 0);

		color[0] = highlight_effect->priv->color.red / 255.0;
		color[1] = highlight_effect->priv->color.green / 255.0;
		color[2] = highlight_effect->priv->color.blue / 255.0;
		color[3] = highlight_effect->priv->color.alpha / 255.0;
		cogl_program_set_uniform_float(highlight_effect->priv->program, highlight_effect->priv->color_uniform, 4, 1, color);
	
		clutter_actor_get_allocation_box(clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(highlight_effect)), &box);
		cogl_program_set_uniform_1f(highlight_effect->priv->program, highlight_effect->priv->x_offset_uniform,
		                            highlight_effect->priv->distance / (box.x2 - box.x1));
		cogl_program_set_uniform_1f(highlight_effect->priv->program, highlight_effect->priv->y_offset_uniform,
		                            highlight_effect->priv->distance / (box.y2 - box.y1));
		
		material = clutter_offscreen_effect_get_target(offscreen_effect);
		user_program = cogl_material_get_user_program(material);
		cogl_material_set_user_program(material, highlight_effect->priv->program);

		CLUTTER_OFFSCREEN_EFFECT_CLASS(tangle_highlight_effect_parent_class)->paint_target(offscreen_effect);	

		cogl_material_set_user_program(material, user_program);

		CLUTTER_OFFSCREEN_EFFECT_CLASS(tangle_highlight_effect_parent_class)->paint_target(offscreen_effect);	
	} else {
		CLUTTER_OFFSCREEN_EFFECT_CLASS(tangle_highlight_effect_parent_class)->paint_target(offscreen_effect);	
	}
}

static void tangle_highlight_effect_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleHighlightEffect* highlight_effect;
	
	highlight_effect = TANGLE_HIGHLIGHT_EFFECT(object);

	switch (prop_id) {
		case PROP_COLOR:
			tangle_highlight_effect_set_color(highlight_effect, clutter_value_get_color(value));
			break;
		case PROP_DISTANCE:
			tangle_highlight_effect_set_distance(highlight_effect, g_value_get_float(value));
			break;			
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_highlight_effect_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleHighlightEffect* highlight_effect;

	highlight_effect = TANGLE_HIGHLIGHT_EFFECT(object);

        switch (prop_id) {
		case PROP_COLOR:
			clutter_value_set_color(value, &highlight_effect->priv->color);
			break;
		case PROP_DISTANCE:
			g_value_set_float(value, highlight_effect->priv->distance);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_highlight_effect_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_highlight_effect_parent_class)->finalize(object);
}

static void tangle_highlight_effect_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_highlight_effect_parent_class)->dispose(object);
}

static void tangle_highlight_effect_class_init(TangleHighlightEffectClass* hightlight_effect_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(hightlight_effect_class);
	ClutterEffectClass* effect_class = CLUTTER_EFFECT_CLASS(hightlight_effect_class);
	ClutterOffscreenEffectClass* offscreen_effect_class = CLUTTER_OFFSCREEN_EFFECT_CLASS(hightlight_effect_class);

	gobject_class->finalize = tangle_highlight_effect_finalize;
	gobject_class->dispose = tangle_highlight_effect_dispose;
	gobject_class->set_property = tangle_highlight_effect_set_property;
	gobject_class->get_property = tangle_highlight_effect_get_property;

	effect_class->pre_paint = tangle_highlight_effect_pre_paint;
	
	offscreen_effect_class->create_texture = tangle_highlight_effect_create_texture;
	offscreen_effect_class->paint_target = tangle_highlight_effect_paint_target;

	/**
	 * TangleHighlightEffect:color:
	 *
	 * The color of which is used when painting the highlight.
	 */
	g_object_class_install_property(gobject_class, PROP_COLOR,
	                                clutter_param_spec_color("color",
	                                                	 "Color",
	                                                	 "The color of which is used when painting the highlight",
	                                                	 &white,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleHighlightEffect:distance:
	 *
	 * The distance in pixels how far the highlight effect is painted.
	 */
	g_object_class_install_property(gobject_class, PROP_DISTANCE,
	                                g_param_spec_float("distance",
	                                "Distance",
	                                "The distance in pixels how far the highlight effect is painted",
	                                0.0, G_MAXFLOAT, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleHighlightEffectPrivate));
}

static void tangle_highlight_effect_init(TangleHighlightEffect* highlight_effect) {
	highlight_effect->priv = G_TYPE_INSTANCE_GET_PRIVATE(highlight_effect, TANGLE_TYPE_HIGHLIGHT_EFFECT, TangleHighlightEffectPrivate);

	highlight_effect->priv->color = white;
}

