/*
 * tangle-texture.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2011 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-texture.h"
#include "tangle-misc.h"
#include "marshalers.h"

G_DEFINE_TYPE(TangleTexture, tangle_texture, TANGLE_TYPE_ACTOR);

enum {
	PROP_0,
	PROP_FILENAME,
	PROP_REPEAT_X,
	PROP_REPEAT_Y
};

enum {
	SIZE_CHANGED,
	LAST_SIGNAL
};

struct _TangleTexturePrivate {
	gchar* filename;
	
	CoglMaterial* material;
	gint width;
	gint height;
	
	guint repeat_x : 1;
	guint repeat_y : 1;
};

static CoglMaterial* material_template = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

ClutterActor* tangle_texture_new(const gchar* filename) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_TEXTURE, "filename", filename, NULL));
}

void tangle_texture_set_from_file(TangleTexture* texture, const gchar* filename) {
	gchar* path;
	CoglHandle new_texture;
	gint width;
	gint height;
	gboolean size_changed;
	
	g_return_if_fail(TANGLE_IS_TEXTURE(texture));

	path = tangle_lookup_filename(filename);
	
	if (!filename) {
		g_free(texture->priv->filename);
		texture->priv->filename = NULL;
		
		if (texture->priv->material) {
			cogl_handle_unref(texture->priv->material);
			texture->priv->material = NULL;
		}
	} else if ((new_texture = cogl_texture_new_from_file(path, COGL_TEXTURE_NONE, COGL_PIXEL_FORMAT_ANY, NULL)) != COGL_INVALID_HANDLE) {
		width = cogl_texture_get_width(new_texture);
		height = cogl_texture_get_height(new_texture);
		size_changed = (width != texture->priv->width || height != texture->priv->height);

		if (!texture->priv->material) {
			texture->priv->material = cogl_material_copy(material_template);
		}
		cogl_material_set_layer(texture->priv->material, 0, new_texture);		
		cogl_handle_unref(new_texture);
		
		if (size_changed) {
			texture->priv->width = width;
			texture->priv->height = height;
		
			g_signal_emit(texture, signals[SIZE_CHANGED], 0, width, height);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(texture));
		} else {
			clutter_actor_queue_redraw(CLUTTER_ACTOR(texture));
		}
		
		g_free(texture->priv->filename);
		texture->priv->filename = g_strdup(filename);
		g_object_notify(G_OBJECT(texture), "filename");
	}
	
	g_free(path);	
}

void tangle_texture_get_repeat(TangleTexture* texture, gboolean* repeat_x_return, gboolean* repeat_y_return) {
	g_return_if_fail(TANGLE_IS_TEXTURE(texture));

	if (repeat_x_return) {
		*repeat_x_return = texture->priv->repeat_x;
	}
	if (repeat_y_return) {
		*repeat_y_return = texture->priv->repeat_y;
	}
}

void tangle_texture_set_repeat(TangleTexture* texture, gboolean repeat_x, gboolean repeat_y) {
	gboolean changed = FALSE;

	g_return_if_fail(TANGLE_IS_TEXTURE(texture));

	g_object_freeze_notify(G_OBJECT(texture));
	
	if (texture->priv->repeat_x != repeat_x) {
		texture->priv->repeat_x = repeat_x;
		g_object_notify(G_OBJECT(texture), "repeat-x");
		changed = TRUE;
	}
	if (texture->priv->repeat_y != repeat_y) {
		texture->priv->repeat_y = repeat_y;
		g_object_notify(G_OBJECT(texture), "repeat-y");
		changed = TRUE;
	}
	
	if (changed) {
		clutter_actor_queue_redraw(CLUTTER_ACTOR(texture));
	}
	
	g_object_thaw_notify(G_OBJECT(texture));
}

static void tangle_texture_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_return, gfloat* natural_width_return, gfloat* max_width_return) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(actor);
	
	if (min_width_return) {
		*min_width_return = 0;
	}
	if (natural_width_return) {
		if (for_height >= 0) {
			*natural_width_return = (gfloat)texture->priv->width / texture->priv->height * for_height;
		} else {
			*natural_width_return = texture->priv->width;
		}
	}
	if (max_width_return) {
		*max_width_return = 0;
	}
}

static void tangle_texture_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_return, gfloat* natural_height_return, gfloat* max_height_return) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(actor);
	
	if (min_height_return) {
		*min_height_return = 0;
	}
	if (natural_height_return) {
		if (for_width >= 0) {
			*natural_height_return = (gfloat)texture->priv->height / texture->priv->width * for_width;
		} else {
			*natural_height_return = texture->priv->height;
		}
	}
	if (max_height_return) {
		*max_height_return = 0;
	}
}

static void tangle_texture_paint_aligned(TangleActor* actor, gfloat width, gfloat height) {
	TangleTexture* texture;
	guint8 paint_opacity;
	gfloat w = 1.0, h = 1.0;
	
	texture = TANGLE_TEXTURE(actor);
	
	paint_opacity = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(actor));

	if (texture->priv->material) {
		if (texture->priv->repeat_x && texture->priv->width > 0) {
			w = width / texture->priv->width;
		}
		if (texture->priv->repeat_y && texture->priv->height > 0) {
			h = height / texture->priv->height;
		}

		cogl_material_set_color4ub(texture->priv->material, paint_opacity, paint_opacity, paint_opacity, paint_opacity);
		cogl_set_source(texture->priv->material);
		cogl_rectangle_with_texture_coords(0.0, 0.0, width, height, 0.0, 0.0, w, h);	
	} else {
		cogl_set_source_color4ub(255, 32, 64, 255);
		cogl_rectangle(0, 0, width, height);	
	}
}

static void tangle_texture_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(object);

	switch (prop_id) {
		case PROP_FILENAME:
			tangle_texture_set_from_file(texture, g_value_get_string(value));
			break;
		case PROP_REPEAT_X:
			tangle_texture_set_repeat(texture, g_value_get_boolean(value), texture->priv->repeat_y);
			break;
		case PROP_REPEAT_Y:
			tangle_texture_set_repeat(texture, texture->priv->repeat_x, g_value_get_boolean(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_texture_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleTexture* texture;

	texture = TANGLE_TEXTURE(object);

        switch (prop_id) {
		case PROP_FILENAME:
			g_value_set_string(value, texture->priv->filename);
			break;
		case PROP_REPEAT_X:
			g_value_set_boolean(value, texture->priv->repeat_x);
			break;
		case PROP_REPEAT_Y:
			g_value_set_boolean(value, texture->priv->repeat_y);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_texture_finalize(GObject* object) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(object);

	g_free(texture->priv->filename);	

	G_OBJECT_CLASS(tangle_texture_parent_class)->finalize(object);
}

static void tangle_texture_dispose(GObject* object) {
	TangleTexture* texture;
	
	texture = TANGLE_TEXTURE(object);

	if (texture->priv->material) {
		cogl_handle_unref(texture->priv->material);
		texture->priv->material = NULL;
	}
	
	G_OBJECT_CLASS(tangle_texture_parent_class)->dispose(object);
}

static void tangle_texture_class_init(TangleTextureClass* texture_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(texture_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(texture_class);

	gobject_class->finalize = tangle_texture_finalize;
	gobject_class->dispose = tangle_texture_dispose;
	gobject_class->set_property = tangle_texture_set_property;
	gobject_class->get_property = tangle_texture_get_property;
	
	actor_class->get_preferred_width = tangle_texture_get_preferred_width;
	actor_class->get_preferred_height = tangle_texture_get_preferred_height;
	actor_class->paint_aligned = tangle_texture_paint_aligned;

	/**
	 * TangleTexture:filename:
	 *
	 * The filename of the image shown.
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "Filename",
	                                "The filename of the image shown",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleTexture:repeat-x:
	 *
	 * Whether to repeat the contents rathen than scaling horizontally.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_X,
	                                g_param_spec_boolean("repeat-x",
	                                "Repeat X",
	                                "Whether to repeat the contents rathen than scaling horizontally",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleTexture:repeat-x:
	 *
	 * Whether to repeat the contents rathen than scaling vertically.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_Y,
	                                g_param_spec_boolean("repeat-y",
	                                "Repeat y",
	                                "Whether to repeat the contents rathen than scaling vertically",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * TangleTexture:size-changed:
	 * @texture: the object which received the signal
	 */
	signals[SIZE_CHANGED] = g_signal_new("size-changed", G_TYPE_FROM_CLASS(gobject_class),
	                                     G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleTextureClass, size_changed),
					     NULL, NULL,
					     tangle_marshal_VOID__INT_INT,
					     G_TYPE_NONE, 2,
					     G_TYPE_INT,
					     G_TYPE_INT);

	g_type_class_add_private (gobject_class, sizeof (TangleTexturePrivate));
}

static void tangle_texture_init(TangleTexture* texture) {
	CoglHandle texture_template;
	
	texture->priv = G_TYPE_INSTANCE_GET_PRIVATE(texture, TANGLE_TYPE_TEXTURE, TangleTexturePrivate);

	texture->priv->width = texture->priv->height = 32;
	
	if (!material_template) {
		material_template = cogl_material_new();
		texture_template = cogl_texture_new_with_size(1, 1, COGL_TEXTURE_NO_SLICING, COGL_PIXEL_FORMAT_RGBA_8888_PRE);
		cogl_material_set_layer(material_template, 0, texture_template);
		cogl_handle_unref(texture_template);
	}
}
