/*
 * tangle-hold-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2011 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-hold-action.h"

G_DEFINE_TYPE(TangleHoldAction, tangle_hold_action, CLUTTER_TYPE_ACTION);

enum {
	PROP_0,
	PROP_HOLD_TIMEOUT,
	PROP_PRESSED
};

enum {
	HELD,
	LAST_SIGNAL
};

struct _TangleHoldActionPrivate {
	guint hold_timeout;

	ClutterActor* stage;
	gulong event_handler_id;
	gulong captured_event_handler_id;
	guint timeout_source_id;
	gfloat start_x, start_y;

	guint pressed : 1;
};

static void disconnect_captured_event_handler(TangleHoldAction* hold_action);
static void set_pressed(TangleHoldAction* hold_action, gboolean is_pressed);
static gboolean on_button_press_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static gboolean on_timeout(gpointer user_data);

static guint signals[LAST_SIGNAL] = { 0 };

ClutterAction* tangle_hold_action_new(void) {

	return CLUTTER_ACTION(g_object_new(TANGLE_TYPE_HOLD_ACTION, NULL));
}

gboolean tangle_hold_action_get_pressed(TangleHoldAction* hold_action) {

	return hold_action->priv->pressed;
}

void tangle_hold_action_interrupt(TangleHoldAction* hold_action) {
	disconnect_captured_event_handler(hold_action);
	set_pressed(hold_action, FALSE);
}

static void tangle_hold_action_set_actor(ClutterActorMeta* actor_meta, ClutterActor* actor) {
	TangleHoldAction* hold_action;
	
	hold_action = TANGLE_HOLD_ACTION(actor_meta);

	if (hold_action->priv->event_handler_id) {
		g_signal_handler_disconnect(clutter_actor_meta_get_actor(actor_meta), hold_action->priv->event_handler_id);
		hold_action->priv->event_handler_id = 0;
	}
	disconnect_captured_event_handler(hold_action);
	
	if (actor) {
		hold_action->priv->event_handler_id = g_signal_connect(actor, "button-press-event", G_CALLBACK(on_button_press_event), actor_meta);
	}
	hold_action->priv->stage = NULL;
	
	CLUTTER_ACTOR_META_CLASS(tangle_hold_action_parent_class)->set_actor(actor_meta, actor);
}

static void tangle_hold_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
        TangleHoldAction* hold_action;

	hold_action = TANGLE_HOLD_ACTION(object);

        switch (prop_id) {
		case PROP_HOLD_TIMEOUT:
			hold_action->priv->hold_timeout = g_value_get_uint(value);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_hold_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleHoldAction* hold_action;

	hold_action = TANGLE_HOLD_ACTION(object);

        switch (prop_id) {
		case PROP_HOLD_TIMEOUT:
			g_value_set_uint(value, hold_action->priv->hold_timeout);
			break;
		case PROP_PRESSED:
			g_value_set_boolean(value, hold_action->priv->pressed);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_hold_action_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_hold_action_parent_class)->finalize(object);
}

static void tangle_hold_action_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_hold_action_parent_class)->dispose(object);
}

static void tangle_hold_action_class_init(TangleHoldActionClass* hold_action_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(hold_action_class);
	ClutterActorMetaClass* actor_meta = CLUTTER_ACTOR_META_CLASS(hold_action_class);

	gobject_class->set_property = tangle_hold_action_set_property;
	gobject_class->get_property = tangle_hold_action_get_property;
	
	actor_meta->set_actor = tangle_hold_action_set_actor;

	/**
	 * TangleHoldAction:hold-timeout:
	 *
	 * The time in milliseconds of which hold (press/tap) must last to emit the ::held signal.
	 */
	g_object_class_install_property(gobject_class, PROP_HOLD_TIMEOUT,
	                                g_param_spec_uint("hold-timeout",
	                                                  "Hold timeout",
	                                                  "The time in milliseconds of which hold (press/tap) must last to emit the held signal",
	                                                  0, G_MAXUINT, 1500,
	                                                  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * TangleHoldAction:pressed:
	 *
	 * Whether the holding (pressing/tapping) is currently in action.
	 */
	g_object_class_install_property(gobject_class, PROP_PRESSED,
	                                g_param_spec_boolean("pressed",
	                                                     "Pressed",
	                                                     "Whether the holding (pressing/tapping) is currently in action.",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	signals[HELD] = g_signal_new("held",
	                             G_TYPE_FROM_CLASS(gobject_class),
				     G_SIGNAL_RUN_LAST,
				     G_STRUCT_OFFSET(TangleHoldActionClass, held),
				     NULL, NULL,
				     g_cclosure_marshal_VOID__VOID,
				     G_TYPE_NONE, 0);

	g_type_class_add_private(gobject_class, sizeof(TangleHoldActionPrivate));
}

static void tangle_hold_action_init(TangleHoldAction* hold_action) {
	hold_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(hold_action, TANGLE_TYPE_HOLD_ACTION, TangleHoldActionPrivate);
	hold_action->priv->hold_timeout = 1500;
}

static void set_pressed(TangleHoldAction* hold_action, gboolean is_pressed) {
	if (hold_action->priv->pressed != is_pressed) {
		hold_action->priv->pressed = is_pressed;
		g_object_notify(G_OBJECT(hold_action), "pressed");
	}
}

static void disconnect_captured_event_handler(TangleHoldAction* hold_action) {
	if (hold_action->priv->captured_event_handler_id) {
		g_signal_handler_disconnect(hold_action->priv->stage, hold_action->priv->captured_event_handler_id);
		hold_action->priv->captured_event_handler_id = 0;
		
		g_source_remove(hold_action->priv->timeout_source_id);
	}
}

static gboolean on_button_press_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleHoldAction* hold_action;
	
	hold_action = TANGLE_HOLD_ACTION(user_data);
	
	if (clutter_actor_meta_get_enabled(CLUTTER_ACTOR_META(hold_action)) &&
	    clutter_event_get_click_count(event) == 1 &&
	    clutter_actor_contains(actor, clutter_event_get_source(event))) {
		if (!hold_action->priv->stage) {
			hold_action->priv->stage = clutter_actor_get_stage(actor);
		}
		clutter_event_get_coords(event, &hold_action->priv->start_x, &hold_action->priv->start_y);
		hold_action->priv->captured_event_handler_id = g_signal_connect_after(hold_action->priv->stage,
		                                                                      "captured-event",
										      G_CALLBACK(on_captured_event),
										      hold_action);
		hold_action->priv->timeout_source_id = g_timeout_add(hold_action->priv->hold_timeout, on_timeout, hold_action);
		set_pressed(hold_action, TRUE);
	}
	
	return FALSE;
}

#define ABS_F(x) ((x) < 0 ? -(x) : (x))

static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleHoldAction* hold_action;
	gfloat x, y;
	ClutterSettings* settings;
	gint double_click_distance;

	hold_action = TANGLE_HOLD_ACTION(user_data);
	
	switch (clutter_event_type(event)) {
		case CLUTTER_BUTTON_RELEASE:
			if (clutter_event_get_click_count(event) == 1) {
				disconnect_captured_event_handler(hold_action);
				set_pressed(hold_action, FALSE);
			}
			break;
		case CLUTTER_MOTION:
			settings = clutter_settings_get_default();
			g_object_get(settings, "double-click-distance", &double_click_distance, NULL);
			clutter_event_get_coords(event, &x, &y);
			if (ABS_F(x - hold_action->priv->start_x) > double_click_distance ||
			    ABS_F(y - hold_action->priv->start_y) > double_click_distance ) {
				disconnect_captured_event_handler(hold_action);
				set_pressed(hold_action, FALSE);
			}
			break;
	}
	
	return FALSE;
}

static gboolean on_timeout(gpointer user_data) {
	TangleHoldAction* hold_action;
	
	hold_action = TANGLE_HOLD_ACTION(user_data);

	g_signal_emit(hold_action, signals[HELD], 0, clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(hold_action)));

	return FALSE;
}
