/*
 * tangle-scroller.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-scroller.h"
#include "tangle-vault.h"

/**
 * SECTION:tangle-scroller
 * @Short_description: A  actor that scrolls its content
 * @Title: TangleScroller
 */

G_DEFINE_TYPE(TangleScroller, tangle_scroller, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_SCROLLING_OFFSET_X,
	PROP_SCROLLING_OFFSET_Y,
	PROP_SCROLLING_WIDTH,
	PROP_SCROLLING_HEIGHT,
	PROP_SCROLLING_THRESHOLD_X,
	PROP_SCROLLING_THRESHOLD_Y,
	PROP_MAX_OVERSHOOT_X,
	PROP_MAX_OVERSHOOT_Y,
	PROP_REPEAT_X,
	PROP_REPEAT_Y,
	PROP_CONTINUOUS_X,
	PROP_CONTINUOUS_Y,
	PROP_KINETIC_SCROLLING,
	PROP_SCROLLING_DECELERATION
};

enum {
	CLAMP_SCROLLING_OFFSET_X,
	CLAMP_SCROLLING_OFFSET_Y,
	LAST_SIGNAL
};

struct _TangleScrollerPrivate {
	gfloat scrolling_offset_x;
	gfloat scrolling_offset_y;
	gfloat scrolling_threshold_x;
	gfloat scrolling_threshold_y;
	gfloat max_overshoot_x;
	gfloat max_overshoot_y;
	gdouble scrolling_deceleration;

	gfloat scrolling_width;
	gfloat scrolling_height;
	
	gfloat start_motion_x;
	guint32 start_motion_x_time;
	gfloat start_motion_y;
	guint32 start_motion_y_time;
	gfloat farest_motion_x;
	guint32 farest_motion_x_time;
	gfloat farest_motion_y;
	guint32 farest_motion_y_time;
	
	guint32 previous_motion_time;

	gfloat start_scrolling_motion_x;
	gfloat start_scrolling_motion_y;
	gfloat start_scrolling_offset_x;
	gfloat start_scrolling_offset_y;

	gulong captured_event_handler_id;
	ClutterAnimation* animation_x;
	ClutterAnimation* animation_y;

	guint repeat_x : 1;
	guint repeat_y : 1;
	guint continuous_x : 1;
	guint continuous_y : 1;
	guint kinetic_scrolling : 1;

	guint descendant_interacting : 1;	
	guint descendant_dragging : 1;
	guint scrolling_effective : 1;
	guint going_right : 1;
	guint going_down : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };

static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static void send_interaction_event(TangleActor* actor, gboolean began);
static gfloat get_width(TangleScroller* 	scroller);
static gfloat get_height(TangleScroller* scroller);
static gfloat normalise_scrolling_offset_x(TangleScroller* scroller, gfloat scrolling_offset_x, gfloat* delta_return);
static gfloat normalise_scrolling_offset_y(TangleScroller* scroller, gfloat scrolling_offset_y, gfloat* delta_return);

ClutterActor* tangle_scroller_new(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_SCROLLER, NULL));
}

gfloat tangle_scroller_get_scrolling_offset_x(TangleScroller* scroller) {

	return scroller->priv->scrolling_offset_x;
}

void tangle_scroller_set_scrolling_offset_x(TangleScroller* scroller, gfloat scrolling_offset_x) {
	gfloat width;
	
	if (scroller->priv->scrolling_offset_x != scrolling_offset_x) {
		scroller->priv->scrolling_offset_x = normalise_scrolling_offset_x(scroller, scrolling_offset_x, NULL);
		clutter_actor_queue_relayout(CLUTTER_ACTOR(scroller));
		g_object_notify(G_OBJECT(scroller), "scrolling-offset-x");
	}	
}

gfloat tangle_scroller_get_scrolling_offset_y(TangleScroller* scroller) {

	return scroller->priv->scrolling_offset_y;
}

void tangle_scroller_set_scrolling_offset_y(TangleScroller* scroller, gfloat scrolling_offset_y) {
	gfloat height;
	
	if (scroller->priv->scrolling_offset_y != scrolling_offset_y) {
		scroller->priv->scrolling_offset_y = normalise_scrolling_offset_y(scroller, scrolling_offset_y, NULL);
		clutter_actor_queue_relayout(CLUTTER_ACTOR(scroller));
		g_object_notify(G_OBJECT(scroller), "scrolling-offset-y");
	}
}

gfloat tangle_scroller_get_scrolling_width(TangleScroller* scroller) {

	return scroller->priv->scrolling_width;
}

gfloat tangle_scroller_get_scrolling_height(TangleScroller* scroller) {

	return scroller->priv->scrolling_height;
}

void tangle_scroller_set_scrolling_threshold_x(TangleScroller* scroller, gfloat scrolling_threshold_x) {
	if (scroller->priv->scrolling_threshold_x != scrolling_threshold_x) {
		scroller->priv->scrolling_threshold_x = scrolling_threshold_x;
		g_object_notify(G_OBJECT(scroller), "scrolling-threshold-x");
	}
}

gfloat tangle_scroller_get_scrolling_threshold_y(TangleScroller* scroller) {

	return scroller->priv->scrolling_threshold_y;
}

void tangle_scroller_set_scrolling_threshold_y(TangleScroller* scroller, gfloat scrolling_threshold_y) {
	if (scroller->priv->scrolling_threshold_y != scrolling_threshold_y) {
		scroller->priv->scrolling_threshold_y = scrolling_threshold_y;
		g_object_notify(G_OBJECT(scroller), "scrolling-threshold-y");
	}
}

void tangle_scroller_set_scrolling_threshold(TangleScroller* scroller, gfloat scrolling_threshold) {
	tangle_scroller_set_scrolling_threshold_x(scroller, scrolling_threshold);
	tangle_scroller_set_scrolling_threshold_y(scroller, scrolling_threshold);
}

gboolean tangle_scroller_get_repeat_x(TangleScroller* scroller) {

	return scroller->priv->repeat_x;
}

void tangle_scroller_set_repeat_x(TangleScroller* scroller, gboolean repeat_x) {
	if (scroller->priv->repeat_x != repeat_x) {
		g_object_freeze_notify(G_OBJECT(scroller));

		scroller->priv->repeat_x = repeat_x;
		g_object_notify(G_OBJECT(scroller), "repeat-x");
		
		if (!repeat_x) {
			tangle_scroller_set_continuous_x(scroller, FALSE);
		}

		g_object_thaw_notify(G_OBJECT(scroller));
	}
}

gboolean tangle_scroller_get_repeat_y(TangleScroller* scroller) {

	return scroller->priv->repeat_y;
}

void tangle_scroller_set_repeat_y(TangleScroller* scroller, gboolean repeat_y) {
	if (scroller->priv->repeat_y != repeat_y) {
		g_object_freeze_notify(G_OBJECT(scroller));

		scroller->priv->repeat_y = repeat_y;
		g_object_notify(G_OBJECT(scroller), "repeat-y");
		
		if (!repeat_y) {
			tangle_scroller_set_continuous_y(scroller, FALSE);
		}

		g_object_thaw_notify(G_OBJECT(scroller));
	}
}

gboolean tangle_scroller_get_continuous_x(TangleScroller* scroller) {

	return scroller->priv->continuous_x;
}

void tangle_scroller_set_continuous_x(TangleScroller* scroller, gboolean continuous_x) {
	if (scroller->priv->continuous_x != continuous_x) {
		g_object_freeze_notify(G_OBJECT(scroller));

		scroller->priv->continuous_x = continuous_x;
		g_object_notify(G_OBJECT(scroller), "continuous-x");

		if (continuous_x) {
			tangle_scroller_set_repeat_x(scroller, TRUE);
		}

		g_object_thaw_notify(G_OBJECT(scroller));
	}
}

gboolean tangle_scroller_get_continuous_y(TangleScroller* scroller) {

	return scroller->priv->continuous_y;
}

void tangle_scroller_set_continuous_y(TangleScroller* scroller, gboolean continuous_y) {
	if (scroller->priv->continuous_y != continuous_y) {
		g_object_freeze_notify(G_OBJECT(scroller));

		scroller->priv->continuous_y = continuous_y;
		g_object_notify(G_OBJECT(scroller), "continuous-y");
		
		if (continuous_y) {
			tangle_scroller_set_repeat_y(scroller, TRUE);
		}

		g_object_thaw_notify(G_OBJECT(scroller));
	}
}

gboolean tangle_scroller_get_kinetic_scrolling(TangleScroller* scroller) {

	return scroller->priv->kinetic_scrolling;
}

void tangle_scroller_set_kinetic_scrolling(TangleScroller* scroller, gboolean kinetic_scrolling) {
	scroller->priv->kinetic_scrolling = kinetic_scrolling;
}

gdouble tangle_scroller_get_scrolling_deceleration(TangleScroller* scroller) {

	return scroller->priv->scrolling_deceleration;
}

void tangle_scroller_set_scrolling_deceleration(TangleScroller* scroller, gdouble scrolling_deceleration) {
	scroller->priv->scrolling_deceleration = scrolling_deceleration;
}

void tangle_scroller_clamp_scrolling_offset_x(TangleScroller* scroller) {
	TangleClamp* clamp;
	
	clamp = tangle_clamp_new(TANGLE_CLAMP_HORIZONTAL, scroller->priv->scrolling_offset_x);
	g_signal_emit(scroller, signals[CLAMP_SCROLLING_OFFSET_X], 0, clamp);
	if (tangle_clamp_get_clamped_value_set(clamp)) {
		tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_IN_OUT_QUAD, 300, "scrolling-offset-x", tangle_clamp_get_clamped_value(clamp), NULL);		
	}
	g_object_unref(clamp);
}

void tangle_scroller_clamp_scrolling_offset_y(TangleScroller* scroller) {
	TangleClamp* clamp;
	
	clamp = tangle_clamp_new(TANGLE_CLAMP_VERTICAL, scroller->priv->scrolling_offset_y);
	g_signal_emit(scroller, signals[CLAMP_SCROLLING_OFFSET_Y], 0, clamp);
	if (tangle_clamp_get_clamped_value_set(clamp)) {
		tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_IN_OUT_QUAD, 300, "scrolling-offset-y", tangle_clamp_get_clamped_value(clamp), NULL);		
	}
	g_object_unref(clamp);
}

void tangle_scroller_clamp_offsets(TangleScroller* scroller) {
	tangle_scroller_clamp_scrolling_offset_x(scroller);
	tangle_scroller_clamp_scrolling_offset_y(scroller);
}

void tangle_scroller_clamp_page_boundaries(TangleScroller* scroller, TangleClamp* clamp) {
	TangleClampDirection direction;
	gfloat original_value;
	gfloat page_size;
	
	g_return_if_fail(TANGLE_IS_SCROLLER(scroller));
	
	direction = tangle_clamp_get_direction(clamp);
	original_value = tangle_clamp_get_original_value(clamp);
	
	if (direction == TANGLE_CLAMP_HORIZONTAL) {
		if (original_value < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (original_value > scroller->priv->scrolling_width) {
			tangle_clamp_set_clamped_value(clamp, scroller->priv->scrolling_width);
		} else {
			page_size = get_width(scroller);
			tangle_clamp_set_clamped_value(clamp, ((guint)((original_value + page_size / 2) / page_size)) * page_size);
		}
	} else if (direction == TANGLE_CLAMP_VERTICAL) {
		if (original_value < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (original_value > scroller->priv->scrolling_height) {
			tangle_clamp_set_clamped_value(clamp, scroller->priv->scrolling_height);
		} else {
			page_size = get_height(scroller);
			tangle_clamp_set_clamped_value(clamp, ((guint)((original_value + page_size / 2) / page_size)) * page_size);
		}
	}
}

static gboolean tangle_scroller_button_press_event(ClutterActor* actor, ClutterButtonEvent* event) {
	TangleScroller* scroller;
		
	scroller = TANGLE_SCROLLER(actor);

	if (!scroller->priv->captured_event_handler_id) {
		if (scroller->priv->animation_x) {
			tangle_actor_stop_animation(TANGLE_ACTOR(scroller), scroller->priv->animation_x);
		}
		if (scroller->priv->animation_y) {
			tangle_actor_stop_animation(TANGLE_ACTOR(scroller), scroller->priv->animation_y);
		}
		scroller->priv->start_scrolling_motion_x = event->x;
		scroller->priv->start_scrolling_motion_y = event->y;
		scroller->priv->scrolling_effective = FALSE;
		scroller->priv->captured_event_handler_id = g_signal_connect(clutter_actor_get_stage(actor), "captured-event", G_CALLBACK(on_captured_event), scroller);
	}

	return FALSE;
}


static void tangle_scroller_allocate_children(TangleWidget* widget, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleScroller* scroller;
	gfloat old_width, old_height;
	ClutterActorBox actor_box;

	scroller = TANGLE_SCROLLER(widget);

	old_width = scroller->priv->scrolling_width;
	old_height = scroller->priv->scrolling_height;
	tangle_widget_get_real_preferred_size(widget, tangle_actor_get_interacting(TANGLE_ACTOR(scroller)), NULL, NULL, &scroller->priv->scrolling_width, &scroller->priv->scrolling_height, NULL, NULL);
	if (scroller->priv->scrolling_width < box->x2 - box->x1) {
		scroller->priv->scrolling_width = box->x2 - box->x1;
	}
	if (scroller->priv->scrolling_height < box->y2 - box->y1) {
		scroller->priv->scrolling_height = box->y2 - box->y1;
	}
	if (old_width != scroller->priv->scrolling_width) {
		g_object_notify(G_OBJECT(scroller), "scrolling-width");
	}
	if (old_height != scroller->priv->scrolling_height) {
		g_object_notify(G_OBJECT(scroller), "scrolling-height");
	}

	actor_box.x1 = box->x1 - scroller->priv->scrolling_offset_x;
	actor_box.y1 = box->y1 - scroller->priv->scrolling_offset_y;
	actor_box.x2 = actor_box.x1 + scroller->priv->scrolling_width;
	actor_box.y2 = actor_box.y1 + scroller->priv->scrolling_height;

printf("%f %f - %f %f\n", actor_box.x1, actor_box.y1, actor_box.x2, actor_box.y2);

	TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->allocate_children(widget, &actor_box, flags);
	
	scroller->priv->scrolling_width -= box->x2 - box->x1;
	scroller->priv->scrolling_height -= box->y2 - box->y1;
}

static void tangle_scroller_paint_children(TangleWidget* widget) {
	TangleScroller* scroller;
	ClutterActorBox actor_box;
	
	scroller = TANGLE_SCROLLER(widget);

	TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
	clutter_actor_get_allocation_box(CLUTTER_ACTOR(scroller), &actor_box);
	
	if (scroller->priv->repeat_x && scroller->priv->scrolling_offset_x < 0.0 &&
	    scroller->priv->repeat_y && scroller->priv->scrolling_offset_y < 0.0) {
		cogl_translate(-scroller->priv->scrolling_width - (actor_box.x2 - actor_box.x1), -scroller->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(scroller->priv->scrolling_width + (actor_box.x2 - actor_box.x1), scroller->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scroller->priv->repeat_x && scroller->priv->scrolling_offset_x < 0.0 &&
	    scroller->priv->repeat_y && scroller->priv->scrolling_offset_y > scroller->priv->scrolling_height) {
		cogl_translate(-scroller->priv->scrolling_width - (actor_box.x2 - actor_box.x1), scroller->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(scroller->priv->scrolling_width + (actor_box.x2 - actor_box.x1), -scroller->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scroller->priv->repeat_x && scroller->priv->scrolling_offset_x < 0.0) {
		cogl_translate(-scroller->priv->scrolling_width - (actor_box.x2 - actor_box.x1), 0.0, 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(scroller->priv->scrolling_width + (actor_box.x2 - actor_box.x1), 0.0, 0.0);
	}
	if (scroller->priv->repeat_y && scroller->priv->scrolling_offset_y < 0.0) {
		cogl_translate(0.0, -scroller->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(0.0, scroller->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scroller->priv->repeat_x && scroller->priv->scrolling_offset_x > scroller->priv->scrolling_width &&
	    scroller->priv->repeat_y && scroller->priv->scrolling_offset_y > scroller->priv->scrolling_height) {
		cogl_translate(scroller->priv->scrolling_width + (actor_box.x2 - actor_box.x1), scroller->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(-scroller->priv->scrolling_width - (actor_box.x2 - actor_box.x1), -scroller->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scroller->priv->repeat_x && scroller->priv->scrolling_offset_x > scroller->priv->scrolling_width &&
	    scroller->priv->repeat_y && scroller->priv->scrolling_offset_y < 0.0) {
		cogl_translate(scroller->priv->scrolling_width + (actor_box.x2 - actor_box.x1), -scroller->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(-scroller->priv->scrolling_width - (actor_box.x2 - actor_box.x1), scroller->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
	}
	if (scroller->priv->repeat_x && scroller->priv->scrolling_offset_x > scroller->priv->scrolling_width) {
		cogl_translate(scroller->priv->scrolling_width + (actor_box.x2 - actor_box.x1), 0.0, 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(-scroller->priv->scrolling_width - (actor_box.x2 - actor_box.x1), 0.0, 0.0);
	}
	if (scroller->priv->repeat_y && scroller->priv->scrolling_offset_y > scroller->priv->scrolling_height) {
		cogl_translate(0.0, scroller->priv->scrolling_height + (actor_box.y2 - actor_box.y1), 0.0);
		TANGLE_WIDGET_CLASS(tangle_scroller_parent_class)->paint_children(widget);
		cogl_translate(0.0, -scroller->priv->scrolling_height - (actor_box.y2 - actor_box.y1), 0.0);
	}
}

static void tangle_scroller_clamp_scrolling_offset_x_impl(TangleScroller* scroller, TangleClamp* clamp) {
	if (!tangle_clamp_get_clamped_value_set(clamp) && !scroller->priv->continuous_x) {
		if (scroller->priv->scrolling_offset_x < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (scroller->priv->scrolling_offset_x > scroller->priv->scrolling_width) {
			tangle_clamp_set_clamped_value(clamp, scroller->priv->scrolling_width);
		}
	}
}

static void tangle_scroller_clamp_scrolling_offset_y_impl(TangleScroller* scroller, TangleClamp* clamp) {
	if (!tangle_clamp_get_clamped_value_set(clamp) && !scroller->priv->continuous_y) {
		if (scroller->priv->scrolling_offset_y < 0.0) {
			tangle_clamp_set_clamped_value(clamp, 0.0);
		} else if (scroller->priv->scrolling_offset_y > scroller->priv->scrolling_height) {
			tangle_clamp_set_clamped_value(clamp, scroller->priv->scrolling_height);
		}
	}
}

static void tangle_scroller_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleScroller* scroller;
	
	scroller = TANGLE_SCROLLER(object);

	switch (prop_id) {
		case PROP_SCROLLING_OFFSET_X:
			tangle_scroller_set_scrolling_offset_x(scroller, g_value_get_float(value));
			break;
		case PROP_SCROLLING_OFFSET_Y:
			tangle_scroller_set_scrolling_offset_y(scroller, g_value_get_float(value));
			break;
		case PROP_SCROLLING_THRESHOLD_X:
			tangle_scroller_set_scrolling_threshold_x(scroller, g_value_get_float(value));
			break;
		case PROP_SCROLLING_THRESHOLD_Y:
			tangle_scroller_set_scrolling_threshold_y(scroller, g_value_get_float(value));
			break;
		case PROP_MAX_OVERSHOOT_X:
			scroller->priv->max_overshoot_x = g_value_get_float(value);
			break;
		case PROP_MAX_OVERSHOOT_Y:
			scroller->priv->max_overshoot_y = g_value_get_float(value);
			break;
		case PROP_REPEAT_X:
			tangle_scroller_set_repeat_x(scroller, g_value_get_boolean(value));
			break;
		case PROP_REPEAT_Y:
			tangle_scroller_set_repeat_y(scroller, g_value_get_boolean(value));
			break;
		case PROP_CONTINUOUS_X:
			tangle_scroller_set_continuous_x(scroller, g_value_get_boolean(value));
			break;
		case PROP_CONTINUOUS_Y:
			tangle_scroller_set_continuous_y(scroller, g_value_get_boolean(value));
			break;
		case PROP_KINETIC_SCROLLING:
			tangle_scroller_set_kinetic_scrolling(scroller, g_value_get_boolean(value));
			break;
		case PROP_SCROLLING_DECELERATION:
			tangle_scroller_set_scrolling_deceleration(scroller, g_value_get_double(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_scroller_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleScroller* scroller;

	scroller = TANGLE_SCROLLER(object);

        switch (prop_id) {
		case PROP_SCROLLING_OFFSET_X:
			g_value_set_float(value, scroller->priv->scrolling_offset_x);
			break;
		case PROP_SCROLLING_OFFSET_Y:
			g_value_set_float(value, scroller->priv->scrolling_offset_y);
			break;
		case PROP_SCROLLING_WIDTH:
			g_value_set_float(value, scroller->priv->scrolling_width);
			break;
		case PROP_SCROLLING_HEIGHT:
			g_value_set_float(value, scroller->priv->scrolling_height);
			break;
		case PROP_SCROLLING_THRESHOLD_X:
			g_value_set_float(value, scroller->priv->scrolling_threshold_x);
			break;
		case PROP_SCROLLING_THRESHOLD_Y:
			g_value_set_float(value, scroller->priv->scrolling_threshold_y);
			break;
		case PROP_MAX_OVERSHOOT_X:
			g_value_set_float(value, scroller->priv->max_overshoot_x);
			break;
		case PROP_MAX_OVERSHOOT_Y:
			g_value_set_float(value, scroller->priv->max_overshoot_y);
			break;
		case PROP_REPEAT_X:
			g_value_set_boolean(value, scroller->priv->repeat_x);
			break;
		case PROP_REPEAT_Y:
			g_value_set_boolean(value, scroller->priv->repeat_y);
			break;
		case PROP_CONTINUOUS_X:
			g_value_set_boolean(value, scroller->priv->continuous_x);
			break;
		case PROP_CONTINUOUS_Y:
			g_value_set_boolean(value, scroller->priv->continuous_y);
			break;
		case PROP_KINETIC_SCROLLING:
			g_value_set_boolean(value, scroller->priv->kinetic_scrolling);
			break;
		case PROP_SCROLLING_DECELERATION:
			g_value_set_double(value, scroller->priv->scrolling_deceleration);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_scroller_dispose(GObject* object) {
	TangleScroller* scroller;
	
	scroller = TANGLE_SCROLLER(object);
	
	if (scroller->priv->captured_event_handler_id) {
		g_signal_handler_disconnect(scroller, scroller->priv->captured_event_handler_id);
		scroller->priv->captured_event_handler_id = 0;
	}

	G_OBJECT_CLASS(tangle_scroller_parent_class)->dispose(object);
}

static void tangle_scroller_class_init(TangleScrollerClass* scroller_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(scroller_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(scroller_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(scroller_class);
	TangleWidgetClass* widget_class = TANGLE_WIDGET_CLASS(scroller_class);

	gobject_class->dispose = tangle_scroller_dispose;
	gobject_class->set_property = tangle_scroller_set_property;
	gobject_class->get_property = tangle_scroller_get_property;
	
	clutter_actor_class->button_press_event = tangle_scroller_button_press_event;
	
	widget_class->allocate_children = tangle_scroller_allocate_children;
	widget_class->paint_children = tangle_scroller_paint_children;
	
	scroller_class->clamp_scrolling_offset_x = tangle_scroller_clamp_scrolling_offset_x_impl;
	scroller_class->clamp_scrolling_offset_y = tangle_scroller_clamp_scrolling_offset_y_impl;
	
	/**
	 * TangleScroller:scrolling-offset-x:
	 *
	 * The offset to the left side of the child actors' bounding box.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_OFFSET_X,
	                                g_param_spec_float("scrolling-offset-x",
	                                                    "Scrolling offset X",
	                                                    "The offset to the left side of the child actors' bounding box",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:scrolling-offset-y:
	 *
	 * The offset to the top side of the child actors' bounding box.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_OFFSET_Y,
	                                g_param_spec_float("scrolling-offset-y",
	                                                    "Scrolling offset Y",
	                                                    "The offset to the top side of the child actors' bounding box",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:scrolling-width:
	 *
	 * The maximum value for the scrolling-offset-x: property.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_WIDTH,
	                                g_param_spec_float("scrolling-width",
	                                                    "Scrolling width",
	                                                    "The maximum value for the scrolling-offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:scrolling-height:
	 *
	 * The maximum value for the scrolling-offset-y: property.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_HEIGHT,
	                                g_param_spec_float("scrolling-height",
	                                                    "Scrolling height",
	                                                    "The maximum value for the scrolling-offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:scrolling-threshold-x:
	 *
	 * The horizontal threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_THRESHOLD_X,
	                                g_param_spec_float("scrolling-threshold-x",
	                                                    "Scrolling threshold X",
	                                                    "The horizontal threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:scrolling-threshold-y:
	 *
	 * The vertical threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_THRESHOLD_Y,
	                                g_param_spec_float("scrolling-threshold-y",
	                                                    "Scrolling threshold Y",
	                                                    "The vertical threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:max-overshoot-x:
	 *
	 * The horizontal maximum of overshooting.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OVERSHOOT_X,
	                                g_param_spec_float("max-overshoot-x",
	                                                    "Max overshoot X",
	                                                    "The horizontal maximum of overshooting",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:max-overshoot-y:
	 *
	 * The vertical maximum of overshooting.
	 */
	g_object_class_install_property(gobject_class, PROP_MAX_OVERSHOOT_Y,
	                                g_param_spec_float("max-overshoot-y",
	                                                    "Max overshoot Y",
	                                                    "The vertical maximum of overshooting",
	                                                   0.0, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:repeat-x:
	 *
	 * Whether to repeat the child actors painting when overshooting horizontally.
	 * Setting this to FALSE sets also :continuous-x to FALSE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_X,
	                                g_param_spec_boolean("repeat-x",
	                                                    "Repeat X",
	                                                    "Whether to repeat the child actors painting when overshooting horizontally",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:repeat-y:
	 *
	 * Whether to repeat the child actors painting when overshooting vertically.
	 * Setting this to FALSE sets also :continuous-y to FALSE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_REPEAT_Y,
	                                g_param_spec_boolean("repeat-y",
	                                                    "Repeat Y",
	                                                    "Whether to repeat the child actors painting when overshooting vertically",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:continuous-x:
	 *
	 * Whether to connect the left and right edges of the child actors to enable infinite horizontal scrolling.
	 * Setting this to TRUE sets also :repeat-x to TRUE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_CONTINUOUS_X,
	                                g_param_spec_boolean("continuous-x",
	                                                    "Continuous X",
	                                                    "Whether to connect the left and right edges of the child actors to enable infinite horizontal scrolling",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:continuous-y:
	 *
	 * Whether to connect the top and bottom edges of the child actors to enable infinite vertical scrolling.
	 * Setting this to TRUE sets also :repeat-y to TRUE as a side effect.
	 */
	g_object_class_install_property(gobject_class, PROP_CONTINUOUS_Y,
	                                g_param_spec_boolean("continuous-y",
	                                                    "Continuous Y",
	                                                    "Whether to connect the top and bottom edges of the child actors to enable infinite vertical scrolling",
	                                                    FALSE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:kinetic-scrolling:
	 *
	 * Whether kinetic scrolling is enabled or not.
	 */
	g_object_class_install_property(gobject_class, PROP_KINETIC_SCROLLING,
	                                g_param_spec_boolean("kinetic-scrolling",
	                                                    "Kinetic scrolling",
	                                                    "Whether kinetic scrolling is enabled or not",
	                                                    TRUE,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScroller:scrolling-deceleration:
	 *
	 * Deceleration of kinetic scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_DECELERATION,
	                                g_param_spec_boolean("scrolling-deceleration",
	                                                    "Scrolling deceleration",
	                                                    "Deceleration of kinetic scrolling",
	                                                    0.7,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	signals[CLAMP_SCROLLING_OFFSET_X] = g_signal_new("clamp-scrolling-offset-x",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleScrollerClass, clamp_scrolling_offset_x),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       TANGLE_TYPE_CLAMP);
	signals[CLAMP_SCROLLING_OFFSET_Y] = g_signal_new("clamp-scrolling-offset-y",
	                                       G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST,
					       G_STRUCT_OFFSET(TangleScrollerClass, clamp_scrolling_offset_y),
					       NULL, NULL,
					       g_cclosure_marshal_VOID__OBJECT,
					       G_TYPE_NONE, 1,
					       TANGLE_TYPE_CLAMP);

	g_type_class_add_private(gobject_class, sizeof(TangleScrollerPrivate));
}

static void tangle_scroller_init(TangleScroller* scroller) {
	scroller->priv = G_TYPE_INSTANCE_GET_PRIVATE(scroller, TANGLE_TYPE_SCROLLER, TangleScrollerPrivate);
	scroller->priv->kinetic_scrolling = TRUE;
	scroller->priv->scrolling_deceleration = 0.7;

	clutter_actor_set_reactive(CLUTTER_ACTOR(scroller), TRUE);
	/* TODO: clutter_actor_set_clip_to_allocation(CLUTTER_ACTOR(scroller), TRUE); */
	g_object_set(CLUTTER_ACTOR(scroller), "clip-to-allocation", TRUE, NULL);
}

static void on_overshooting_animation_completed(ClutterAnimation* animation, gpointer user_data) {
	TangleVault* vault;
	TangleScroller* scroller;
	const gchar* property;
	gfloat value;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 3, TANGLE_TYPE_SCROLLER, &scroller, G_TYPE_STRING, &property, G_TYPE_FLOAT, &value);
	
	tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_IN_OUT_QUAD, 350, property, value, NULL);
}

static void handle_overshooting_animation(TangleScroller* scroller, ClutterAnimation* animation, const gchar* property, gfloat value) {
	TangleVault* vault;

	vault = tangle_vault_new(3, TANGLE_TYPE_SCROLLER, scroller, G_TYPE_STRING, property, G_TYPE_FLOAT, value);
	tangle_signal_connect_vault_flags(animation, "completed", G_CALLBACK(on_overshooting_animation_completed), vault, G_CONNECT_AFTER);
}

#define ABS_F(x) ((x) < 0 ? -(x) : (x))
static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleScroller* scroller;
	gboolean handled = TRUE;
	ClutterBackend* backend;
	guint32 event_time;
	gfloat x, y, dx, dy;
	gdouble vx, vy, ax, ay;
	guint tx, ty;
	TangleClamp* clamp;
	gfloat delta;

	if (event->type == CLUTTER_MOTION || event->type == CLUTTER_BUTTON_RELEASE) {
		scroller = TANGLE_SCROLLER(user_data);	
		
		g_object_ref(scroller);

		backend = clutter_get_default_backend();
		
		clutter_event_get_coords(event, &x, &y);
		event_time = clutter_event_get_time(event);
		dx = x - scroller->priv->start_scrolling_motion_x;
		dy = y - scroller->priv->start_scrolling_motion_y;

		if (scroller->priv->scrolling_effective ||
		    (!scroller->priv->descendant_dragging &&
		     (!scroller->priv->descendant_interacting ||
		      ABS_F(dx) > scroller->priv->scrolling_threshold_y ||
		      ABS_F(dy) > scroller->priv->scrolling_threshold_x))) {

			if (!scroller->priv->scrolling_effective ||
			    event_time - scroller->priv->previous_motion_time > clutter_backend_get_double_click_time(backend)) {
				if (!scroller->priv->scrolling_effective) {		
					scroller->priv->scrolling_effective = TRUE;
					scroller->priv->start_scrolling_motion_x = x;
					scroller->priv->start_scrolling_motion_y = y;
					scroller->priv->start_scrolling_offset_x = scroller->priv->scrolling_offset_x;
					scroller->priv->start_scrolling_offset_y = scroller->priv->scrolling_offset_y;
					dx = dy = 0;
				}
				scroller->priv->going_right = scroller->priv->start_motion_x < x;
				scroller->priv->going_down = scroller->priv->start_motion_y < y;
				scroller->priv->start_motion_x = scroller->priv->farest_motion_x = x;
				scroller->priv->start_motion_y = scroller->priv->farest_motion_y = y;
				scroller->priv->start_motion_x_time = scroller->priv->start_motion_y_time = scroller->priv->farest_motion_x_time = scroller->priv->farest_motion_y_time = event_time;
				
			} else {
				if ((scroller->priv->going_right && x > scroller->priv->farest_motion_x) ||
				    (!scroller->priv->going_right && x < scroller->priv->farest_motion_x)) {
					scroller->priv->farest_motion_x = x;
					scroller->priv->farest_motion_x_time = event_time;
				} else if (ABS_F(x - scroller->priv->farest_motion_x) > clutter_backend_get_double_click_distance(backend)) {				
					scroller->priv->start_motion_x = scroller->priv->farest_motion_x;
					scroller->priv->start_motion_x_time = scroller->priv->farest_motion_x_time;
					scroller->priv->farest_motion_x = x;
					scroller->priv->farest_motion_x_time = event_time;
					scroller->priv->going_right = scroller->priv->start_motion_x < x;
				}
				if ((scroller->priv->going_down && y > scroller->priv->farest_motion_y) ||
				    (!scroller->priv->going_down && y < scroller->priv->farest_motion_y)) {
					scroller->priv->farest_motion_y = y;
					scroller->priv->farest_motion_y_time = event_time;
				} else if (ABS_F(y - scroller->priv->farest_motion_y) > clutter_backend_get_double_click_distance(backend)) {				
					scroller->priv->start_motion_y = scroller->priv->farest_motion_y;
					scroller->priv->start_motion_y_time = scroller->priv->farest_motion_y_time;
					scroller->priv->farest_motion_y = y;
					scroller->priv->farest_motion_y_time = event_time;
					scroller->priv->going_down = scroller->priv->start_motion_y < y;
				}
			}

			tangle_scroller_set_scrolling_offset_x(scroller, scroller->priv->start_scrolling_offset_x - dx);
			tangle_scroller_set_scrolling_offset_y(scroller, scroller->priv->start_scrolling_offset_y - dy);

			if (event->type == CLUTTER_BUTTON_RELEASE && scroller->priv->kinetic_scrolling &&
			    event_time - scroller->priv->previous_motion_time < clutter_backend_get_double_click_time(backend)) {
				if (event_time - scroller->priv->start_motion_x_time > 0) {
					vx = (gdouble)(x - scroller->priv->start_motion_x) / (event_time - scroller->priv->start_motion_x_time);

					clamp = tangle_clamp_new(TANGLE_CLAMP_HORIZONTAL, normalise_scrolling_offset_x(scroller, scroller->priv->scrolling_offset_x + (scroller->priv->going_right ? -1 : 1) * vx * vx * 1000 / scroller->priv->scrolling_deceleration / 2, &delta));
					g_signal_emit(scroller, signals[CLAMP_SCROLLING_OFFSET_X], 0, clamp);
					if (tangle_clamp_get_clamped_value_set(clamp)) {
						ax = vx * vx * 1000 / ABS_F(tangle_clamp_get_clamped_value(clamp) - scroller->priv->scrolling_offset_x + delta) / 2;
					} else {
						ax = scroller->priv->scrolling_deceleration;

					}
					g_object_unref(clamp);

					tx = ABS_F(vx * 1000.0 / ax);

					if (tx > 0) {
						dx = (scroller->priv->going_right ? -1 : 1) * vx * vx * 1000 / ax / 2;
						if (scroller->priv->continuous_x) {
							scroller->priv->animation_x = tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_OUT_QUAD, tx,
														  "scrolling-offset-x", scroller->priv->scrolling_offset_x + dx,
														  NULL);
							g_object_add_weak_pointer(G_OBJECT(scroller->priv->animation_x), (gpointer*)&scroller->priv->animation_x);
							g_signal_connect_swapped(scroller->priv->animation_x, "completed", G_CALLBACK(tangle_scroller_clamp_scrolling_offset_x), scroller);
						} else {
							if (scroller->priv->scrolling_offset_x + dx < -scroller->priv->max_overshoot_x) {
								tx *= -(scroller->priv->scrolling_offset_x + scroller->priv->max_overshoot_x) / dx;
								dx = -(scroller->priv->scrolling_offset_x + scroller->priv->max_overshoot_x);
							} else if (scroller->priv->scrolling_offset_x + dx > scroller->priv->scrolling_width + scroller->priv->max_overshoot_x) {
								tx *= (scroller->priv->scrolling_width - scroller->priv->scrolling_offset_x + scroller->priv->max_overshoot_x) / dx;
								dx = (scroller->priv->scrolling_width - scroller->priv->scrolling_offset_x + scroller->priv->max_overshoot_x);					
							}

							if (tx > 0) {
								scroller->priv->animation_x = tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_OUT_QUAD, tx,
															  "scrolling-offset-x", scroller->priv->scrolling_offset_x + dx,
															  NULL);
								g_object_add_weak_pointer(G_OBJECT(scroller->priv->animation_x), (gpointer*)&scroller->priv->animation_x);
								g_signal_connect_swapped(scroller->priv->animation_x, "completed", G_CALLBACK(tangle_scroller_clamp_scrolling_offset_x), scroller);
								if (scroller->priv->scrolling_offset_x + dx < 0.0) {
									handle_overshooting_animation(scroller, scroller->priv->animation_x, "scrolling-offset-x", 0.0);
								} else if (scroller->priv->scrolling_offset_x + dx > scroller->priv->scrolling_width) {
									handle_overshooting_animation(scroller, scroller->priv->animation_x, "scrolling-offset-x", scroller->priv->scrolling_width);
								}					
							} else {
								tangle_scroller_clamp_scrolling_offset_x(scroller);
							}
						}
					} else {
						tangle_scroller_clamp_scrolling_offset_x(scroller);
					}
				}

				if (event_time - scroller->priv->start_motion_y_time > 0) {
					vy = (gdouble)(y - scroller->priv->start_motion_y) / (event_time - scroller->priv->start_motion_y_time);

					clamp = tangle_clamp_new(TANGLE_CLAMP_VERTICAL, normalise_scrolling_offset_y(scroller, scroller->priv->scrolling_offset_y + (scroller->priv->going_down ? -1 : 1) * vy * vy * 1000 / scroller->priv->scrolling_deceleration / 2, &delta));
					g_signal_emit(scroller, signals[CLAMP_SCROLLING_OFFSET_Y], 0, clamp);
					if (tangle_clamp_get_clamped_value_set(clamp)) {
						ay = vy * vy * 1000 / ABS_F(tangle_clamp_get_clamped_value(clamp) - scroller->priv->scrolling_offset_y + delta) / 2;
					} else {
						ay = scroller->priv->scrolling_deceleration;
					}
					g_object_unref(clamp);

					ty = ABS_F(vy * 1000.0 / ay);

					if (ty > 0) {
						dy = (scroller->priv->going_down ? -1 : 1) * vy * vy * 1000 / ay / 2;
						if (scroller->priv->continuous_y) {					
							scroller->priv->animation_y = tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_OUT_QUAD, ty,
														  "scrolling-offset-y", scroller->priv->scrolling_offset_y + dy,
														  NULL);
							g_object_add_weak_pointer(G_OBJECT(scroller->priv->animation_y), (gpointer*)&scroller->priv->animation_y);
							g_signal_connect_swapped(scroller->priv->animation_y, "completed", G_CALLBACK(tangle_scroller_clamp_scrolling_offset_y), scroller);
						} else {
							if (scroller->priv->scrolling_offset_y + dy < -scroller->priv->max_overshoot_y) {
								ty *= -(scroller->priv->scrolling_offset_y + scroller->priv->max_overshoot_y) / dy;
								dy = -(scroller->priv->scrolling_offset_y + scroller->priv->max_overshoot_y);
							} else if (scroller->priv->scrolling_offset_y + dy > scroller->priv->scrolling_height + scroller->priv->max_overshoot_y) {
								ty *= (scroller->priv->scrolling_height - scroller->priv->scrolling_offset_y + scroller->priv->max_overshoot_y) / dy;
								dy = (scroller->priv->scrolling_height - scroller->priv->scrolling_offset_y + scroller->priv->max_overshoot_y);					
							}

							if (ty > 0) {
								scroller->priv->animation_y = tangle_actor_animate(TANGLE_ACTOR(scroller), CLUTTER_EASE_OUT_QUAD, ty,
															  "scrolling-offset-y", scroller->priv->scrolling_offset_y + dy,
															  NULL);
								g_object_add_weak_pointer(G_OBJECT(scroller->priv->animation_y), (gpointer*)&scroller->priv->animation_y);
								g_signal_connect_swapped(scroller->priv->animation_y, "completed", G_CALLBACK(tangle_scroller_clamp_scrolling_offset_y), scroller);
								if (scroller->priv->scrolling_offset_y + dy < 0.0) {
									handle_overshooting_animation(scroller, scroller->priv->animation_y, "scrolling-offset-y", 0.0);
								} else if (scroller->priv->scrolling_offset_y + dy > scroller->priv->scrolling_height) {
									handle_overshooting_animation(scroller, scroller->priv->animation_y, "scrolling-offset-y", scroller->priv->scrolling_height);
								}					
							} else {
								tangle_scroller_clamp_scrolling_offset_y(scroller);
							}
						}					
					} else {
						tangle_scroller_clamp_scrolling_offset_y(scroller);
					}
				}
			} else if (event->type == CLUTTER_BUTTON_RELEASE) {
				tangle_scroller_clamp_offsets(scroller);
			}

			scroller->priv->previous_motion_time = event_time;
		}
		
		if (event->type == CLUTTER_BUTTON_RELEASE) {
			g_signal_handler_disconnect(actor, scroller->priv->captured_event_handler_id);
			scroller->priv->captured_event_handler_id = 0;
			send_interaction_event(TANGLE_ACTOR(scroller), FALSE);
		}

		g_object_unref(scroller);
	}

	return FALSE;
}

static void send_interaction_event(TangleActor* actor, gboolean began) {
}

static gfloat get_width(TangleScroller* scroller) {
	ClutterActorBox actor_box;
	gfloat width;
	
	clutter_actor_get_allocation_box(CLUTTER_ACTOR(scroller), &actor_box);
	if ((width = actor_box.x2 - actor_box.x1) == 0.0) {
		width = G_MINFLOAT;
	}
	
	return width;
}

static gfloat get_height(TangleScroller* scroller) {
	ClutterActorBox actor_box;
	gfloat height;
	
	clutter_actor_get_allocation_box(CLUTTER_ACTOR(scroller), &actor_box);
	if ((height = actor_box.y2 - actor_box.y1) == 0.0) {
		height = G_MINFLOAT;
	}
	
	return height;
}

static gfloat normalise_scrolling_offset_x(TangleScroller* scroller, gfloat scrolling_offset_x, gfloat* delta_return) {
	gfloat width;
	
	if (scroller->priv->continuous_x) {
		width = scroller->priv->scrolling_width + get_width(scroller);
		if (scrolling_offset_x < 0.0) {
			scrolling_offset_x -= (((gint)(scrolling_offset_x / width)) - 1) * width;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_x / width)) - 1) * width;
			}
		} else if (scrolling_offset_x > scroller->priv->scrolling_width + width) {
			scrolling_offset_x -= (((gint)(scrolling_offset_x / width))) * width;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_x / width))) * width;
			}
		}
	} else {
		if (scrolling_offset_x < -scroller->priv->max_overshoot_x) {
			if (delta_return) {
				*delta_return = -scroller->priv->max_overshoot_x - scrolling_offset_x;
			}
			scrolling_offset_x = -scroller->priv->max_overshoot_x;
		} else if (scrolling_offset_x > scroller->priv->scrolling_width + scroller->priv->max_overshoot_x) {
			if (delta_return) {
				*delta_return = scroller->priv->scrolling_width + scroller->priv->max_overshoot_x - scrolling_offset_x;
			}
			scrolling_offset_x = scroller->priv->scrolling_width + scroller->priv->max_overshoot_x;
		}
	}
	
printf("%f / %f\n", scrolling_offset_x, scroller->priv->scrolling_width);
	
	return scrolling_offset_x;
}

static gfloat normalise_scrolling_offset_y(TangleScroller* scroller, gfloat scrolling_offset_y, gfloat* delta_return) {
	gfloat height;
	gfloat delta = 0.0;
	
	if (scroller->priv->continuous_y) {
			height = scroller->priv->scrolling_height + get_height(scroller);
		if (scrolling_offset_y < 0.0) {
			scrolling_offset_y -= (((gint)(scrolling_offset_y / height)) - 1) * height;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_y / height)) - 1) * height;
			}
		} else if (scrolling_offset_y > scroller->priv->scrolling_height + height) {
			scrolling_offset_y -= (((gint)(scrolling_offset_y / height))) * height;
			if (delta_return) {
				*delta_return = (((gint)(scrolling_offset_y / height))) * height;
			}
		}
	} else {
		if (scrolling_offset_y < -scroller->priv->max_overshoot_y) {
			if (delta_return) {
				*delta_return = -scroller->priv->max_overshoot_y - scrolling_offset_y;
			}
			scrolling_offset_y = -scroller->priv->max_overshoot_y;
		} else if (scrolling_offset_y > scroller->priv->scrolling_height + scroller->priv->max_overshoot_y) {
			if (delta_return) {
				*delta_return = scroller->priv->scrolling_height + scroller->priv->max_overshoot_y - scrolling_offset_y;
			}
			scrolling_offset_y = scroller->priv->scrolling_height + scroller->priv->max_overshoot_y;
		}
	}
	
	return scrolling_offset_y;
}
