/*
 * tangle-button.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-button.h"
#include "tangle-misc.h"
#include "marshalers.h"

/**
 * SECTION:tangle-button
 * @Short_description: 	A widget with clicked functionality
 * @Title: TangleButton
 */

G_DEFINE_TYPE(TangleButton, tangle_button, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_BUTTON_MODE,
	PROP_NORMAL_BACKGROUND_ACTOR,
	PROP_INTERACTIVE_BACKGROUND_ACTOR,
	PROP_SELECTED_BACKGROUND_ACTOR,
	PROP_SELECTED,
	PROP_CLICKED_ACTION
};

enum {
	CLICKED,
	LAST_SIGNAL
};

struct _TangleButtonPrivate {
	TangleButtonMode mode;
	ClutterActor* normal_background_actor;
	ClutterActor* interactive_background_actor;
	ClutterActor* selected_background_actor;
	TangleAction* clicked_action;

	ClutterActor* button_release_event_handler_actor;

	guint active : 1;
	guint inside : 1;
	guint selected : 1;
};

static const GEnumValue button_mode_values[] = {
	{ TANGLE_BUTTON_CLICKABLE, "TANGLE_BUTTON_CLICKABLE", "clickable" },
	{ TANGLE_BUTTON_SELECTABLE, "TANGLE_BUTTON_SELECTABLE", "selectable" },
	{ TANGLE_BUTTON_RADIO, "TANGLE_BUTTON_RADIO", "radio" },
	{ 0, NULL, NULL }
};

static gboolean on_button_release_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static void begin_interacting(TangleButton* button);
static void end_interacting(TangleButton* button);

static guint signals[LAST_SIGNAL];

GType tangle_button_mode_get_type(void) {
	static GType type = 0;
	
	if (!type) {
		type = g_enum_register_static("TangleButtonMode", button_mode_values);
	}
	
	return type;
}

ClutterActor* tangle_button_new(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, NULL));
}

ClutterActor* tangle_button_new_with_background_actor(ClutterActor* normal_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "normal-background-actor", normal_background_actor, NULL));
}

ClutterActor* tangle_button_new_with_background_actors(ClutterActor* normal_background_actor, ClutterActor* interactive_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "normal-background-actor", normal_background_actor, "interactive-background-actor", interactive_background_actor, NULL));
}

ClutterActor* tangle_button_new_selectable(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "button-mode", TANGLE_BUTTON_SELECTABLE, NULL));
}

ClutterActor* tangle_button_new_selectable_with_background_actor(ClutterActor* normal_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "button-mode", TANGLE_BUTTON_SELECTABLE, "normal-background-actor", normal_background_actor, "prefer-background-size", TRUE, NULL));
}

ClutterActor* tangle_button_new_selectable_with_background_actors(ClutterActor* normal_background_actor, ClutterActor* interactive_background_actor, ClutterActor* selected_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "button-mode", TANGLE_BUTTON_SELECTABLE, "normal-background-actor", normal_background_actor, "interactive-background-actor", interactive_background_actor, "selected-background-actor", selected_background_actor, NULL));
}

ClutterActor* tangle_button_new_radio(void) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "button-mode", TANGLE_BUTTON_RADIO, NULL));
}

ClutterActor* tangle_button_new_radio_with_background_actor(ClutterActor* normal_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "button-mode", TANGLE_BUTTON_RADIO, "normal-background-actor", normal_background_actor, "prefer-background-size", TRUE, NULL));
}

ClutterActor* tangle_button_new_radio_with_background_actors(ClutterActor* normal_background_actor, ClutterActor* interactive_background_actor, ClutterActor* selected_background_actor) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_BUTTON, "button-mode", TANGLE_BUTTON_RADIO, "normal-background-actor", normal_background_actor, "interactive-background-actor", interactive_background_actor, "selected-background-actor", selected_background_actor, NULL));
}

TangleButtonMode tangle_button_get_mode(TangleButton* button) {

	return button->priv->mode;
}

void tangle_button_set_mode(TangleButton* button, TangleButtonMode mode) {
	g_return_if_fail(mode >= TANGLE_BUTTON_CLICKABLE && mode <= TANGLE_BUTTON_RADIO);
	
	if (button->priv->mode != mode) {
		button->priv->mode = mode;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(button));
		g_object_notify(G_OBJECT(button), "button-mode");
	}
}

ClutterActor* tangle_button_get_normal_background_actor(TangleButton* button) {
	g_return_val_if_fail(TANGLE_IS_BUTTON(button), NULL);

	return button->priv->normal_background_actor;
}

void tangle_button_set_normal_background_actor(TangleButton* button, ClutterActor* actor) {
	g_return_if_fail(TANGLE_IS_BUTTON(button));
        
	if (button->priv->normal_background_actor != actor) {
		if (button->priv->normal_background_actor) {
			clutter_actor_unparent(button->priv->normal_background_actor);
		}
		button->priv->normal_background_actor = actor;
		if (button->priv->normal_background_actor) {
			clutter_actor_push_internal(CLUTTER_ACTOR(button));
			clutter_actor_set_parent(button->priv->normal_background_actor, CLUTTER_ACTOR(button));	
			clutter_actor_pop_internal(CLUTTER_ACTOR(button));
		}

		if (!button->priv->active || !button->priv->interactive_background_actor) {
			tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
		}
	}
}

ClutterActor* tangle_button_get_interactive_background_actor(TangleButton* button) {
	g_return_val_if_fail(TANGLE_IS_BUTTON(button), NULL);

	return button->priv->interactive_background_actor;
}

void tangle_button_set_interactive_background_actor(TangleButton* button, ClutterActor* actor) {
	gboolean had_interactive_background_actor = FALSE;
	
	g_return_if_fail(TANGLE_IS_BUTTON(button));

	if (button->priv->interactive_background_actor != actor) {
		if (button->priv->interactive_background_actor) {
			clutter_actor_unparent(button->priv->interactive_background_actor);
			had_interactive_background_actor = TRUE;
		}
		button->priv->interactive_background_actor = actor;
		if (button->priv->interactive_background_actor) {
			clutter_actor_push_internal(CLUTTER_ACTOR(button));
			clutter_actor_set_parent(button->priv->interactive_background_actor, CLUTTER_ACTOR(button));	
			clutter_actor_pop_internal(CLUTTER_ACTOR(button));
		}

		if (button->priv->active) {
			if (button->priv->interactive_background_actor) {
				tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->interactive_background_actor);
			} else if (had_interactive_background_actor) {
				tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
			}
		}
	}
}

ClutterActor* tangle_button_get_selected_background_actor(TangleButton* button) {
	g_return_val_if_fail(TANGLE_IS_BUTTON(button), NULL);

	return button->priv->selected_background_actor;
}

void tangle_button_set_selected_background_actor(TangleButton* button, ClutterActor* actor) {
	gboolean had_selected_background_actor = FALSE;
	
	g_return_if_fail(TANGLE_IS_BUTTON(button));

	if (button->priv->selected_background_actor != actor) {
		if (button->priv->selected_background_actor) {
			clutter_actor_unparent(button->priv->selected_background_actor);
			had_selected_background_actor = TRUE;
		}
		button->priv->selected_background_actor = actor;
		if (button->priv->selected_background_actor) {
			clutter_actor_push_internal(CLUTTER_ACTOR(button));
			clutter_actor_set_parent(button->priv->selected_background_actor, CLUTTER_ACTOR(button));	
			clutter_actor_pop_internal(CLUTTER_ACTOR(button));
		}

		if (button->priv->selected) {
			if (button->priv->selected_background_actor) {
				tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->selected_background_actor);
			} else if (had_selected_background_actor) {
				tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
			}
		}
	}
}

gboolean tangle_button_get_selected(TangleButton* button) {
	gboolean selected = FALSE;
	
	if ((button->priv->mode == TANGLE_BUTTON_SELECTABLE || button->priv->mode == TANGLE_BUTTON_RADIO)) {
		selected = button->priv->selected;
	}
	
	return selected;
}
	
void tangle_button_set_selected(TangleButton* button, gboolean selected) {
	g_return_if_fail((button->priv->mode == TANGLE_BUTTON_SELECTABLE || button->priv->mode == TANGLE_BUTTON_RADIO));
	
	if (button->priv->selected != selected) {
		button->priv->selected = selected;
		if (!button->priv->active) {
			if ((button->priv->mode == TANGLE_BUTTON_SELECTABLE || button->priv->mode == TANGLE_BUTTON_RADIO) &&
			    button->priv->selected && button->priv->selected_background_actor) {
				tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->selected_background_actor);
			} else {
				tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
			}
		}
		g_object_notify(G_OBJECT(button), "selected");
	}
}

static gboolean tangle_button_clicked(TangleButton* button) {
	TangleProperties* properties;
	
	if (button->priv->clicked_action) {
		properties = tangle_properties_new();
		tangle_action_execute_full(button->priv->clicked_action, G_OBJECT(button), "clicked", properties);
		g_object_unref(G_OBJECT(properties));
	}
	
	return FALSE;
}

static void tangle_button_map(ClutterActor* actor) {
	TangleButton* button;

	button = TANGLE_BUTTON(actor);

	CLUTTER_ACTOR_CLASS(tangle_button_parent_class)->map(actor);

	if (button->priv->normal_background_actor) {
		clutter_actor_map(button->priv->normal_background_actor);
	}
	if (button->priv->interactive_background_actor) {
		clutter_actor_map(button->priv->interactive_background_actor);
	}
	if (button->priv->selected_background_actor) {
		clutter_actor_map(button->priv->selected_background_actor);
	}
}

static void tangle_button_unmap(ClutterActor* actor) {
	TangleButton* button;

	button = TANGLE_BUTTON(actor);

	if (button->priv->normal_background_actor) {
		clutter_actor_unmap(button->priv->normal_background_actor);
	}
	if (button->priv->interactive_background_actor) {
		clutter_actor_unmap(button->priv->interactive_background_actor);
	}
	if (button->priv->selected_background_actor) {
		clutter_actor_unmap(button->priv->selected_background_actor);
	}

	CLUTTER_ACTOR_CLASS(tangle_button_parent_class)->unmap(actor);
}

static void tangle_button_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleButton* button;
	
	button = TANGLE_BUTTON(object);

	switch (prop_id) {
		case PROP_BUTTON_MODE:
			tangle_button_set_mode(button, g_value_get_enum(value));
			break;
		case PROP_NORMAL_BACKGROUND_ACTOR:
			tangle_button_set_normal_background_actor(button, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_INTERACTIVE_BACKGROUND_ACTOR:
			tangle_button_set_interactive_background_actor(button, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_SELECTED_BACKGROUND_ACTOR:
			tangle_button_set_selected_background_actor(button, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_SELECTED:
			tangle_button_set_selected(button, g_value_get_boolean(value));
			break;
		case PROP_CLICKED_ACTION:
			if (button->priv->clicked_action) {
				g_object_unref(button->priv->clicked_action);
			}
			button->priv->clicked_action = TANGLE_ACTION(g_value_get_object(value));
			g_object_ref(button->priv->clicked_action);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_button_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleButton* button;

	button = TANGLE_BUTTON(object);

        switch (prop_id) {
		case PROP_BUTTON_MODE:
			g_value_set_enum(value, button->priv->mode);
			break;
		case PROP_NORMAL_BACKGROUND_ACTOR:
			g_value_set_object(value, button->priv->normal_background_actor);
			break;
		case PROP_INTERACTIVE_BACKGROUND_ACTOR:
			g_value_set_object(value, button->priv->interactive_background_actor);
			break;
		case PROP_SELECTED_BACKGROUND_ACTOR:
			g_value_set_object(value, button->priv->selected_background_actor);
			break;
		case PROP_SELECTED:
			g_value_set_boolean(value, button->priv->selected);
			break;
		case PROP_CLICKED_ACTION:
			g_value_set_object(value, button->priv->clicked_action);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_button_dispose(GObject* object) {
	TangleButton* button;
	
	button = TANGLE_BUTTON(object);
	
	TANGLE_UNPARENT_AND_NULLIFY_ACTOR(button->priv->normal_background_actor);
	TANGLE_UNPARENT_AND_NULLIFY_ACTOR(button->priv->interactive_background_actor);
	TANGLE_UNPARENT_AND_NULLIFY_ACTOR(button->priv->selected_background_actor);
	TANGLE_UNREF_AND_NULLIFY_OBJECT(button->priv->clicked_action);
	
	if (button->priv->button_release_event_handler_actor) {
		g_signal_handlers_disconnect_by_func(button->priv->button_release_event_handler_actor, G_CALLBACK(on_button_release_event), button);
		button->priv->button_release_event_handler_actor = NULL;
	}

	G_OBJECT_CLASS(tangle_button_parent_class)->dispose(object);
}

static void tangle_button_class_init(TangleButtonClass* button_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS (button_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(button_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(button_class);

	gobject_class->dispose = tangle_button_dispose;
	gobject_class->set_property = tangle_button_set_property;
	gobject_class->get_property = tangle_button_get_property;

	clutter_actor_class->map = tangle_button_map;
	clutter_actor_class->unmap = tangle_button_unmap;

	button_class->clicked = tangle_button_clicked;

	/**
	 * TangleButton:selected:
	 */
	g_object_class_install_property(gobject_class, PROP_BUTTON_MODE,
	                                g_param_spec_enum("button-mode",
	                                "Button mode",
	                                "The mode of the button",
	                                TANGLE_TYPE_BUTTON_MODE,
					TANGLE_BUTTON_CLICKABLE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleClutterActor:normal-background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_NORMAL_BACKGROUND_ACTOR,
	                                g_param_spec_object("normal-background-actor",
	                                                    "Normal background actor",
	                                                    "The background actor used in normal (noninteractive) state",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * TangleClutterActor:interactive-background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_INTERACTIVE_BACKGROUND_ACTOR,
	                                g_param_spec_object("interactive-background-actor",
	                                                    "Interactive background actor",
	                                                    "The background actor used in interactive (interacting) state",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleClutterActor:selected-background-actor:
	 */
	g_object_class_install_property(gobject_class, PROP_SELECTED_BACKGROUND_ACTOR,
	                                g_param_spec_object("selected-background-actor",
	                                                    "Selected background actor",
	                                                    "The background actor used in selected state",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleButton:selected:
	 */
	g_object_class_install_property(gobject_class, PROP_SELECTED,
	                                g_param_spec_boolean("selected",
	                                "Selected",
	                                "Whether the button is selected or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleButton:on-clicked-action:
	 */
	g_object_class_install_property(gobject_class, PROP_CLICKED_ACTION,
	                                g_param_spec_object("clicked-action",
	                                "Clicked action",
	                                "Action that is executed on clicked signal",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * TangleButton:clicked:
	 * @actor: the object which received the signal
	 */
	signals[CLICKED] = g_signal_new("clicked", G_TYPE_FROM_CLASS(gobject_class),
	                                G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleButtonClass, clicked),
					g_signal_accumulator_true_handled, NULL,
					tangle_marshal_BOOLEAN__VOID,
					G_TYPE_BOOLEAN, 0);

	g_type_class_add_private(gobject_class, sizeof(TangleButtonPrivate));
}

static void start_interacting(TangleButton* button) {
	tangle_actor_set_interacting(TANGLE_ACTOR(button), TRUE);
	if (button->priv->interactive_background_actor) {
		tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->interactive_background_actor);
	}
	
	button->priv->button_release_event_handler_actor = clutter_actor_get_stage(CLUTTER_ACTOR(button));
	g_object_add_weak_pointer(G_OBJECT(button->priv->button_release_event_handler_actor), (gpointer*)&button->priv->button_release_event_handler_actor);
	g_signal_connect(button->priv->button_release_event_handler_actor, "button-release-event", G_CALLBACK(on_button_release_event), button);

	tangle_actor_set_interacting(TANGLE_ACTOR(button), TRUE);
}

static void end_interacting(TangleButton* button) {
	tangle_actor_set_interacting(TANGLE_ACTOR(button), FALSE);
	if ((button->priv->mode == TANGLE_BUTTON_SELECTABLE || button->priv->mode == TANGLE_BUTTON_RADIO) && button->priv->selected && button->priv->selected_background_actor) {
		tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->selected_background_actor);
	} else {
		tangle_widget_change_background_actor(TANGLE_WIDGET(button), button->priv->normal_background_actor);
	}

	tangle_actor_set_interacting(TANGLE_ACTOR(button), FALSE);
}

static gboolean on_button_release_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleButton* button;
	gboolean handled;

	button = TANGLE_BUTTON(user_data);

	if (button->priv->inside) {
		if (button->priv->mode == TANGLE_BUTTON_SELECTABLE) {
			button->priv->selected = !button->priv->selected;
			g_object_notify(G_OBJECT(button), "selected");
		} else if (button->priv->mode == TANGLE_BUTTON_RADIO && !button->priv->selected) {
			button->priv->selected = TRUE;
			g_object_notify(G_OBJECT(button), "selected");
		}
	}

	end_interacting(button);
	button->priv->active = FALSE;;

	if (button->priv->inside) {
		g_signal_emit(button, signals[CLICKED], 0, &handled);
	}

	if (button->priv->button_release_event_handler_actor) {
		g_signal_handlers_disconnect_by_func(button->priv->button_release_event_handler_actor, G_CALLBACK(on_button_release_event), button);
		button->priv->button_release_event_handler_actor = NULL;
	}

	return FALSE;
}

static gboolean on_button_press_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	start_interacting(TANGLE_BUTTON(actor));
	TANGLE_BUTTON(actor)->priv->active = TRUE;
	TANGLE_BUTTON(actor)->priv->inside = TRUE;
	
	return FALSE;
}

static gboolean on_enter_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	if (TANGLE_BUTTON(actor)->priv->active) {
		start_interacting(TANGLE_BUTTON(actor));
	}
	TANGLE_BUTTON(actor)->priv->inside = TRUE;
	
	return FALSE;
}

static gboolean on_leave_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	if (TANGLE_BUTTON(actor)->priv->active) {
		end_interacting(TANGLE_BUTTON(actor));
	}
	TANGLE_BUTTON(actor)->priv->inside = FALSE;
	
	return FALSE;

}

static void on_notify_ancestor_interacting(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleButton* button;
	
	button = TANGLE_BUTTON(object);

	if (tangle_actor_get_ancestor_interacting(TANGLE_ACTOR(button))) {
		end_interacting(button);
		button->priv->active = FALSE;
	}
}

static void tangle_button_init(TangleButton* button) {
	button->priv = G_TYPE_INSTANCE_GET_PRIVATE(button, TANGLE_TYPE_BUTTON, TangleButtonPrivate);
	clutter_actor_set_reactive(CLUTTER_ACTOR(button), TRUE);
	g_signal_connect(button, "button-press-event", G_CALLBACK(on_button_press_event), NULL);
	g_signal_connect(button, "enter-event", G_CALLBACK(on_enter_event), NULL);
	g_signal_connect(button, "leave-event", G_CALLBACK(on_leave_event), NULL);
	g_signal_connect(button, "notify::ancestor-interacting", G_CALLBACK(on_notify_ancestor_interacting), NULL);
}
