
/*
 * tangle-actor.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#ifndef __TANGLE_ACTOR_H__
#define __TANGLE_ACTOR_H__

#include <clutter/clutter.h>

#define TANGLE_TYPE_ACTOR (tangle_actor_get_type())
#define TANGLE_ACTOR(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj), TANGLE_TYPE_ACTOR, TangleActor))
#define TANGLE_IS_ACTOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE((obj), TANGLE_TYPE_ACTOR))
#define TANGLE_ACTOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST((klass), TANGLE_TYPE_ACTOR, TangleActorClass))
#define TANGLE_IS_ACTOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), TANGLE_TYPE_ACTOR))
#define TANGLE_ACTOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS((obj), TANGLE_TYPE_ACTOR, TangleActorClass))

/**
 * TangleActorFlags:
 *
 * Addition information on the state of an actor.
 */
typedef enum {
	TANGLE_ACTOR_ON_SCREEN = 1 << 0,
	TANGLE_ACTOR_ALLOCATED = 1 << 1,
	TANGLE_ACTOR_HIDING = 1 << 2,
	TANGLE_ACTOR_SHOWING = 1 << 3
} TangleActorFlags;

typedef struct _TangleActor TangleActor;
typedef struct _TangleActorPrivate TangleActorPrivate;

#include "tangle-transition.h"

/**
 * TangleActor:
 * @flags: #TangleActorFlags
 *
 * Base class for actors.
 */
struct _TangleActor {
	ClutterActor parent_instance;

	TangleActorPrivate* priv;
	TangleActorFlags flags;
};

typedef struct _TangleActorClass {
	ClutterActorClass parent_class;
	
	ClutterTimeline* (*animate_transition)(TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box);
	void (*transition_completed)(TangleActor* actor, ClutterActorBox* old_box, ClutterActorBox* current_box);
	void (*show_completed)(TangleActor* actor, ClutterActorBox* current_box);
	void (*get_preferred_width)(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p);
	void (*get_preferred_height)(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p);
	void (*paint_aligned)(TangleActor* actor, gfloat width, gfloat height);
} TangleActorClass;

GType tangle_actor_get_type (void) G_GNUC_CONST;

#define TANGLE_ACTOR_SET_FLAGS(a,f) (((TangleActor*)(a))->flags |= (f))
#define TANGLE_ACTOR_UNSET_FLAGS(a,f)  (((TangleActor*)(a))->flags &= ~(f))
#define TANGLE_ACTOR_IS_ON_SCREEN(a) ((((TangleActor*)(a))->flags & TANGLE_ACTOR_ON_SCREEN) != FALSE)
#define TANGLE_ACTOR_IS_MAPPED_ON_SCREEN(a) (CLUTTER_ACTOR_IS_MAPPED(a) && ((((TangleActor*)(a))->flags & TANGLE_ACTOR_ON_SCREEN) != FALSE))
#define TANGLE_ACTOR_IS_ALLOCATED(a) ((((TangleActor*)(a))->flags & TANGLE_ACTOR_ALLOCATED) != FALSE)
#define TANGLE_ACTOR_IS_HIDING(a) ((((TangleActor*)(a))->flags & TANGLE_ACTOR_HIDING) != FALSE)
#define TANGLE_ACTOR_IS_SHOWING(a) ((((TangleActor*)(a))->flags & TANGLE_ACTOR_SHOWING) != FALSE)

ClutterAnimation* tangle_actor_animate(TangleActor* actor, gulong mode, guint duration, ...);
ClutterAnimation* tangle_actor_animate_with_timeline(TangleActor* actor, gulong mode, ClutterTimeline* timeline, ...);
ClutterAnimation* tangle_actor_animatev(TangleActor* actor, gulong mode, guint duration, gint n_properties, const gchar* const property_names[], const GValue* property_values);
gboolean tangle_actor_stop_animation(TangleActor* actor, ClutterAnimation* animation);

TangleTransition* tangle_actor_get_transition(TangleActor* actor);
void tangle_actor_set_transition(TangleActor* actor, TangleTransition* transition);
gboolean tangle_actor_should_transition_scale_children(TangleActor* actor);
void tangle_actor_apply_transition_scale(TangleActor* actor);
void tangle_actor_get_transition_move(TangleActor* actor, gfloat* x_return, gfloat* y_return);
void tangle_actor_set_transition_move(TangleActor* actor, gfloat x, gfloat y);
void tangle_actor_get_transition_scale(TangleActor* actor, gdouble* x_return, gdouble* y_return);
void tangle_actor_set_transition_scale(TangleActor* actor, gdouble x, gdouble y);

void tangle_actor_show(TangleActor* actor);
void tangle_actor_hide_animated(TangleActor* actor);
void tangle_actor_hide_all_animated(TangleActor* actor);
void tangle_actor_destroy_animated(TangleActor* actor);

gint tangle_actor_get_depth_position(TangleActor* actor);
void tangle_actor_set_depth_position(TangleActor* actor, gint depth_position);

typedef struct {
	gfloat left;
	gfloat top;
	gfloat right;
	gfloat bottom;
} TangleSpacing;

void tangle_actor_get_margin(TangleActor* actor, TangleSpacing* spacing);
void tangle_actor_set_margin(TangleActor* actor, const TangleSpacing* spacing);
gboolean tangle_actor_get_margin_set(TangleActor* actor);
void tangle_actor_set_margin_set(TangleActor* actor, gboolean is_set);

void tangle_actor_get_max_size(TangleActor* actor, gfloat* max_width_p, gfloat* max_height_p);
void tangle_actor_set_max_size(TangleActor* actor, gfloat max_width, gfloat max_height);

gdouble tangle_actor_get_aspect_ratio(TangleActor* actor);
void tangle_actor_set_aspect_ratio(TangleActor* actor, gdouble aspect_ratio);
gboolean tangle_actor_get_aspect_ratio_set(TangleActor* actor);
void tangle_actor_set_aspect_ratio_set(TangleActor* actor, gboolean is_set);

gdouble tangle_actor_get_alignment_x(TangleActor* actor);
void tangle_actor_set_alignment_x(TangleActor* actor, gdouble alignment_x);
gdouble tangle_actor_get_alignment_y(TangleActor* actor);
void tangle_actor_set_alignment_y(TangleActor* actor, gdouble alignment_y);

void tangle_actor_get_aligned_allocation(TangleActor* actor, ClutterActorBox* actor_box);

gboolean tangle_actor_get_interacting(TangleActor* actor);
void tangle_actor_set_interacting(TangleActor* actor, gboolean interacting);

gboolean tangle_actor_get_dragging(TangleActor* actor);
void tangle_actor_set_dragging(TangleActor* actor, gboolean dragging);

gboolean tangle_actor_get_ancestor_interacting(TangleActor* actor);
gboolean tangle_actor_get_ancestor_dragging(TangleActor* actor);

void tangle_actor_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p);
void tangle_actor_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p);
void tangle_actor_get_preferred_size(TangleActor* actor, gboolean interacting, gfloat* min_width_p, gfloat* min_height_p, gfloat* natural_width_p, gfloat* natural_height_p, gfloat* max_width_p, gfloat* max_height_p);

gdouble tangle_actor_get_interactive_size_multiplier(TangleActor* actor);
void tangle_actor_set_interactive_size_multiplier(TangleActor* actor, gdouble multiplier);
gboolean tangle_actor_get_interactive_size_multiplier_set(TangleActor* actor);
void tangle_actor_set_interactive_size_multiplier_set(TangleActor* actor, gboolean is_set);

void tangle_actor_get_layout_allocation(TangleActor* actor, ClutterActorBox* actor_box);
void tangle_actor_set_layout_allocation(TangleActor* actor, const ClutterActorBox* actor_box);

#define TANGLE_TYPE_SPACING (tangle_spacing_get_type())
#define TANGLE_SPACING(s) ((TangleSpacing*)(s))

GType tangle_spacing_get_type(void);

TangleSpacing* tangle_spacing_new(gfloat top, gfloat right, gfloat bottom, gfloat left);
TangleSpacing* tangle_spacing_copy(const TangleSpacing* spacing);
void tangle_spacing_free(TangleSpacing* spacing);
gboolean tangle_spacing_equal(const TangleSpacing* spacing_a, const TangleSpacing* spacing_b);

gboolean tangle_spacing_parse_from_json(TangleSpacing* spacing, JsonNode* node);

ClutterAction* tangle_actor_get_action_by_type(ClutterActor* actor, GType type);

#endif
