/*
 * tangle-object.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-object.h"
#include "tangle-private.h"

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface);

G_DEFINE_ABSTRACT_TYPE_WITH_CODE(TangleObject, tangle_object, G_TYPE_OBJECT,
                                 G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

enum {
	PROP_0,
	PROP_TEMPLATE
};

struct _TangleObjectPrivate {
	GHashTable* template_parameters;
	gboolean template_found;
};

static ClutterScriptableIface* parent_scriptable_iface = NULL;

static void tangle_object_set_property(GObject* gobject, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleObject* object;
	
	object = TANGLE_OBJECT(gobject);

	switch (prop_id) {
		case PROP_TEMPLATE:
			if (g_value_get_object(value) != NULL) {
				tangle_template_apply_properties(TANGLE_TEMPLATE(g_value_get_object(value)), gobject);
			}
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_object_get_property(GObject* gobject, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleObject* object;

	object = TANGLE_OBJECT(gobject);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
		        break;
        }
}

static void tangle_object_finalize(GObject* gobject) {
	G_OBJECT_CLASS(tangle_object_parent_class)->finalize(gobject);
}

static void tangle_object_dispose(GObject* gobject) {
	TangleObject* object;
	
	object = TANGLE_OBJECT(gobject);
	
	if (object->priv->template_parameters) {
		g_hash_table_destroy(object->priv->template_parameters);
		object->priv->template_parameters = NULL;
	}
	
	G_OBJECT_CLASS(tangle_object_parent_class)->dispose(gobject);
}

static void tangle_object_class_init(TangleObjectClass* object_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(object_class);

	gobject_class->finalize = tangle_object_finalize;
	gobject_class->dispose = tangle_object_dispose;
	gobject_class->set_property = tangle_object_set_property;
	gobject_class->get_property = tangle_object_get_property;

	/**
	 * TangleObject:template:
	 *
	 * Applies the given #TangleTemplate to this object.
	 */
	g_object_class_install_property(gobject_class, PROP_TEMPLATE,
	                                g_param_spec_object("template",
	                                "Template",
	                                "Applies the given template to this object",
	                                TANGLE_TYPE_TEMPLATE,
	                                G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof (TangleObjectPrivate));
}

static void tangle_object_init(TangleObject* object) {
	object->priv = G_TYPE_INSTANCE_GET_PRIVATE(object, TANGLE_TYPE_OBJECT, TangleObjectPrivate);
}

static gboolean tangle_object_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node) {
	gboolean retvalue;
	TangleObject* object;
	
	object = TANGLE_OBJECT(scriptable);

	if (!(retvalue = _tangle_scriptable_parse_custom_node(scriptable, script, value, name, node, &object->priv->template_parameters, &object->priv->template_found)) &&
	    parent_scriptable_iface->parse_custom_node) {
		retvalue = parent_scriptable_iface->parse_custom_node(scriptable, script, value, name, node);
	}
	
	return retvalue;
}

static void tangle_object_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	TangleObject* object;
	
	object = TANGLE_OBJECT(scriptable);

	_tangle_scriptable_set_custom_property(scriptable, script, name, value, &object->priv->template_parameters, object->priv->template_found);
}

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface) {
	if (!(parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}
	
	iface->parse_custom_node = tangle_object_parse_custom_node;
	iface->set_custom_property = tangle_object_set_custom_property;
}
