/*
 * tangle-private.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-private.h"

typedef struct _TemplateParameter {
	gchar* name;
	GValue value;
} TemplateParameter;

gboolean _tangle_scriptable_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node, GHashTable** template_parameters_pointer, gboolean* template_found_pointer) {
	gboolean retvalue = FALSE;
	gpointer type_class;
	GValue* value_copy;
	GParamSpec* param_spec;
	const gchar* s;
	GObject* object;
	gchar* id;
	
	type_class = g_type_class_ref(G_OBJECT_TYPE(scriptable));
	
	if (name[0] == '$') {
		if (!*template_parameters_pointer) {
			*template_parameters_pointer = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, (GDestroyNotify)g_value_unset);
		}
		if (g_hash_table_lookup(*template_parameters_pointer, name + 1)) {
			g_warning("The value of a template parameter '%s' is defined more than once for the object '%s', skipped.", name, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!clutter_script_parse_node(script, value, name, node, NULL)) {
			g_warning("The value of a template parameter '%s' could not be parsed for the object '%s', skipped.", name, clutter_get_script_id(G_OBJECT(scriptable)));
		} else {
			value_copy = g_new0(GValue, 1);
			g_value_init(value_copy, G_VALUE_TYPE(value));
			g_value_copy(value, value_copy);
			g_hash_table_insert(*template_parameters_pointer, g_strdup(name + 1), value_copy);

			retvalue = TRUE;
		}
	} else if (name[0] == '@') {
		if (!clutter_script_parse_node(script, value, name, node, NULL)) {
			g_warning("The value of a signal handler property '%s' could not be parsed for the object '%s', skipped.", name, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!G_VALUE_HOLDS_STRING(value)) {
			g_warning("The value of a signal handler property '%s' of athe object '%s' was not string, skipped.", name, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!tangle_signal_connect_dynamically_from_script(G_OBJECT(scriptable), name + 1, g_value_get_string(value), script)) {
			g_warning("Failed to connect signal handler property '%s' of the object '%s', skipped.", name, clutter_get_script_id(G_OBJECT(scriptable)));
		} else {
			retvalue = TRUE;
		}
	} else if (!strcmp(name, "template")) {
		*template_found_pointer = TRUE;
	} else if ((param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name)) &&
	           G_IS_PARAM_SPEC_OBJECT(param_spec) &&
		   !g_type_is_a(G_PARAM_SPEC_VALUE_TYPE(param_spec), TANGLE_TYPE_TEMPLATE) &&
	           (s = json_node_get_string(node)) &&
		   (object = clutter_script_get_object(script, s)) &&
		   TANGLE_IS_TEMPLATE(object)) {
		if (!g_type_is_a(tangle_template_get_object_type(TANGLE_TEMPLATE(object)), G_PARAM_SPEC_VALUE_TYPE(param_spec))) {
			g_warning("The type of the template '%s' does not match to the property '%s', skipped.", s, name);
		} else {
			id = g_strconcat(clutter_get_script_id(G_OBJECT(scriptable)), "$", name, NULL);
			if (!(object = tangle_template_create_object_with_id(TANGLE_TEMPLATE(object), id))) {
				g_warning("Could not instantiate the template '%s' for the property '%s', skipped.", s, name);
			} else {
				g_value_init(value, G_PARAM_SPEC_VALUE_TYPE(param_spec));
				g_value_set_object(value, object);

				retvalue = TRUE;

			}
			g_free(id);
		}
	}

	g_type_class_unref(type_class);

	return retvalue;
}

static void set_template_parameter(gpointer key, gpointer value, gpointer user_data) {
	TangleTemplate* template;
	
	template = TANGLE_TEMPLATE(user_data);
	
	tangle_template_set_parameter_value(template, (gchar*)key, (GValue*)value);
}

gboolean _tangle_scriptable_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value, GHashTable** template_parameters_pointer, gboolean template_found) {
	TangleTemplate* template;

	if (*template_parameters_pointer && !template_found) {
		g_warning("Template parameters found but no template for the object '%s', skipped.", clutter_get_script_id(G_OBJECT(scriptable)));

		g_hash_table_destroy(*template_parameters_pointer);
		*template_parameters_pointer = NULL;
	}

	if (!strcmp(name, "template")) {
		template = TANGLE_TEMPLATE(g_value_get_object(value));
		if (*template_parameters_pointer) {
			tangle_template_clear_parameter_values(template);
			g_hash_table_foreach(*template_parameters_pointer, set_template_parameter, template);
			if (!tangle_template_are_parameters_complete(template)) {
				g_warning("Missing template parameters for the template '%s' for the object '%s', skipped.", clutter_get_script_id(G_OBJECT(template)), clutter_get_script_id(G_OBJECT(scriptable)));
			} else {
				tangle_template_apply_properties(template, G_OBJECT(scriptable));
			}
			tangle_template_clear_parameter_values(template);
		} else if (tangle_template_has_parameters(template)) {
			g_warning("The template '%s' has template parameters but those are not found for the object '%s', skipped.", clutter_get_script_id(G_OBJECT(template)), clutter_get_script_id(G_OBJECT(scriptable)));
		} else {
			tangle_template_apply_properties(template, G_OBJECT(scriptable));
		}
	} else if (name[0] != '$' && name[0] != '@') {
		g_object_set_property(G_OBJECT(scriptable), name, value);
	}
	
	return TRUE;
}
