/*
 * tangle-apply-template-action.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-apply-template-action.h"
#include "tangle-misc.h"

/**
 * SECTION:tangle-apply-template-action
 * @Short_description: An action that applies the given template to the given object
 * @Title: TangleApplyTemplateAction
 *
 * This action calls tangle_template_apply_properties() when executed.
 */

G_DEFINE_TYPE(TangleApplyTemplateAction, tangle_apply_template_action, TANGLE_TYPE_ACTION);

enum {
	PROP_0,
	PROP_TEMPLATE,
	PROP_OBJECT
	/* TODO:
	PROP_MODE,
	PROP_DURATION
	*/
};

struct _TangleApplyTemplateActionPrivate {
	TangleTemplate* template;
	GObject* object;
};

TangleAction* tangle_apply_template_action_new(TangleTemplate* template, GObject* object) {	

	return TANGLE_ACTION(g_object_new(TANGLE_TYPE_APPLY_TEMPLATE_ACTION, "template", template, "object", object, NULL));
}

TangleTemplate* tangle_apply_template_action_get_template(TangleApplyTemplateAction* action) {

	return action->priv->template;
}

void tangle_apply_template_action_set_template(TangleApplyTemplateAction* action, TangleTemplate* template) {
	if (action->priv->template != template) {
		if (template) {
			g_object_ref(template);
		}
		if (action->priv->template) {
			g_object_unref(action->priv->template);
		}
		action->priv->template = template;
		g_object_notify(G_OBJECT(action), "template");
	}
}

GObject* tangle_apply_template_action_get_object(TangleApplyTemplateAction* action) {

	return action->priv->object;
}

void tangle_apply_template_action_set_object(TangleApplyTemplateAction* action, GObject* object) {
	if (action->priv->object != object) {
		if (object) {
			g_object_ref(object);
		}
		if (action->priv->object) {
			g_object_unref(action->priv->object);
		}
		action->priv->object = object;
		g_object_notify(G_OBJECT(action), "object");
	}
}

static void tangle_apply_template_action_execute(TangleAction* tangle_action, GObject* source, const gchar* trigger, TangleProperties* properties) {
	TangleApplyTemplateAction* action;
	
	action = TANGLE_APPLY_TEMPLATE_ACTION(tangle_action);
	
	if (action->priv->template && action->priv->object) {
		tangle_template_apply_properties(action->priv->template, action->priv->object);
	}
}

static void tangle_apply_template_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleApplyTemplateAction* action;
	
	action = TANGLE_APPLY_TEMPLATE_ACTION(object);

	switch (prop_id) {
		case PROP_TEMPLATE:
			tangle_apply_template_action_set_template(action, TANGLE_TEMPLATE(g_value_get_object(value)));
			break;
		case PROP_OBJECT:
			tangle_apply_template_action_set_object(action, g_value_get_object(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_apply_template_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleApplyTemplateAction* action;

	action = TANGLE_APPLY_TEMPLATE_ACTION(object);

        switch (prop_id) {
		case PROP_TEMPLATE:
			g_value_set_object(value, action->priv->template);
			break;
		case PROP_OBJECT:
			g_value_set_object(value, action->priv->object);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_apply_template_action_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_apply_template_action_parent_class)->finalize(object);
}

static void tangle_apply_template_action_dispose(GObject* object) {
	TangleApplyTemplateAction* action;
	
	action = TANGLE_APPLY_TEMPLATE_ACTION(object);
	
	TANGLE_UNREF_AND_NULLIFY_OBJECT_POINTER(action->priv->template);
	TANGLE_UNREF_AND_NULLIFY_OBJECT_POINTER(action->priv->object);

	G_OBJECT_CLASS(tangle_apply_template_action_parent_class)->dispose(object);
}

static void tangle_apply_template_action_class_init(TangleApplyTemplateActionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	TangleActionClass* action_class = TANGLE_ACTION_CLASS(klass);

	gobject_class->finalize = tangle_apply_template_action_finalize;
	gobject_class->dispose = tangle_apply_template_action_dispose;
	gobject_class->set_property = tangle_apply_template_action_set_property;
	gobject_class->get_property = tangle_apply_template_action_get_property;
	
	action_class->execute = tangle_apply_template_action_execute;

	/**
	 * TangleApplyTemplateAction:template:
	 *
	 * The template that is applied to a given object.
	 */
	g_object_class_install_property(gobject_class, PROP_TEMPLATE,
	                                g_param_spec_object("template",
	                                "Template",
	                                "The template that is applied to a given object",
	                                TANGLE_TYPE_TEMPLATE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleApplyTemplateAction:object:
	 *
	 * The object to which a template is applied.
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT,
	                                g_param_spec_object("object",
	                                "Object",
	                                "The object to which a template is applied",
	                                G_TYPE_OBJECT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleApplyTemplateActionPrivate));
}

static void tangle_apply_template_action_init(TangleApplyTemplateAction* apply_template_action) {
	apply_template_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(apply_template_action, TANGLE_TYPE_APPLY_TEMPLATE_ACTION, TangleApplyTemplateActionPrivate);
}

