/*
 * tangle-binding.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-binding.h"

/**
 * SECTION:tangle-binding
 * @Short_description: A connection between two property values
 * @Title: TangleBinding
 *
 * #TangleBinding connects two properties of one or two objects
 * so that the values are continuously the same.
 *
 * The implementation tracks both properties, and sets the other
 * when either one changes. However, in case of write-only
 * properties, the binding will be unidirectional.
 */

G_DEFINE_TYPE(TangleBinding, tangle_binding, G_TYPE_OBJECT);

#define MAX_NOTIFY_COUNT 10

enum {
	PROP_0,
	PROP_OBJECT_A,
	PROP_PROPERTY_A,
	PROP_OBJECT_B,
	PROP_PROPERTY_B,
	PROP_DIRECTION
};

struct _TangleBindingPrivate {
	GObject* objects[2];
	gchar* properties[2];
	TangleBindingDirection direction;
	guint notify_counts[2];
	gulong notify_signal_handler_ids[2];
};

static GQuark quark_property_bindings = 0;

static gulong connect_notify_signal(GObject* object, const gchar* property, GCallback callback, gpointer user_data);
static void disconnect_notify_signals(TangleBinding* binding);
static void on_notify_a(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_notify_b(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_destroy_notify(gpointer user_data, GObject* where_the_object_was);

TangleBinding* tangle_binding_new(GObject* object_a, const gchar* property_a, GObject* object_b, const gchar* property_b) {

	return TANGLE_BINDING(g_object_new(TANGLE_TYPE_BINDING, "object-a", object_a, "property-a", property_a, "object-b", object_b, "property-b", property_b, NULL));
}

GObject* tangle_binding_get_object_a(TangleBinding* binding) {

	return binding->priv->objects[0];
}

const gchar* tangle_binding_get_property_a(TangleBinding* binding) {

	return binding->priv->properties[0];
}

GObject* tangle_binding_get_object_b(TangleBinding* binding) {

	return binding->priv->objects[1];
}

const gchar* tangle_binding_get_property_b(TangleBinding* binding) {

	return binding->priv->properties[1];
}

TangleBindingDirection tangle_binding_get_direction(TangleBinding* binding) {

	return binding->priv->direction;
}

GList* tangle_object_get_bindings_readonly(GObject* object, const gchar* property) {
	GList* list = NULL;
	GHashTable* hash_table;
	
	if ((hash_table = (GHashTable*)g_object_get_qdata(object, quark_property_bindings))) {
		list = g_hash_table_lookup(hash_table, property);
	}
	
	return list;
}

static void tangle_binding_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(object);

	switch (prop_id) {
		case PROP_OBJECT_A:
			binding->priv->objects[0] = g_value_get_object(value);
			break;
		case PROP_PROPERTY_A:
			binding->priv->properties[0] = g_strdup(g_value_get_string(value));
			break;
		case PROP_OBJECT_B:
			binding->priv->objects[1] = g_value_get_object(value);
			break;
		case PROP_PROPERTY_B:
			binding->priv->properties[1] = g_strdup(g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_binding_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleBinding* binding;

	binding = TANGLE_BINDING(object);

        switch (prop_id) {
		case PROP_OBJECT_A:
			g_value_set_object(value, binding->priv->objects[0]);
			break;
		case PROP_PROPERTY_A:
			g_value_set_string(value, binding->priv->properties[0]);
			break;
		case PROP_OBJECT_B:
			g_value_set_object(value, binding->priv->objects[1]);
			break;
		case PROP_PROPERTY_B:
			g_value_set_string(value, binding->priv->properties[1]);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_binding_constructed(GObject* object) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(object);
	
	g_return_if_fail(binding->priv->objects[0]);
	g_return_if_fail(binding->priv->properties[0]);
	g_return_if_fail(binding->priv->objects[1]);
	g_return_if_fail(binding->priv->properties[1]);

	g_object_weak_ref(binding->priv->objects[0], on_destroy_notify, binding);
	g_object_weak_ref(binding->priv->objects[1], on_destroy_notify, binding);

	/* TODO: read-only properties ?! */
	
	if ((binding->priv->notify_signal_handler_ids[0] = connect_notify_signal(binding->priv->objects[0], binding->priv->properties[0], G_CALLBACK(on_notify_a), binding))) {
		binding->priv->direction |= TANGLE_BINDING_A_TO_B;
	}
	if ((binding->priv->notify_signal_handler_ids[1] = connect_notify_signal(binding->priv->objects[1], binding->priv->properties[1], G_CALLBACK(on_notify_b), binding))) {
		binding->priv->direction |= TANGLE_BINDING_B_TO_A;
	}
	
	if (binding->priv->direction == TANGLE_BINDING_NONE) {
		g_warning("Both properties '%s' and '%s' were write only.", binding->priv->properties[0], binding->priv->properties[1]);
	}
}

static void tangle_binding_finalize(GObject* object) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(object);
	
	disconnect_notify_signals(binding);
	g_free(binding->priv->properties[0]);
	g_free(binding->priv->properties[1]);

	G_OBJECT_CLASS(tangle_binding_parent_class)->finalize(object);
}

static void tangle_binding_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_binding_parent_class)->dispose(object);
}

static void tangle_binding_class_init(TangleBindingClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->constructed = tangle_binding_constructed;
	gobject_class->finalize = tangle_binding_finalize;
	gobject_class->dispose = tangle_binding_dispose;
	gobject_class->set_property = tangle_binding_set_property;
	gobject_class->get_property = tangle_binding_get_property;

	quark_property_bindings = g_quark_from_static_string("tangle-property-bindings");

	/**
	 * TangleBinding:object-a:
	 *
	 * The first object of which property is binded.
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_A,
	                                g_param_spec_object("object-a",
	                                "Object A",
	                                "The first object of which property is binded",
	                                G_TYPE_OBJECT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:property-a:
	 *
	 * The name of the first binded property.
	 */
	g_object_class_install_property(gobject_class, PROP_PROPERTY_A,
	                                g_param_spec_string("property-a",
	                                "Property A",
	                                "The name of the first binded property",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:object-b:
	 *
	 * The second object of which property is binded.
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_B,
	                                g_param_spec_object("object-b",
	                                "Object B",
	                                "The second object of which property is binded",
	                                G_TYPE_OBJECT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleBinding:property-b:
	 *
	 * The name of the second binded property.
	 */
	g_object_class_install_property(gobject_class, PROP_PROPERTY_B,
	                                g_param_spec_string("property-b",
	                                "Property B",
	                                "The name of the second binded property",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleBindingPrivate));
}

static void tangle_binding_init(TangleBinding* binding) {
	binding->priv = G_TYPE_INSTANCE_GET_PRIVATE(binding, TANGLE_TYPE_BINDING, TangleBindingPrivate);
}

static gulong connect_notify_signal(GObject* object, const gchar* property, GCallback callback, gpointer user_data) {
	gulong handler_id = 0;
	GHashTable* hash_table;
	gchar* key;
	GList* list;
	GParamSpec* param_spec;
	gchar* s;
	
	if (!(param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), property))) {
		g_warning("Object has no property called '%s'.", property);
	} else {
		if (!(hash_table = (GHashTable*)g_object_get_qdata(object, quark_property_bindings))) {
			hash_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, (GDestroyNotify)g_list_free);
			g_object_set_qdata(object, quark_property_bindings, hash_table);
		}
		
		if (g_hash_table_lookup_extended(hash_table, property, (gpointer*)&key, (gpointer*)&list)) {
			g_hash_table_steal(hash_table, key);
		} else {
			list = NULL;
			key = g_strdup(property);
		}
		list = g_list_prepend(list, user_data);
		g_hash_table_insert(hash_table, key, list);
	
		if (param_spec->flags & G_PARAM_READABLE) {
			s = g_strconcat("notify::", property, NULL);
			handler_id = g_signal_connect(object, s, callback, user_data);
			g_free(s);
		}
	}
	
	return handler_id;
}

static void disconnect_notify_signal(GObject* object, const gchar* property, gulong handler_id, gpointer user_data) {
	GHashTable* hash_table;
	gchar* key;
	GList* list;
	
	if (object) {
		if ((hash_table = (GHashTable*)g_object_get_qdata(object, quark_property_bindings)) &&
		    g_hash_table_lookup_extended(hash_table, property, (gpointer*)&key, (gpointer*)&list)) {
			g_hash_table_steal(hash_table, key);
			list = g_list_remove(list, user_data);
			g_hash_table_insert(hash_table, key, list);
		}
		
		if (handler_id) {
			g_signal_handler_disconnect(object, handler_id);
		}
	}
}


static void disconnect_notify_signals(TangleBinding* binding) {
	disconnect_notify_signal(binding->priv->objects[0], binding->priv->properties[0], binding->priv->notify_signal_handler_ids[0], binding);
	disconnect_notify_signal(binding->priv->objects[1], binding->priv->properties[1], binding->priv->notify_signal_handler_ids[1], binding);
			
	binding->priv->direction = TANGLE_BINDING_NONE;
}

static void syncronize(TangleBinding* binding, gint from_index, gint to_index) {
	GValue value;

	binding->priv->notify_counts[from_index]++;

	if (binding->priv->notify_counts[from_index] > MAX_NOTIFY_COUNT) {
		g_warning("Possible loop when synchronizing binding property '%s'.", binding->priv->properties[from_index]);
	} else {
		g_object_get_property(binding->priv->objects[from_index], binding->priv->properties[from_index], &value);
		g_object_set_property(binding->priv->objects[to_index], binding->priv->properties[to_index], &value);
		g_value_unset(&value);
	}

	binding->priv->notify_counts[from_index]--;
}

static void on_notify_a(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	syncronize(TANGLE_BINDING(user_data), 0, 1);
}

static void on_notify_b(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	syncronize(TANGLE_BINDING(user_data), 1, 0);
}

static void on_destroy_notify(gpointer user_data, GObject* where_the_object_was) {
	TangleBinding* binding;
	
	binding = TANGLE_BINDING(user_data);
	
	if (binding->priv->objects[0] == where_the_object_was) {
		binding->priv->objects[0] = NULL;
	} else {
		binding->priv->objects[1] = NULL;	
	}

	disconnect_notify_signals(binding);
}
